% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ddfORD.R
\name{ddfORD}
\alias{ddfORD}
\alias{print.ddfORD}
\alias{plot.ddfORD}
\alias{coef.ddfORD}
\alias{logLik.ddfORD}
\alias{AIC.ddfORD}
\alias{BIC.ddfORD}
\title{DDF detection for ordinal data.}
\usage{
ddfORD(Data, group, focal.name, model = "adjacent", type = "both", match = "zscore",
anchor = NULL, purify = FALSE, nrIter = 10, p.adjust.method = "none",
parametrization = "irt", alpha = 0.05)

\method{print}{ddfORD}(x, ...)

\method{coef}{ddfORD}(object, SE = FALSE, simplify = FALSE, ...)

\method{logLik}{ddfORD}(object, item = "all", ...)

\method{AIC}{ddfORD}(object, item = "all", ...)

\method{BIC}{ddfORD}(object, item = "all", ...)

\method{plot}{ddfORD}(x, item = "all", title, plot.type, group.names,
  ...)
}
\arguments{
\item{Data}{matrix or data.frame: the ordinarily scored data matrix only or the ordinarily scored
data matrix plus the vector of group. See \strong{Details}.}

\item{group}{numeric or character: either the binary vector of group membership or
the column indicator of group membership. See \strong{Details}.}

\item{focal.name}{numeric or character: indicates the level of \code{group} which corresponds to
focal group}

\item{model}{character: logistic regression model for ordinal data (either \code{"adjacent"} (default) or \code{"cumulative"}).
See \strong{Details}.}

\item{type}{character: type of DDF to be tested (either \code{"both"} (default), \code{"udif"}, or \code{"nudif"}).
See \strong{Details}.}

\item{match}{specifies matching criterion. Can be either \code{"zscore"} (default, standardized total score),
\code{"score"} (total test score), or vector of the same length as number of observations in \code{Data}. See \strong{Details}.}

\item{anchor}{Either \code{NULL} (default) or a vector of item names or item identifiers specifying which items are
currently considered as anchor (DDF free) items. Argument is ignored if \code{match} is not \code{"zscore"} or \code{"score"}.}

\item{purify}{logical: should the item purification be applied? (default is \code{FALSE}). See \strong{Details}.}

\item{nrIter}{numeric: the maximal number of iterations in the item purification (default is 10).}

\item{p.adjust.method}{character: method for multiple comparison correction.
See \strong{Details}.}

\item{parametrization}{character: parametrization of regression coefficients. Possible options are
\code{"irt"} (default) and \code{"classic"}. See \strong{Details}.}

\item{alpha}{numeric: significance level (default is 0.05).}

\item{x}{an object of 'ddfORD' class}

\item{...}{other generic parameters for \code{print} function.}

\item{object}{an object of 'ddfORD' class}

\item{SE}{logical: should be standard errors also returned? (default is \code{FALSE}).}

\item{simplify}{logical: should the result be simplified to a matrix? (default is \code{FALSE}).}

\item{item}{numeric or character: either the vector of column indicator (number or column name) or \code{'all'}
(default) for all items.}

\item{title}{string: title of plot.}

\item{plot.type}{character: which plot should be displayed for cumulative logistic
regression model. Either \code{"category"} for category probabilities or
\code{"cumulative"} for cumulative probabilities.}

\item{group.names}{character: names of reference and focal group.}
}
\value{
A list of class 'ddfORD' with the following arguments:
\describe{
  \item{\code{Sval}}{the values of likelihood ratio test statistics.}
  \item{\code{ordPAR}}{the estimates of the final model.}
  \item{\code{ordSE}}{standard errors of the estimates of the final model.}
  \item{\code{parM0}}{the estimates of null model.}
  \item{\code{parM1}}{the estimates of alternative model.}
  \item{\code{model}}{model used for DDF detection.}
  \item{\code{alpha}}{numeric: significance level.}
  \item{\code{DDFitems}}{either the column indicators of the items which were detected as DDF, or \code{"No DDF item detected"}.}
  \item{\code{type}}{character: type of DDF that was tested.}
  \item{\code{purification}}{\code{purify} value.}
  \item{\code{nrPur}}{number of iterations in item purification process. Returned only if \code{purify}
  is \code{TRUE}.}
  \item{\code{ddfPur}}{a binary matrix with one row per iteration of item purification and one column per item.
  "1" in i-th row and j-th column means that j-th item was identified as DDF in i-1-th iteration. Returned only
  if \code{purify} is \code{TRUE}.}
  \item{\code{conv.puri}}{logical indicating whether item purification process converged before the maximal number
  \code{nrIter} of iterations. Returned only if \code{purify} is \code{TRUE}.}
  \item{\code{p.adjust.method}}{character: method for multiple comparison correction which was applied.}
  \item{\code{pval}}{the p-values by likelihood ratio test.}
  \item{\code{adj.pval}}{the adjusted p-values by likelihood ratio test using \code{p.adjust.method}.}
  \item{\code{df}}{the degress of freedom of likelihood ratio test.}
  \item{\code{group}}{the vector of group membership.}
  \item{\code{Data}}{the data matrix.}
  \item{\code{match}}{matching criterion.}
  \item{\code{group.names}}{levels of grouping variable.}
  \item{\code{llM0}}{log-likelihood of null model.}
  \item{\code{llM1}}{log-likelihood of alternative model.}
  \item{\code{AICM0}}{AIC of null model.}
  \item{\code{AICM1}}{AIC of alternative model.}
  \item{\code{BICM0}}{BIC of null model.}
  \item{\code{BICM1}}{BIC of alternative model.}
  \item{\code{parametrization}}{Parameters' parametrization.}
  }
}
\description{
Performs DDF detection procedure for ordinal data based either on adjacent category logit model
or on cumulative logit model and likelihood ratio test of submodel.
}
\details{
DDF detection procedure for ordinal data based either on adjacent category logit model
or on cumulative logit model.

The \code{Data} is a matrix or data.frame which rows represents examinee ordinarily scored answers and
columns correspond to the items. The \code{group} must be either a vector of the same length as \code{nrow(Data)}
or column indicator of \code{Data}.

The \code{model} corresponds to model to be used for DDF detection. Options are \code{"adjacent"}
for adjacent category logit model or \code{"cumulative"} for cumulative logit model.

The \code{type} corresponds to type of DDF to be tested. Possible values are \code{"both"}
to detect any DDF (uniform and/or non-uniform), \code{"udif"} to detect only uniform DDF or
\code{"nudif"} to detect only non-uniform DDF.

Argument \code{match} represents the matching criterion. It can be either the standardized test score (default,
\code{"zscore"}), total test score (\code{"score"}), or any other continuous or discrete variable of the same
length as number of observations in \code{Data}.

A set of anchor items (DDF free) can be specified through the \code{anchor} argument. It need to be a vector of either
item names (as specified in column names of \code{Data}) or item identifiers (integers specifying the column number).
In case anchor items are provided, only these items are used to compute matching criterion \code{match}. If the \code{match}
argument is not either \code{"zscore"} or \code{"score"}, \code{anchor} argument is ignored.  When anchor items are
provided, purification is not applied.

The \code{p.adjust.method} is a character for \code{p.adjust} function from the \code{stats} package. Possible values are
\code{"holm"}, \code{"hochberg"}, \code{"hommel"}, \code{"bonferroni"}, \code{"BH"}, \code{"BY"}, \code{"fdr"}, and
\code{"none"}. See also \code{\link[stats]{p.adjust}} for more information.

Argument \code{parametrization} is a character which specifies parametrization of regression parameters. Default option
is \code{"irt"} which returns IRT parametrization (difficulty-discrimination). Option \code{"classic"} returns
intercept-slope parametrization with effect of group membership and interaction with matching criterion.

The output of the \code{ddfORD()} function is displayed by the \code{print.ddfORD} function.

The characteristic curve for item specified in \code{item} option can be plotted. For default
option \code{"all"} of item, characteristic curves of all converged items are plotted.
The drawn curves represent best model.

Missing values are allowed but discarded for item estimation. They must be coded as \code{NA}
for both, \code{Data} and \code{group} parameters.
}
\examples{
\dontrun{
# loading data
data(dataMedicalgraded, package = "ShinyItemAnalysis")
Data <- dataMedicalgraded[, 1:5]
group <- dataMedicalgraded[, 101]

# Testing both DDF effects with adjacent category logit model
(x <- ddfORD(Data, group, focal.name = 1, model = "adjacent"))

# graphical devices
plot(x, item = 3)
plot(x, item = "X2003")
plot(x, item = "X2003", group.names = c("Group 1", "Group 2"))

# estimated parameters
coef(x)
coef(x, SE = T) # with SE
coef(x, SE = T, simplify = T) # with SE, simplified

# AIC, BIC, log-likelihood
AIC(x); BIC(x); logLik(x)
# AIC, BIC, log-likelihood for the first item
AIC(x, item = 1); BIC(x, item = 1); logLik(x, item = 1)

# Testing both DDF effects with Benjamini-Hochberg adjustment method
ddfORD(Data, group, focal.name = 1, model = "adjacent", p.adjust.method = "BH")

# Testing both DDF effects with item purification
ddfORD(Data, group, focal.name = 1, model = "adjacent", purify = T)

# Testing uniform DDF effects
ddfORD(Data, group, focal.name = 1, model = "adjacent", type = "udif")
# Testing non-uniform DDF effects
ddfORD(Data, group, focal.name = 1, model = "adjacent", type = "nudif")

# Testing both DDF effects with total score as matching criterion
ddfORD(Data, group, focal.name = 1, model = "adjacent", match = "score")

# Testing both DDF effects with cumulative logit model
# using IRT parametrization
(x <- ddfORD(Data, group, focal.name = 1, model = "cumulative", parametrization = "irt"))

# graphical devices
plot(x, item = 3, plot.type = "cumulative")
plot(x, item = 3, plot.type = "category")

# estimated parameters in IRT parametrization
coef(x, simplify = T)
}

}
\references{
Agresti, A. (2010). Analysis of ordinal categorical data. Second edition. John Wiley & Sons.
}
\seealso{
\code{\link[stats]{p.adjust}} \code{\link[VGAM]{vglm}}
}
\author{
Adela Hladka (nee Drabinova) \cr
Institute of Computer Science, The Czech Academy of Sciences \cr
Faculty of Mathematics and Physics, Charles University \cr
hladka@cs.cas.cz \cr

Patricia Martinkova \cr
Institute of Computer Science, The Czech Academy of Sciences \cr
martinkova@cs.cas.cz \cr
}
\keyword{DDF}
