% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/core_stats.R
\name{calc_stats}
\alias{calc_stats}
\alias{calc_stats.data.frame}
\alias{calc_stats.drift_dm}
\alias{calc_stats.fits_ids_dm}
\title{Calculate Statistics}
\usage{
calc_stats(object, type, ...)

\method{calc_stats}{data.frame}(
  object,
  type,
  ...,
  conds = NULL,
  verbose = 0,
  average = FALSE,
  split_by_ID = TRUE,
  b_coding = NULL
)

\method{calc_stats}{drift_dm}(object, type, ..., conds = NULL)

\method{calc_stats}{fits_ids_dm}(object, type, ..., verbose = 1, average = FALSE)
}
\arguments{
\item{object}{an object for which statistics are calculated. This can be a
\link{data.frame} of observed data, a \link{drift_dm} object, or a
\code{fits_ids_dm} object (see \link{estimate_model_ids}).}

\item{type}{a character vector, specifying the statistics to calculate.
Supported values include \code{"cafs"}, \code{"quantiles"}, \code{"delta_funs"}, and
\code{"fit_stats"}.}

\item{...}{additional arguments passed to the respective method and the
underlying calculation functions (see Details for mandatory arguments).}

\item{conds}{optional character vector specifying conditions to include.
Conditions must match those found in the \code{object}.}

\item{verbose}{integer, indicating if information about the progress
should be displayed. 0 -> no information, 1 -> a progress bar. Default is 0.}

\item{average}{logical. If \code{TRUE}, averages the statistics across individuals
where applicable. Default is \code{FALSE}.}

\item{split_by_ID}{logical. If \code{TRUE}, statistics are calculated separately
for each individual ID in \code{object} (when \code{object} is a \link{data.frame}). Default
is \code{TRUE}.}

\item{b_coding}{a list for boundary coding (see \link{b_coding}). Only
relevant when \code{object} is a \link{data.frame}. For other \code{object} types, the
\code{b_coding} of the \code{Object} is used.}
}
\value{
If \code{type} is a single character string, then a \link{data.frame} is returned.
If \code{type} contains multiple character strings (i.e., is a character vector)
a list with the calculated statistics (with entries being \link{data.frame}s) is
returned.

Each returned \link{data.frame} has a certain class label and may store additional
attributes required for the custom \code{plot()} functions. If a list is returned,
then that list will have the class label \code{list_stats_dm} (to easily create
multiple panels using the respective \code{plot()} method).
}
\description{
\code{calc_stats} provides an interface for calculating statistics/metrics on
model predictions and/or observed data. Supported statistics include
Conditional Accuracy Functions (CAFs), Quantiles, Delta Functions, and Fit
Statistics. Results can be aggregated across individuals.
}
\details{
\code{calc_stats} is a generic function to handle the calculation of different
statistics/metrics for the supported object types. Per default, it returns
the requested statistics/metrics.
\subsection{Conditional Accuracy Function (CAFs)}{

CAFs are a way to quantify response accuracy against speed. To calculate
CAFs, RTs (whether correct or incorrect) are first binned and then the
percent correct responses per bin is calculated.

When calculating model-based CAFs, a joint CDF combining both the pdf
of correct and incorrect responses is calculated. Afterwards, this CDF
is separated into even-spaced segments and the contribution of
the pdf associated with a correct response relative to the joint CDF is
calculated.

The number of bins can be controlled by passing the argument \code{n_bins}.
The default is 5.
}

\subsection{Quantiles}{

For observed response times, the function \link[stats:quantile]{stats::quantile} is used with
default settings.

Which quantiles are calcuated can be controlled by providing the
probabilites, \code{probs}, with values in \eqn{[0, 1]}. Default is
\code{seq(0.1, 0.9, 0.1)}.
}

\subsection{Delta Functions}{

Delta functions calculate the difference between quantiles
of two conditions against their mean:
\itemize{
\item \eqn{Delta_i = Q_{i,j} - Q_{i,k}}
\item \eqn{Avg_i = 0.5 \cdot Q_{i,j} + 0.5 \cdot Q_{i,k}}
}

With i indicating a quantile, and j and k two conditions.

To calculate delta functions, users have to specify:
\itemize{
\item \code{minuends}: character vector, specifying condition(s) j. Must be in
\code{conds(drift_dm_obj)}.
\item \code{subtrahends}: character vector, specifying condition(s) k. Must be in
\code{conds(drift_dm_obj)}
\item \code{dvs}: character, indicating which quantile columns to use.
Default is "Quant_<u_label>". If multiple dvs are provided,
then minuends and subtrahends must have the same length,
and matching occurs pairwise. In this case, if only one
minuend/subtrahend is specified, minuend and subtrahend are recycled to
the necessary length.
}
}

\subsection{Fit Statistics}{

Calculates the Akaike and Bayesian Information Criteria (AIC and BIC). Users
can provide a \code{k} argument to penalize the AIC statistic (see \link[stats:AIC]{stats::AIC}
and \link{AIC.fits_ids_dm})
}
}
\examples{
# Example 1: Calculate CAFs and Quantiles from a model ---------------------
# get a model for demonstration purpose
a_model <- ssp_dm(dx = .0025, dt = .0025, t_max = 2)
# and then calculate cafs and quantiles
some_stats <- calc_stats(a_model, type = c("cafs", "quantiles"))
head(some_stats$cafs)
head(some_stats$quantiles)

# Example 2: Calculate a Delta Function from a data.frame ------------------
# get a data set for demonstration purpose
some_data <- ulrich_simon_data
conds(some_data) # relevant for minuends and subtrahends
some_stats <- calc_stats(
  a_model,
  type = "delta_funs",
  minuends = "incomp",
  subtrahends = "comp"
)
head(some_stats)


# Example 3: Calculate Quantiles from a fits_ids_dm object -----------------
# get an auxiliary fits_ids_dm object
all_fits <- get_example_fits_ids()
some_stats <- calc_stats(all_fits, type = "quantiles")
head(some_stats) # note the ID column

# one can also request that the statistics are averaged across individuals
head(
  calc_stats(all_fits, type = "quantiles", average = TRUE)
)

}
