% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/score.R
\name{score}
\alias{score}
\title{Compute (weighted) comorbidity scores}
\usage{
score(x, weights = NULL, assign0)
}
\arguments{
\item{x}{An object of class \code{comorbidty} returned by a call to the \code{\link[=comorbidity]{comorbidity()}} function.}

\item{weights}{A string denoting the weighting system to be used, which will depend on the mapping algorithm.

Possible values for the Charlson index are:
\itemize{
\item \code{charlson}, for the original weights by Charlson et al. (1987);
\item \code{quan}, for the revised weights by Quan et al. (2011).
}

Possible values for the Elixhauser score are:
\itemize{
\item \code{vw}, for the weights by van Walraven et al. (2009);
\item \code{swiss}, for the Swiss Elixhauser weights by Sharma et al. (2021).
}

Defaults to \code{NULL}, in which case an unweighted score will be used.}

\item{assign0}{A logical value denoting whether to apply a hierarchy of comorbidities: should a comorbidity be present in a patient with different degrees of severity, then the milder form will be assigned a value of 0 when calculating the score.
By doing this, a type of comorbidity is not counted more than once in each patient.
If \code{assign0 = TRUE}, the comorbidities that are affected by this argument are:
\itemize{
\item "Mild liver disease" (\code{mld}) and "Moderate/severe liver disease" (\code{msld}) for the Charlson score;
\item "Diabetes" (\code{diab}) and "Diabetes with complications" (\code{diabwc}) for the Charlson score;
\item "Cancer" (\code{canc}) and "Metastatic solid tumour" (\code{metacanc}) for the Charlson score;
\item "Hypertension, uncomplicated" (\code{hypunc}) and "Hypertension, complicated" (\code{hypc}) for the Elixhauser score;
\item "Diabetes, uncomplicated" (\code{diabunc}) and "Diabetes, complicated" (\code{diabc}) for the Elixhauser score;
\item "Solid tumour" (\code{solidtum}) and "Metastatic cancer" (\code{metacanc}) for the Elixhauser score.
}}
}
\value{
A numeric vector with the (possibly weighted) comorbidity score for each subject from the input dataset.
}
\description{
Compute (weighted) comorbidity scores
}
\examples{
set.seed(1)
x <- data.frame(
  id = sample(1:15, size = 200, replace = TRUE),
  code = sample_diag(200),
  stringsAsFactors = FALSE
)

# Charlson score based on ICD-10 diagnostic codes:
x1 <- comorbidity(x = x, id = "id", code = "code", map = "charlson_icd10_quan", assign0 = FALSE)
score(x = x1, weights = "charlson", assign0 = FALSE)

# Elixhauser score based on ICD-10 diagnostic codes:
x2 <- comorbidity(x = x, id = "id", code = "code", map = "elixhauser_icd10_quan", assign0 = FALSE)
score(x = x2, weights = "vw", assign0 = FALSE)

# Checking the `assign0` argument.
# Please make sure to check the example in the documentation of the
# `comorbidity()` function first, with ?comorbidity().
# We use the same dataset for a single subject with two codes, for
# complicated and uncomplicated diabetes:
x3 <- data.frame(
  id = 1,
  code = c("E100", "E102"),
  stringsAsFactors = FALSE
)
# Then, we calculate the Quan-ICD10 Charlson score:
ccF <- comorbidity(x = x3, id = "id", code = "code", map = "charlson_icd10_quan", assign0 = FALSE)
ccF[, c("diab", "diabwc")]
# If we calculate the unweighted score with `assign0 = FALSE`, both diabetes
# conditions are counted:
score(x = ccF, assign0 = FALSE)
# Conversely, with `assign0 = TRUE`, only the most severe is considered:
score(x = ccF, assign0 = TRUE)
}
\references{
Charlson ME, Pompei P, Ales KL, et al. \emph{A new method of classifying prognostic comorbidity in longitudinal studies: development and validation}. Journal of Chronic Diseases 1987; 40:373-383.

Quan H, Li B, Couris CM, et al. \emph{Updating and validating the Charlson Comorbidity Index and Score for risk adjustment in hospital discharge abstracts using data from 6 countries}. American Journal of Epidemiology 2011; 173(6):676-682.

van Walraven C, Austin PC, Jennings A, Quan H and Forster AJ. \emph{A modification of the Elixhauser comorbidity measures into a point system for hospital death using administrative data}. Medical Care 2009; 47(6):626-633.

Sharma N, Schwendimann R, Endrich O, et al. \emph{Comparing Charlson and Elixhauser comorbidity indices with different weightings to predict in-hospital mortality: an analysis of national inpatient data}. BMC Health Services Research 2021; 21(13).
}
