#' Visualize time intervals across subjects/parameters.
#' @param paramVar Character vector with variable of \code{data}
#' to represent in the y-axis.
#' @param paramLab (optional) String with label for \code{paramVar}.
#' @param paramVarSep (optional) String with separator used to combined \code{paramVar}
#' if multiple.
#' @param paramGroupVar (optional) Character vector with variable(s) to group/order
#' the \code{paramVar} elements in the y-axis.
#' @param timeStartVar String with variable with the start of the time interval.
#' @param timeStartLab (optional) String with label for \code{timeStartVar}.
#' @param timeEndVar String with variable with the end of the time interval.
#' @param timeEndLab (optional) String with label for \code{timeEndVar}.
#' @param timeStartShapeVar (optional) String with variable used for the shape
#' of the start of the time interval.
#' @param timeStartShapeLab (optional) String with label for \code{timeStartShapeVar}.
#' @param timeEndShapeVar (optional) String with variable used for the shape
#' of the end of the time interval.
#' @param timeEndShapeLab (optional) String with label for \code{timeEndShapeVar}.
#' @param shapePalette (optional) Character vector with shape palette for
#' \code{timeStartShapeVar} and \code{timeEndShapeVar}.
#' @param xLab,yLab (optional) String with labels for the x/y-axis.
#' @param colorVar (optional) String with color variable.
#' @param colorLab (optional) String with label for \code{colorVar}.
#' @param alpha (optional) Numeric with transparency, 1 by default.
#' @inheritParams clinDataReview-common-args
#' @inheritParams tableClinData
#' @inherit scatterplotClinData return
#' @author Laure Cougnaud
#' @import plotly
#' @importFrom clinUtils formatVarForPlotLabel getColorPalette getShapePalette
#' @export
timeProfileIntervalPlot <- function(data,
	paramVar, paramLab = getLabelVar(paramVar, labelVars = labelVars),
	paramVarSep = " - ",
	paramGroupVar = NULL,
	timeStartVar, timeStartLab = getLabelVar(timeStartVar, labelVars = labelVars),
	timeEndVar, timeEndLab = getLabelVar(timeEndVar, labelVars = labelVars),
	# shape
	timeStartShapeVar = NULL, timeStartShapeLab = getLabelVar(timeStartShapeVar, labelVars = labelVars),
	timeEndShapeVar = NULL, timeEndShapeLab = getLabelVar(timeEndShapeVar, labelVars = labelVars),
	shapePalette = NULL,
	# color
	colorVar = NULL, colorLab = getLabelVar(timeStartShapeVar, labelVars = labelVars),
	colorPalette = NULL,
	# transparency
	alpha = 1,
	# labels
	yLab = NULL, #paste(paramLab, collapse = "\n"),
	xLab = paste(c(timeStartLab, timeEndLab), collapse = " and "),
	title = NULL,
	labelVars = NULL,
	# interactivity:
	width = 800, height = NULL,
	hoverVars, hoverLab,
	# path to subject-specific report
	idVar = "USUBJID", idLab = getLabelVar(idVar, labelVars = labelVars),
	pathVar = NULL, pathLab = getLabelVar(pathVar, labelVars = labelVars),
	# table
	table = FALSE, 
	tableVars, tableLab,
	tableButton = TRUE, tablePars = list(),
	id = paste0("plotClinData", sample.int(n = 1000, size = 1)),
	verbose = FALSE){

	# store input parameter values for further use
	plotArgs <- c(as.list(environment()))

	## Format data
	
	# fill missing time:
#	timeStartVar <- paste0(timeStartVar, "Plot")
#	timeEndVarPlot <- paste0(timeEndVar, "Plot")
#	data <- formatTimeInterval(
#		data = data, 
#		timeStartVar = timeStartVar, timeEndVar = timeEndVar, 
#		timeStartVarNew = timeStartVar, timeEndVarNew = timeEndVarPlot,
#		subjectVar = subjectVar,
#		timeLim = timeLim, timeLimData = timeLimData, 
#		timeLimStartVar = timeLimStartVar, timeLimEndVar = timeLimEndVar
#	)
#	if(!all(data[, c(timeStartVar, timeEndVar)] == data[, c(timeStartVar, timeEndVarPlot)], na.rm = TRUE))
#		stop("Issue of extraction of missing time.")
	
	# concatenate variable(s) if multiple are specified
	if(length(paramVar) > 1){
		data$yVar <- apply(data[, paramVar], 1, paste, collapse = paramVarSep)	
	}else{
		data$yVar <- data[, paramVar]
	}
	
	data$yVar <- formatVarForPlotLabel(
		data = data, 
		paramVar = "yVar", paramGroupVar = paramGroupVar, 
		width = 20, revert = TRUE
	)
	
	yLevels <- levels(data$yVar)
	
	# plotly ignores missing values by default,
	# so convert y-variable to numeric to ensure that all values are represented
	data$yVar <- as.numeric(data$yVar)
	
	# variables used to uniquely identify a record
	# between the table and the plot
#	keyVars <- c(paramVar, timeStartVar, timeEndVarPlot)
	# issue when key variable is too long in highlight_key
	# so use row IDs
	data$key <- as.character(seq_len(nrow(data)))
	keyVar <- "key"
	
	# To have correct symbols matching in plotly
	# - shape variable must be factor
	# - if multiple shape vars, each var should contain all levels
	if(!is.null(timeStartShapeVar) | !is.null(timeEndShapeVar)){
		getLevels <- function(x)
			if(is.factor(x))	levels(x)	else	unique(x)
		shapeLevels <- c(
			if(!is.null(timeStartShapeVar))	getLevels(data[, timeStartShapeVar]), 
			if(!is.null(timeEndShapeVar))	getLevels(data[, timeEndShapeVar])
		)
		shapeLevels <- unique(shapeLevels)
		if(!is.null(timeStartShapeVar))
			data[, timeStartShapeVar] <- factor(data[, timeStartShapeVar], levels = shapeLevels)
		if(!is.null(timeEndShapeVar))
			data[, timeEndShapeVar] <- factor(data[, timeEndShapeVar], levels = shapeLevels)
	}
	
	# format data to: 'SharedData' object
	if(missing(hoverVars)){
		
		hoverVars <- c(paramVar, timeStartVar, timeStartShapeVar, timeEndVar, timeEndShapeVar)
		hoverLab <- c(
			getLabelVar(var = paramVar, label = paramLab, labelVars = labelVars),
			getLabelVar(var = timeStartVar, label = timeStartLab, labelVars = labelVars),
			getLabelVar(var = timeEndVar, label = timeEndLab, labelVars = labelVars),
			getLabelVar(var = timeStartShapeVar, label = timeStartShapeLab, labelVars = labelVars),
			getLabelVar(var = timeEndShapeVar, label = timeEndShapeLab, labelVars = labelVars)
		)
		
	}else	if(missing(hoverLab)){
		hoverLab <- getLabelVar(hoverVars, labelVars = labelVars)
	}
	hoverVars <- unique(hoverVars);hoverLab <- hoverLab[hoverVars]
	
	## Convert to SharedData
	convertToSharedDataIntPlot <- function(data)
		formatDataForPlotClinData(
			data = data, 
			id = id,
			keyVar = keyVar,
			labelVars = labelVars,
			hoverVars = hoverVars, hoverLab = hoverLab,
			hoverByVar = c(paramVar, timeStartVar, timeEndVar)
		)
	dataPlotSharedData <- convertToSharedDataIntPlot(data)
	
	if(is.null(colorPalette)){
		colorPaletteOpt <- getOption("clinDataReview.colors")
		if(!is.null(colorVar)){
			colorPalette <- getColorPalette(
				x = data[, colorVar],
				palette = colorPaletteOpt
			)
		}else	colorPalette <- getColorPalette(n = 1, palette = colorPaletteOpt)
	}

	linesYVar <- regmatches(
		x = yLevels, 
		m = gregexpr(pattern = "\n", text = yLevels, fixed = TRUE)
	)
	nLinesY <- sum(sapply(linesYVar, length) + 1)
	plotHeight <- sum(nLinesY) * 20
	
	# top margin: top bar + title
	topMargin <- 30 + ifelse(!is.null(title), 20, 0) 
	# bottom margin: # x-axis + x-lab + legend
	bottomMargin <- 20 + ifelse(!is.null(xLab), 20, 0) + ifelse(!is.null(colorVar), 20, 0)
	layoutMargin <- list(t = topMargin, b = bottomMargin)
	
	# get plot dim
	if(is.null(height)){
		height <- topMargin + plotHeight + bottomMargin
	}
	
	pl <- plot_ly(width = width, height = height)

	## markers for symbols
	# always display symbols to:
	# 1) have a hover (not possible in rectangle)
	# 2) have a symbol when start and end is the same
	if(!is.null(timeStartShapeVar) | !is.null(timeEndShapeVar)){
		if(is.null(shapePalette)){
			shapePalette <- getShapePalette(
				x = shapeLevels, 
				palette = getOption("clinDataReview.shapes")
			)
		}else	shapePalette <- shapePalette[shapeLevels]
	}else	shapePalette <- NULL

	addMarkers <- function(p, showlegend = FALSE, ...){
		add_markers(
			data = dataPlotSharedData, 
			p = p,
			y = varToFm("yVar"),
			color = if(!is.null(colorVar))	varToFm(colorVar)	else	I(colorPalette), 
			colors = if(!is.null(colorVar))	colorPalette,
			hovertemplate = varToFm("hover"),
			showlegend = showlegend,
			opacity = alpha,
			...
		)
	}
	
	pl <- addMarkers(
		p = pl,
		x = varToFm(timeStartVar), 
		symbol = if(!is.null(timeStartShapeVar))	varToFm(timeStartShapeVar),
		symbols = if(!is.null(shapePalette))	I(shapePalette)
	)

	pl <- addMarkers(
		p = pl,
		x = varToFm(timeEndVar), 
		symbol = if(!is.null(timeEndShapeVar))	varToFm(timeEndShapeVar),
		symbols = if(!is.null(shapePalette))	I(shapePalette)
	)
	
	# create legend only with color
	# because plotly doesn't support multiple legend
	if(!is.null(colorVar))
		pl <- addMarkers(
			p = pl, x = varToFm(timeStartVar), 
			showlegend = TRUE, visible = "legendonly"
		)
	
	## segments for time-interval
	# plotly returns an error when no non-missing values in start/end time vars
	# and 'crossed' segments if start value == end value
	idxSegments <- which(
		rowSums(is.na(data[, c(timeStartVar, timeEndVar)])) != length(c(timeStartVar, timeEndVar)) &
		data[, timeStartVar] != data[, timeEndVar]
	)
	if(length(idxSegments) > 0){
		dataPlotSharedDataSeg <- convertToSharedDataIntPlot(data = data[idxSegments, , drop = FALSE])
		pl <- pl %>% add_segments(
			data = dataPlotSharedDataSeg,
			x = varToFm(timeStartVar), 
			xend = varToFm(timeEndVar),
			y = varToFm("yVar"),
			yend = varToFm("yVar"),
			color = if(!is.null(colorVar))	varToFm(colorVar)	else	I(colorPalette), 
			colors = if(!is.null(colorVar))	colorPalette,
			showlegend = FALSE,
			opacity = alpha
		)
	}

#	xMax <- max(data[, c(timeStartVar, timeEndVar)], na.rm = TRUE)
	pl <- pl %>% layout(
		title = title,
		# standoff: distance between axis text and title
		#  adjusted to not have overlapping legend when only one y element
		xaxis = list(title = list(text = xLab, standoff = 0)),
		yaxis = list(
			showgrid = TRUE,
			title = list(text = yLab),
			type = "array", 
			tickvals = seq_along(yLevels), 
			ticktext = yLevels,
			tickangle = 0,
			range = c(0.5, length(yLevels)+0.5)
		),
		# x/y are in normalized coordinates
		showlegend = TRUE, # print legend even if only one y-element
		legend = list(
			orientation = "h", 
			xanchor = "center", x = 0.5, 
			yanchor = "top", y = -min(bottomMargin/plotHeight, 2)
		),
		hovermode = "closest",
		# margin in pixels
		margin = layoutMargin
	)
	
	# specific formatting for clinical data
	pl <- formatPlotlyClinData(
		data = data, pl = pl,
		# extract patient profile based on the 'key' variable
		# (not yVar because different records could be grouped in the same yVar)
		# (should be included in the plot)
		idVar = keyVar, pathVar = pathVar,
		# extract ID from 'y' column in the plot output object directly
		idVarPlot = "key", idFromDataPlot = TRUE, 
		# no label for patient prof filename because y is a numeric (not informative)
#		labelVarPlot = "y",
		id = id, 
		verbose = verbose,
		pathDownload = FALSE # open in new tab
	)
	
	# create associated table
	if(table){
		
		tableVars <- getPlotTableVars(
			plotFunction = "timeProfileIntervalPlot", 
			plotArgs = plotArgs
		)
		tableLab <- attr(tableVars, "tableLab")
		
		table <- tableClinData(
			data = data, 
			keyVar = keyVar, idVar = idVar, 
			pathVar = pathVar, pathLab = pathLab,
			tableVars = tableVars,
			tableLab = tableLab,
			tableButton = tableButton, tablePars = tablePars,
			id = id, 
			labelVars = labelVars,
			verbose = verbose
		)
		res <- list(plot = pl, table = table)
		
		class(res) <- c("clinDataReview", class(res))
		
	}else res <- pl
	
	return(res)
	
	
	
}
