% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/detect_cpts.R
\name{detect.th}
\alias{detect.th}
\title{Multiple change-point detection in the cross-covariance structure
of multivariate high-dimensional time series using a thresholding
based procedure and, wherever possible, extraction of the component
time series where the changes occurred}
\usage{
detect.th(
  X,
  approach = c("euclidean", "infinity"),
  th_max = 2.25,
  th_sum = 0.65,
  pointsgen = 10,
  scales = -1,
  preaverage_gen = FALSE,
  scal_gen = 3,
  min_dist = 1
)
}
\arguments{
\item{X}{A numerical matrix representing the multivariate time series,
with the columns representing its components.}

\item{approach}{A character string, which defines the metric to be used
in order to detect the change-points. If approach = ``euclidean'', which
is also the default value, then the \eqn{L_2} metric will be followed
for the detection. If approach = ``infinity'', then the \eqn{L_{\infty}}
metric will be used for the detection.}

\item{th_max}{A positive real number with default value equal to 2.25. It is
used to define the threshold if the \eqn{L_{\infty}} metric is chosen in
\code{approach} .}

\item{th_sum}{A positive real number with default value equal to 0.65. It is
used to define the threshold if the \eqn{L_2} metric is chosen in
\code{approach}.}

\item{pointsgen}{A positive integer with default value equal to 10. It
defines the distance between two consecutive end- or start-points of
the right- or left-expanding intervals, respectively; see Details for
more information.}

\item{scales}{Negative integers for wavelet scales, with a small negative
integer representing  a fine scale. The default value is equal to -1.}

\item{preaverage_gen}{A logical variable with default value equal to
\code{FALSE}. If \code{FALSE}, then pre-averaging the data is not
required. If \code{TRUE}, then we need to pre-average the data before
proceeding with the detection of the change-points.}

\item{scal_gen}{A positive integer number with default value equal to 3. It
is used to define the way we pre-average the given data sequence only if
\code{preaverage_gen = TRUE}. See the Details in \code{\link{preaverage}}
for more information on how we pre-average.}

\item{min_dist}{A positive integer number with default value equal to 1. It
is used in order to provide the minimum distance acceptable between
detected change-points if such restrictions apply.}
}
\value{
A list with the following components:
\tabular{ll}{
\cr \code{changepoints} \tab The locations of the detected change-points.
\cr \code{no.of.cpts} \tab The number of the detected change-points.
\cr \code{time_series} \tab A list with two components that indicates which combinations
\cr \tab of time series are responsible for each change-point detected. See the outcome
\cr \tab values \code{time_series_indicator} and \code{most_important} of the function
\cr \tab \code{\link{match.cpt.ts}} for more information.
   }
  If the minimum distance between the detected change-points is less than
  the value given in \code{min_dist}, then only the number and the locations of
  the ``pruned'' change-points are returned.
}
\description{
This function detects multiple change-points in the cross-covariance
structure of a multivariate time series using a thresholding based procedure.
It also, wherever possible, returns the relevant, transformed
time series where each change-point was detected. See Details for a brief
explanation.
}
\details{
The time series \eqn{X_t} is of dimensionality \eqn{p} and we are
  looking for changes in the cross-covariance structure between the
  different time series components
  \eqn{X_{t}^{(1)}, X_{t}^{(2)}, ..., X_{t}^{(p)}}. We first use a
  wavelet-based approach for the various given scales in \code{scales}
  in order to transform the given time series \eqn{X_t} to a multiplicative
  model \eqn{Y_{t}^{(k)} = \sigma^{(k)}_t (Z_t^{(k)})^2; t=1,2,\ldots,T; k = 1,2,\ldots,d,}
  where \eqn{Z_t^{(k)}} is a sequence of standard normal random variables,
  \eqn{E(Y_t^{(k)}) = \sigma_t^{(k)}}, and \eqn{d} is the new
  dimensionality, which depends on the value given in \code{scales}.
  The function has been extensively tested on fMRI data, hence, its parameters
  have been tuned for this data type. The function might not work well in other
  structures, such as time series that are negatively serially correlated.
}
\examples{
  set.seed(111)
  A <- matrix(rnorm(20*400), nrow = 400) ## No change-point
  M1 <- detect.th(A, approach = 'euclidean', scales = -1)
  M2 <- detect.th(A, approach = 'infinity', scales = -1)
  M1
  M2

  set.seed(111)
  num.nodes <- 40 # number of nodes
  etaA.1    <- 0.95
  etaA.2    <- 0.05
  pcor1     <- GeneNet::ggm.simulate.pcor(num.nodes, etaA = etaA.1)
  pcor2     <- GeneNet::ggm.simulate.pcor(num.nodes, etaA = etaA.2)

  n <- 100
  data1 <- GeneNet::ggm.simulate.data(n, pcor1)
  data2 <- GeneNet::ggm.simulate.data(n, pcor2)

  X1 <- rbind(data1, data2) ## change-point at 100
  N1 <- detect.th(X1, approach = 'euclidean', scales = -1)
  N2 <- detect.th(X1, approach = 'infinity', scales = -1)
  N1$changepoints
  N1$time_series
  N2$changepoints
  N2$time_series
}
\references{
``Cross-covariance isolate detect: a new change-point method
  for estimating dynamic functional connectivity'', Anastasiou et al (2020),
  preprint <doi:10.1101/2020.12.20.423696>.
}
\seealso{
\code{\link{detect.ic}}.
}
\author{
Andreas Anastasiou, \email{anastasiou.andreas@ucy.ac.cy}
}
