% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ODRI.R
\name{quantreg_or1}
\alias{quantreg_or1}
\title{Bayesian Quantile Regression for Ordinal Model
with more than 3 outcomes}
\usage{
quantreg_or1(y, x, b0 = 0, B0, d0 = 0, D0, mcmc = 15000, p, tune = 0.1)
}
\arguments{
\item{y}{observed ordinal outcomes, column vector of dimension \eqn{(n x 1)}.}

\item{x}{covariate matrix of dimension \eqn{(n x k)} including a column of ones with or without column names.}

\item{b0}{prior mean for normal distribution to sample \eqn{\beta}. Default is 0.}

\item{B0}{prior variance for normal distribution to sample \eqn{\beta}.}

\item{d0}{prior mean for covariance matrix of normal distribution to sample \eqn{\delta}. Default is 0.}

\item{D0}{prior variance for normal distribution to sample \eqn{\delta}.}

\item{mcmc}{number of MCMC iterations, post burn-in.}

\item{p}{quantile level or skewness parameter, p in (0,1).}

\item{tune}{tuning parameter to adjust MH acceptance rate.}
}
\value{
Returns a list with components:
\itemize{
\item{\code{postMeanbeta}: }{a vector with mean of sampled
 \eqn{\beta} for each covariate.}
 \item{\code{postMeandelta}: }{a vector with mean of sampled
 \eqn{\delta} for each cut point.}
 \item{\code{postStdbeta}: }{a vector with standard deviation
 of sampled \eqn{\beta} for each covariate.}
 \item{\code{postStddelta}: }{a vector with standard deviation
 of sampled \eqn{\delta} for each cut point.}
 \item{\code{gamma}: }{a vector of cut points including Inf and
 -Inf.}
 \item{\code{catt}}
 \item{\code{acceptancerate}: }{a scalar to judge the acceptance
 rate of samples.}
 \item{\code{allQuantDIC}: }{results of the DIC criteria.}
 \item{\code{logMargLikelihood}: }{a scalar value for log marginal likelihood.}
 \item{\code{beta}: }{a matrix with all sampled values for \eqn{\beta}.}
 \item{\code{delta}: }{a matrix with all sampled values for \eqn{\delta}.}
}
}
\description{
This function estimates Bayesian quantile regression for ordinal model with
more than 3 outcomes and reports the posterior mean, posterior standard deviation, and 95
percent credible intervals of \eqn{(\beta, \delta)}.
}
\details{
Function implements the Bayesian quantile regression for
ordinal model with more than 3 outcomes using a combination of Gibbs sampling
and Metropolis-Hastings algorithm.

Function initializes prior and then iteratively
samples \eqn{\beta}, \eqn{\delta} and latent variable z.
Burn-in is taken as \eqn{0.25*mcmc} and \eqn{nsim = burn}-\eqn{in + mcmc}.
}
\examples{
 set.seed(101)
 data("data25j4")
 x <- data25j4$x
 y <- data25j4$y
 k <- dim(x)[2]
 J <- dim(as.array(unique(y)))[1]
 D0 <- 0.25*diag(J - 2)
 output <- quantreg_or1(y = y,x = x, B0 = 10*diag(k), D0 = D0,
 mcmc = 50, p = 0.25, tune = 1)


 # Number of burn-in draws: 12.5
 # Number of retained draws: 50
 # Summary of estimated beta:


 #             Post Mean  Post Std   Upper Credible Lower Credible
 # beta_0       -3.0358   0.4365        -2.1475        -3.7713
 # beta_1        3.9663   0.7769         5.2397         2.7132
 # beta_2        3.7117   0.6381         4.8780         2.6946
 # delta_1       0.1859   0.3603         0.7324        -0.4374
 # delta_2       0.4329   0.1969         0.7629         0.2176

 # Model Marginal Likelihood: -572.54
 # Model acceptance rate: 46
 # Model DIC: 1280.39

}
\references{
Rahman, M. A. (2016). “Bayesian
Quantile Regression for Ordinal Models.”
Bayesian Analysis, 11(1): 1-24.

Yu, K. and Moyeed, R. A. (2001). “Bayesian Quantile Regression.” Statistics and
Probability Letters, 54(4): 437–447.

Casella, G., George E. I. (1992). “Explaining the Gibbs Sampler."
The American Statistician, 46(3): 167-174.

Geman, S., and Geman, D. (1984). “Stochastic Relaxation,
Gibbs Distributions, and the Bayesian Restoration of Images."
IEEE Transactions an Pattern Analysis and Machine Intelligence,
6(6): 721-741.

 Chib, S., Greenberg E. (1995). "Understanding the Metropolis-Hastings
 Algorithm." The American Statistician, 49(4): 327-335.

 Hastings, W.K. (1970). "Monte Carlo Sampling Methods Using
 Markov Chains and Their Applications." Biometrika, 57: 1317-1340.
}
\seealso{
tcltk, \link[stats]{rnorm}, \link[stats]{qnorm},
Gibbs sampler, Metropolis-Hastings algorithm
}
