#' @title 
#' Step 3: Optimizing parameters to fit real data
#' 
#' @description
#' This function is designed to fit the optimal parameters of black-box
#'  functions (models) to real-world data. Provided that the black-box
#'  function adheres to the specified interface
#'  (demo:  
#'     \code{\link[binaryRL]{TD}}, 
#'     \code{\link[binaryRL]{RSTD}}, 
#'     \code{\link[binaryRL]{Utility}}
#'  )
#'  , this function can employ the various optimization algorithms detailed 
#'  below to find the best- fitting parameters for your model.
#'  
#'  The function provides several optimization algorithms:
#'   \itemize{
#'     \item 1. L-BFGS-B (from \code{stats::optim})
#'     \item 2. Simulated Annealing (\code{GenSA::GenSA})
#'     \item 3. Genetic Algorithm (\code{GA::ga})
#'     \item 4. Differential Evolution (\code{DEoptim::DEoptim})
#'     \item 5. Particle Swarm Optimization (\code{pso::psoptim})
#'     \item 6. Bayesian Optimization (\code{mlrMBO::mbo})
#'     \item 7. Covariance Matrix Adapting Evolutionary Strategy (\code{cmaes::cma_es})
#'     \item 8. Nonlinear Optimization (\code{nloptr::nloptr})
#'   }
#' 
#'  For more information, please refer to the homepage of this package:
#'  \url{https://yuki-961004.github.io/binaryRL/}
#'  
#' @note
#' The primary difference between `fit_p` and `rcv_d` lies in their data
#'  source: `fit_p` fits parameters to real data, while `rcv_d` fits
#'  parameters to synthetic data generated by models. Many studies
#'  inappropriately skip the `rcv_d` step.
#'
#' Imagine `fit_p` as the actual boxing match, and `rcv_d` as the weigh-in.
#'  Boxers of different weight classes shouldn't compete directly.
#'  
#' Similarly, if a competing model lacks the ability of parameter or model
#'  recovering, even if your proposed model outperforms it in `fit_p`,
#'  it doesn't necessarily make your proposed model a good one.
#'  
#' @param data [data.frame] 
#' This data should include the following mandatory columns: 
#'  \itemize{
#'    \item "sub"
#'    \item "time_line" (e.g., "Block", "Trial")
#'    \item "L_choice"
#'    \item "R_choice"
#'    \item "L_reward"
#'    \item "R_reward"
#'    \item "sub_choose"
#'  }
#'  
#' @param id [vector]
#' Specifies the subject ID(s) for whom optimal parameters are to be fitted.
#'  If you intend to fit all subjects within your dataset, it is strongly
#'  recommended to use \code{id = unique(data$Subject)}. This approach accounts
#'  for cases where subject IDs in the dataset may not be simple numeric
#'  sequences (e.g., "1", "2", "3", "4") or may contain string
#'  entries (e.g., "1", "2", "3", "004"). Using \code{id = 1:4} could lead to 
#'  errors if IDs are not sequentially numbered integers.
#'  
#'  \code{default: id = NULL}
#'  
#' @param n_trials [integer]
#' Represents the total number of trials a single subject experienced
#'  in the experiment. If this parameter is kept at its default value
#'  of `NULL`, the program will automatically detect how many trials
#'  a subject experienced from the provided data. This information
#'  is primarily used for calculating model fit statistics such as
#'  AIC (Akaike Information Criterion) and BIC (Bayesian Information
#'  Criterion).
#'  
#'  \code{default: n_trials = NULL}
#'  
#' @param fit_model [list] 
#' A collection of functions applied to fit models to the data.
#' 
#' @param funcs [character]
#' A character vector containing the names of all user-defined functions
#'  required for the computation. When parallel computation is enabled
#'  (i.e., `nc > 1`), user-defined models and their custom functions might
#'  not be automatically accessible within the parallel environment.
#'
#' Therefore, if you have created your own reinforcement learning model
#'  that modifies the package's default four default functions 
#'  (default functions: 
#'     \code{util_func = \link[binaryRL]{func_gamma}}, 
#'     \code{rate_func = \link[binaryRL]{func_eta}}, 
#'     \code{expl_func = \link[binaryRL]{func_epsilon}}
#'     \code{bias_func = \link[binaryRL]{func_pi}}
#'     \code{prob_func = \link[binaryRL]{func_tau}}
#'  ), 
#'  you must explicitly provide the names of your custom functions as a 
#'  vector here.
#' 
#' @param model_name [list] 
#' The name of fit modals
#' 
#' @param lower [list] 
#' The lower bounds for model fit models
#' 
#' @param upper [list] 
#' The upper bounds for model fit models
#' 
#' @param initial_params [vector]
#' Initial values for the free parameters that the optimization algorithm will
#'  search from. These are primarily relevant when using algorithms that require
#'  an explicit starting point, such as \code{L-BFGS-B}. If not specified,
#'  the function will automatically generate initial values close to zero.
#'  
#'  \code{default: initial_params = NA}.
#'
#' @param initial_size [integer]
#' This parameter corresponds to the \strong{population size} in genetic 
#'  algorithms (\code{GA}). It specifies the number of initial candidate
#'  solutions that the algorithm starts with for its evolutionary search.
#'  This parameter is only required for optimization algorithms that operate on
#'  a population, such as `GA` or `DEoptim`. 
#'  
#'  \code{default: initial_size = 50}.
#'  
#' @param iteration [integer] 
#' The number of iterations the optimization algorithm will perform
#'  when searching for the best-fitting parameters during the fitting
#'  phase. A higher number of iterations may increase the likelihood of 
#'  finding a global optimum but also increases computation time.
#' 
#' @param seed [integer] 
#' Random seed. This ensures that the results are 
#'  reproducible and remain the same each time the function is run. 
#'  
#'  \code{default: seed = 123}
#'  
#' @param nc [integer]
#' Number of cores to use for parallel processing. Since fitting
#' optimal parameters for each subject is an independent task,
#' parallel computation can significantly speed up the fitting process:
#' \itemize{
#'   \item \strong{`nc = 1`}: The fitting proceeds sequentially.
#'   Parameters for one subject are fitted completely before moving
#'   to the next subject.
#'   \item \strong{`nc > 1`}: The fitting is performed in parallel
#'   across subjects. For example, if `nc = 4`, the algorithm will
#'   simultaneously fit data for four subjects. Once these are complete,
#'   it will proceed to fit the next batch of subjects (e.g., subjects
#'   5-8), and so on, until all subjects are processed.
#' }
#' 
#'  \code{default: nc = 1}
#'  
#' @param algorithm [character] 
#' Choose an algorithm package from
#'  `L-BFGS-B`, `GenSA`, `GA`, `DEoptim`, `PSO`, `Bayesian`, `CMA-ES`.
#'  
#' In addition, any algorithm from the `nloptr` package is also
#'  supported. If your chosen `nloptr` algorithm requires a local search,
#'  you need to input a character vector. The first element represents
#'  the algorithm used for global search, and the second element represents
#'  the algorithm used for local search.
#'
#' @returns 
#' The optimal parameters found by the algorithm for each subject,
#'  along with the model fit calculated using these parameters.
#'  This is returned as an object of class \code{binaryRL} containing results
#'  for all subjects with all models.
#'  
#' @examples
#' \dontrun{
#' comparison <- binaryRL::fit_p(
#'   data = binaryRL::Mason_2024_Exp2,
#'   id = unique(binaryRL::Mason_2024_Exp2$Subject),
#' ##-----------------------------------------------------------------------------##
#' ##----------------------------- black-box function ----------------------------##
#'   #funcs = c("your_funcs"),
#'   fit_model = list(binaryRL::TD, binaryRL::RSTD, binaryRL::Utility),
#'   model_name = c("TD", "RSTD", "Utility"),
#'   lower = list(c(0, 0), c(0, 0, 0), c(0, 0, 0)),
#'   upper = list(c(1, 10), c(1, 1, 10), c(1, 1, 10)),
#' ##----------------------------- interation number -----------------------------##
#'   iteration = 10,
#' ##-------------------------------- algorithms ---------------------------------##
#'   nc = 1,                 # <nc > 1>: parallel computation across subjects
#'   # Base R Optimization
#'   algorithm = "L-BFGS-B"  # Gradient-Based (stats)
#' ##-----------------------------------------------------------------------------##
#'   # Specialized External Optimization
#'   #algorithm = "GenSA"    # Simulated Annealing (GenSA)
#'   #algorithm = "GA"       # Genetic Algorithm (GA)
#'   #algorithm = "DEoptim"  # Differential Evolution (DEoptim)
#'   #algorithm = "PSO"      # Particle Swarm Optimization (pso)
#'   #algorithm = "Bayesian" # Bayesian Optimization (mlrMBO)
#'   #algorithm = "CMA-ES"   # Covariance Matrix Adapting (cmaes)
#' ##-----------------------------------------------------------------------------##
#'   # Optimization Library (nloptr)
#'   #algorithm = c("NLOPT_GN_MLSL", "NLOPT_LN_BOBYQA")
#' ##-------------------------------- algorithms ---------------------------------##
#' #################################################################################
#' )
#'
#' result <- dplyr::bind_rows(comparison)
#'
#' # Ensure the output directory exists before writing
#' if (!dir.exists("../OUTPUT")) {
#'   dir.create("../OUTPUT", recursive = TRUE)
#' }
#' 
#' write.csv(result, "../OUTPUT/result_comparison.csv", row.names = FALSE)
#' }
#' 
fit_p <- function(
  data,
  id = NULL,
  n_trials = NULL,
  fit_model = list(TD, RSTD, Utility),
  funcs = NULL,
  model_name = c("TD", "RSTD", "Utility"),
  lower = list(c(0, 0), c(0, 0, 0), c(0, 0, 0)),
  upper = list(c(1, 1), c(1, 1, 1), c(1, 1, 1)),
  initial_params = NA,
  initial_size = 50,
  iteration = 10,
  seed = 123,
  nc = 1,
  algorithm
){
  # 事前准备. 探测信息
  info <- suppressWarnings(suppressMessages(detect_information(data = data)))
  
  if (is.null(n_trials)) {
    n_trials <- info[["n_trials"]]
  }
  
  if (is.null(id)) {
    id <- info[["all_ids"]]
  }
  
  # 创建空list用于储存结果
  model_comparison <- list()
  model_result <- list()
  
  # Check for internally parallel algorithms
  if (nc == 1) {
    
    for (i in 1:length(fit_model)){
      
      message(paste0(
        "\n", 
        "Fitting Model: ", model_name[i], 
        "\n"
      ))
      
      n_subjects <- length(id)
      
      # 进度条
      progressr::handlers(progressr::handler_txtprogressbar)
      
      progressr::with_progress({
        
        p <- progressr::progressor(steps = n_subjects)
        
        for (j in 1:n_subjects) {
          
          p()
          
          n_params <- length(lower[[i]])
          
          binaryRL_res <- binaryRL::optimize_para(
            data = data,
            id = id[j],
            n_params = n_params,
            n_trials = n_trials,
            obj_func = fit_model[[i]],
            lower = lower[[i]],
            upper = upper[[i]],
            iteration = iteration,
            seed = seed,
            initial_params = initial_params,
            initial_size = initial_size,
            algorithm = algorithm 
          )
          
          model_result[[j]] <- data.frame(
            fit_model = model_name[i],
            Subject = id[j],
            ACC = binaryRL_res$acc,
            LogL = -binaryRL_res$ll,
            AIC = binaryRL_res$aic,
            BIC = binaryRL_res$bic
          )
          
          for (k in 1:n_params) {
            model_result[[j]][1, k + 6] <- binaryRL_res$output[k]
            names(model_result[[j]])[k + 6] <- paste0("param_", k)
          }
        }
      })
      model_comparison[[i]] <- model_result
    }
  }
  else {
    
    for (i in 1:length(fit_model)){
      
      message(paste0(
        "\n", 
        "Fitting Model: ", model_name[i], 
        "\n"
      ))
      
      if (base::.Platform$OS.type == "windows") {
        future::plan(future::multisession, workers = nc)
      } else { # 包括 macOS, Linux, Unix
        future::plan(future::multicore, workers = nc)
      }
      
      doFuture::registerDoFuture()
      
      n_subjects <- length(id)
      
      # 进度条
      progressr::handlers(progressr::handler_txtprogressbar)
      
      progressr::with_progress({
        
        p <- progressr::progressor(steps = n_subjects)
        
        doRNG::registerDoRNG(seed = seed)
        
        model_result <- foreach::foreach(
          j = 1:n_subjects, .combine = rbind,
          .packages = c("binaryRL"),
          .export = funcs
        ) %dorng% {
          n_params <- length(lower[[i]])
          
          binaryRL_res <- binaryRL::optimize_para(
            data = data,
            id = id[j],
            n_params = n_params,
            n_trials = n_trials,
            obj_func = fit_model[[i]],
            lower = lower[[i]],
            upper = upper[[i]],
            iteration = iteration,
            seed = seed,
            initial_params = initial_params,
            initial_size = initial_size,
            algorithm = algorithm
          )
          
          result_j <- data.frame(
            fit_model = model_name[i],
            Subject = id[j],
            ACC = binaryRL_res$acc,
            LogL = -binaryRL_res$ll,
            AIC = binaryRL_res$aic,
            BIC = binaryRL_res$bic
          )
          
          for (k in 1:n_params) {
            result_j[1, k + 6] <- binaryRL_res$output[k]
            names(result_j)[k + 6] <- paste0("param_", k)
          }
          
          # 在foreach循环内更新进度条
          p() 
          return(result_j)
        }
      })
      
      # 將結果包在一個 list 裡面，保持結構一致性
      model_comparison[[i]] <- list(model_result) 
    }
  }

  result <- model_comparison
  
  return(result)
}