% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayes_sim_unknownvar.R
\name{bayes_sim_unknownvar}
\alias{bayes_sim_unknownvar}
\title{Bayesian Simulation with Composite Sampling}
\usage{
bayes_sim_unknownvar(
  n,
  p = NULL,
  u,
  C,
  R,
  Xn = NULL,
  Vn,
  Vbeta_d,
  Vbeta_a_inv,
  mu_beta_d,
  mu_beta_a,
  a_sig_a,
  b_sig_a,
  a_sig_d,
  b_sig_d,
  alt = "greater",
  alpha,
  mc_iter
)
}
\arguments{
\item{n}{sample size (either vector or scalar)}

\item{p}{column dimension of design matrix \code{Xn}. If \code{Xn = NULL},
\code{p} must be specified to denote the column dimension of the default
design matrix generated
by the function.}

\item{u}{a scalar or vector to evaluate \deqn{u'\beta > C,}
where \eqn{\beta} is an unknown parameter that is to be estimated.}

\item{C}{constant value to be compared to when evaluating \eqn{u'\beta > C}}

\item{R}{number of iterations we want to pass through to check for
satisfaction of the analysis stage objective. The proportion of those
iterations meeting the analysis objective corresponds to the approximated
Bayesian assurance.}

\item{Xn}{design matrix that characterizes where the data is to be generated
from. This is specifically given by the normal linear regression model
\deqn{yn = Xn\beta + \epsilon,} \deqn{\epsilon ~ N(0, \sigma^2 Vn),}
where \eqn{\sigma^2} is unknown in this setting. Note that
\code{Xn} must have column dimension \code{p}.}

\item{Vn}{an \code{n} by \code{n} correlation matrix for the marginal distribution
of the sample data \code{yn}. Takes on an identity matrix when set to \code{NULL}.}

\item{Vbeta_d}{correlation matrix that helps describe the prior information
on \eqn{\beta} in the design stage}

\item{Vbeta_a_inv}{inverse-correlation matrix that helps describe the prior
information on \eqn{\beta} in the analysis stage}

\item{mu_beta_d}{design stage mean}

\item{mu_beta_a}{analysis stage mean}

\item{a_sig_a, b_sig_a}{shape and scale parameters of the inverse
gamma distribution where variance \eqn{\sigma^2}
is sampled from in the analysis stage}

\item{a_sig_d, b_sig_d}{shape and scale parameters of the inverse gamma
distribution where variance \eqn{\sigma^2} is sampled from in the design
stage}

\item{alt}{specifies alternative test case, where \code{alt = "greater"}
tests if \eqn{u'\beta > C}, \code{alt = "less"} tests if \eqn{u'\beta < C},
and \code{alt = "two.sided"} performs a two-sided test. By default,
\code{alt = "greater"}.}

\item{alpha}{significance level}

\item{mc_iter}{number of MC samples evaluated under the analysis objective}
}
\value{
a list of objects corresponding to the assurance approximations
\itemize{
\item{assurance_table:} table of sample size and corresponding assurance
values
\item{assur_plot:} plot of assurance values
\item{mc_samples:} number of Monte Carlo samples that were generated
and evaluated
}
}
\description{
Approximates the Bayesian assurance of a one-sided hypothesis test
through Monte Carlo sampling with the added assumption that
the variance is unknown.
}
\examples{

## O'Hagan and Stevens (2001) include a series of examples with 
## pre-specified parameters that we will be using to replicate their 
## results through our Bayesian assurance simulation.
## The inputs are as follows:
\donttest{
n <- 285
p <- 4 ## includes two parameter measures (cost and efficacy) for each of 
       ## the two treatments, for a total of p = 4 parameters. 
K <- 20000
C <- 0
u <- as.matrix(c(-K, 1, K, -1))

## Set up correlation matrices
Vbeta_a_inv <- matrix(rep(0, p^2), nrow = p, ncol = p)

sigsq <- 4.04^2
tau1 <- tau2 <- 8700
sig <- sqrt(sigsq)
Vn <- matrix(0, nrow = n*p, ncol = n*p)
Vn[1:n, 1:n] <- diag(n)
Vn[(2*n - (n-1)):(2*n), (2*n - (n-1)):(2*n)] <- (tau1 / sig)^2 * diag(n)
Vn[(3*n - (n-1)):(3*n), (3*n - (n-1)):(3*n)] <- diag(n)
Vn[(4*n - (n-1)):(4*n), (4*n - (n-1)):(4*n)] <- (tau2 / sig)^2 * diag(n)

Vbeta_d <- (1 / sigsq) * 
matrix(c(4, 0, 3, 0, 0, 10^7, 0, 0, 3, 0, 4, 0, 0, 0, 0, 10^7),
nrow = 4, ncol = 4)
mu_beta_d <- as.matrix(c(5, 6000, 6.5, 7200))
mu_beta_a <- as.matrix(rep(0, p))
alpha <- 0.05
epsilon <- 10e-7
a_sig_d <- (sigsq / epsilon) + 2
b_sig_d <- sigsq * (a_sig_d - 1)
a_sig_a <- -p / 2
b_sig_a <- 0

bayesassurance::bayes_sim_unknownvar(n = n, p = 4, 
u = as.matrix(c(-K, 1, K, -1)), C = 0, R = 40,
Xn = NULL, Vn = Vn, Vbeta_d = Vbeta_d, 
Vbeta_a_inv = Vbeta_a_inv, mu_beta_d = mu_beta_d,
mu_beta_a = mu_beta_a, a_sig_a = a_sig_a, b_sig_a = b_sig_a, 
a_sig_d = a_sig_d, b_sig_d = b_sig_d, alt = "two.sided", alpha = 0.05, 
mc_iter = 500)
}
}
\seealso{
\code{\link{bayes_sim}} for the Bayesian assurance function
for known variance.
}
