## Tests for tableby


context("Testing the tableby output")

options(stringsAsFactors=FALSE)
set.seed(100)
nsubj <- 90 # need multiple of 3
mdat <- data.frame(Group = c(rep("High", nsubj/3), rep("Med", nsubj/3), rep("Low", nsubj/3)),
                   Sex = sample(c("Male", "Female"), nsubj, replace=TRUE),
                   Age = round(rnorm(nsubj, mean=40, sd=5)),
                   Phase = ordered(sample(c("I", "II", "III"), nsubj, replace=TRUE), levels=c("I", "II", "III")),
                   ht_in = round(rnorm(nsubj, mean=65, sd=5)),
                   time = round(runif(nsubj,0,7)),
                   status = rbinom(nsubj, 1, prob=0.4),
                   dt = as.Date(round(rnorm(90, mean=100, sd=2000)), origin="1950/01/01"),
                   missing = as.character(NA),
                   trt = factor(sample(c("A", "B"), nsubj, replace=TRUE)),
                   ethan = factor(c(NA, NA, NA, sample(c("Ethan", "Heinzen"), nsubj - 3, replace=TRUE))),
                   weights = c(20, 1.5, rep(1, nsubj - 2)))
mdat$Group.fac <- factor(mdat$Group)
attr(mdat$ht_in, "label") <- "Height in Inches"
attr(mdat$trt, "label") <- "Treatment Arm"
attr(mdat$Age, "label") <- "Age in Years"

###########################################################################################################
#### Basic two-sided tableby
###########################################################################################################

test_that("A basic two-sided tableby call--no labels, no missings", {
  expect_identical(
    capture.output(summary(tableby(Group ~ Sex + time + dt, data = mdat), text = TRUE)),
    c("--------------------------------------------------------------------------------------------",
      "                   High (N=30)    Low (N=30)     Med (N=30)     Total (N=90)   p value      ",
      "----------------- -------------- -------------- -------------- -------------- --------------",
      "Sex                                                                                    0.733",
      "   Female         15 (50%)       17 (56.7%)     14 (46.7%)     46 (51.1%)    "               ,
      "   Male           15 (50%)       13 (43.3%)     16 (53.3%)     44 (48.9%)    "               ,
      "time                                                                                   0.025",
      "   Mean (SD)      4.57 (1.81)    3.17 (2.04)    3.83 (2)       3.86 (2.01)   "               ,
      "   Q1, Q3         3.25, 6        1.25, 5        2, 5           2, 6          "               ,
      "   Range          0 - 7          0 - 6          1 - 7          0 - 7         "               ,
      "dt                                                                                     0.391",
      "   median         1950-01-07     1951-06-13     1948-09-13     1949-10-07    "               ,
      "   Range          1935-08-15 -   1937-02-08 -   1939-04-01 -   1935-08-15 -  "               ,
      "                  1968-05-14     1959-09-06     1958-07-30     1968-05-14    "               ,
      "--------------------------------------------------------------------------------------------"
    )
  )
})

test_that("A basic two-sided tableby call--labels, no missings", {
  expect_identical(
    capture.output(summary(tableby(Group ~ Age + trt, data = mdat), text = TRUE)),
    c("--------------------------------------------------------------------------------------------",
      "                   High (N=30)    Low (N=30)     Med (N=30)     Total (N=90)   p value      ",
      "----------------- -------------- -------------- -------------- -------------- --------------",
      "Age in Years                                                                           0.906",
      "   Mean (SD)      40 (6.22)      39.6 (3.87)    39.4 (5.57)    39.7 (5.26)   "               ,
      "   Q1, Q3         36, 44.5       37.2, 41.8     35.2, 44       36, 43        "               ,
      "   Range          29 - 53        32 - 48        30 - 52        29 - 53       "               ,
      "Treatment Arm                                                                          0.659",
      "   A              14 (46.7%)     11 (36.7%)     11 (36.7%)     36 (40%)      "               ,
      "   B              16 (53.3%)     19 (63.3%)     19 (63.3%)     54 (60%)      "               ,
      "--------------------------------------------------------------------------------------------"
    )
  )
})

test_that("A basic two-sided tableby call--no labels, some missings", {
  expect_identical(
    capture.output(summary(tableby(Group ~ ethan, data = mdat), text = TRUE)),
    c("--------------------------------------------------------------------------------------------",
      "                   High (N=30)    Low (N=30)     Med (N=30)     Total (N=90)   p value      ",
      "----------------- -------------- -------------- -------------- -------------- --------------",
      "ethan                                                                                  0.178",
      "   N-Miss         3              0              0              3             "               ,
      "   Ethan          17 (63%)       13 (43.3%)     12 (40%)       42 (48.3%)    "               ,
      "   Heinzen        10 (37%)       17 (56.7%)     18 (60%)       45 (51.7%)    "               ,
      "--------------------------------------------------------------------------------------------"
    )
  )
})

###########################################################################################################
#### Basic one-sided tableby
###########################################################################################################

test_that("A basic one-sided tableby call--no labels, no missings", {
  expect_identical(
    capture.output(summary(tableby(~ Sex + time + dt, data = mdat), text = TRUE)),
    c("------------------------------------",
      "                     Overall (N=90) ",
      "------------------- ----------------",
      "Sex                "                 ,
      "   Female           46 (51.1%)      ",
      "   Male             44 (48.9%)      ",
      "time               ",
      "   Mean (SD)        3.86 (2.01)     ",
      "   Q1, Q3           2, 6            ",
      "   Range            0 - 7           ",
      "dt                 "                 ,
      "   median           1949-10-07      ",
      "   Range            1935-08-15 -    ",
      "                    1968-05-14      ",
      "------------------------------------"
    )
  )
})

test_that("A basic one-sided tableby call--labels, no missings", {
  expect_identical(
    capture.output(summary(tableby(~ Age + trt, data = mdat), text = TRUE)),
    c("------------------------------------",
      "                     Overall (N=90) ",
      "------------------- ----------------",
      "Age in Years       "                 ,
      "   Mean (SD)        39.7 (5.26)     ",
      "   Q1, Q3           36, 43          ",
      "   Range            29 - 53         ",
      "Treatment Arm      "                 ,
      "   A                36 (40%)        ",
      "   B                54 (60%)        ",
      "------------------------------------"
    )
  )
})

test_that("A basic one-sided tableby call--no labels, some missings (Sarah Jenkins's Error)", {
  expect_identical(
    capture.output(summary(tableby(~ ethan, data = mdat), text = TRUE)),
    c("------------------------------------",
      "                     Overall (N=90) ",
      "------------------- ----------------",
      "ethan              "                 ,
      "   N-Miss           3               ",
      "   Ethan            42 (48.3%)      ",
      "   Heinzen          45 (51.7%)      ",
      "------------------------------------"
    )
  )
})

###########################################################################################################
#### Change totals/p-values
###########################################################################################################

test_that("A basic two-sided tableby call--no p-value, no total", {
  expect_identical(
    capture.output(summary(tableby(Group ~ Age + Sex, data = mdat, test = FALSE, total = FALSE), text = TRUE)),
    c("----------------------------------------------------------",
      "                  High (N=30)   Low (N=30)    Med (N=30)  ",
      "---------------- ------------- ------------- -------------",
      "Age in Years    "                                          ,
      "   Mean (SD)     40 (6.22)     39.6 (3.87)   39.4 (5.57)  ",
      "   Q1, Q3        36, 44.5      37.2, 41.8    35.2, 44     ",
      "   Range         29 - 53       32 - 48       30 - 52      ",
      "Sex             "                                          ,
      "   Female        15 (50%)      17 (56.7%)    14 (46.7%)   ",
      "   Male          15 (50%)      13 (43.3%)    16 (53.3%)   ",
      "----------------------------------------------------------"
    )
  )
  expect_identical(
    capture.output(summary(tableby(Group ~ Age + Sex, data = mdat), test = FALSE, total = FALSE, text = TRUE)),
    c("----------------------------------------------------------",
      "                  High (N=30)   Low (N=30)    Med (N=30)  ",
      "---------------- ------------- ------------- -------------",
      "Age in Years    "                                          ,
      "   Mean (SD)     40 (6.22)     39.6 (3.87)   39.4 (5.57)  ",
      "   Q1, Q3        36, 44.5      37.2, 41.8    35.2, 44     ",
      "   Range         29 - 53       32 - 48       30 - 52      ",
      "Sex             "                                          ,
      "   Female        15 (50%)      17 (56.7%)    14 (46.7%)   ",
      "   Male          15 (50%)      13 (43.3%)    16 (53.3%)   ",
      "----------------------------------------------------------"
    )
  )
})

test_that("A basic two-sided tableby call--p-value, no total", {
  expect_identical(
    capture.output(summary(tableby(Group ~ Age + Sex, data = mdat, total = FALSE), text = TRUE)),
    c("------------------------------------------------------------------------",
      "                  High (N=30)   Low (N=30)    Med (N=30)    p value     ",
      "---------------- ------------- ------------- ------------- -------------",
      "Age in Years                                                       0.906",
      "   Mean (SD)     40 (6.22)     39.6 (3.87)   39.4 (5.57)  "              ,
      "   Q1, Q3        36, 44.5      37.2, 41.8    35.2, 44     "              ,
      "   Range         29 - 53       32 - 48       30 - 52      "              ,
      "Sex                                                                0.733",
      "   Female        15 (50%)      17 (56.7%)    14 (46.7%)   "              ,
      "   Male          15 (50%)      13 (43.3%)    16 (53.3%)   "              ,
      "------------------------------------------------------------------------"
    )
  )
  expect_identical(
    capture.output(summary(tableby(Group ~ Age + Sex, data = mdat), total = FALSE, text = TRUE)),
    c("------------------------------------------------------------------------",
      "                  High (N=30)   Low (N=30)    Med (N=30)    p value     ",
      "---------------- ------------- ------------- ------------- -------------",
      "Age in Years                                                       0.906",
      "   Mean (SD)     40 (6.22)     39.6 (3.87)   39.4 (5.57)  "              ,
      "   Q1, Q3        36, 44.5      37.2, 41.8    35.2, 44     "              ,
      "   Range         29 - 53       32 - 48       30 - 52      "              ,
      "Sex                                                                0.733",
      "   Female        15 (50%)      17 (56.7%)    14 (46.7%)   "              ,
      "   Male          15 (50%)      13 (43.3%)    16 (53.3%)   "              ,
      "------------------------------------------------------------------------"
    )
  )
})

###########################################################################################################
#### markdown output
###########################################################################################################

test_that("A basic two-sided tableby markdown output", {
  expect_identical(
    capture.output(summary(tableby(Group ~ Age + Sex + ethan + dt, data = mdat, total = FALSE))),
    c("------------------------------------------------------------------------",
      "                  High (N=30)   Low (N=30)    Med (N=30)    p value     ",
      "---------------- ------------- ------------- ------------- -------------",
      "**Age in Years**                                                   0.906",
      ""                                                                        ,
      "&nbsp;&nbsp;     40 (6.22)     39.6 (3.87)   39.4 (5.57)  "              ,
      "&nbsp;                                                    "              ,
      "Mean (SD)                                                 "              ,
      ""                                                                        ,
      "&nbsp;&nbsp;     36, 44.5      37.2, 41.8    35.2, 44     "              ,
      "&nbsp;                                                    "              ,
      "Q1, Q3                                                    "              ,
      ""                                                                        ,
      "&nbsp;&nbsp;     29 - 53       32 - 48       30 - 52      "              ,
      "&nbsp;                                                    "              ,
      "Range                                                     "              ,
      ""                                                                        ,
      "**Sex**                                                            0.733",
      ""                                                                        ,
      "&nbsp;&nbsp;     15 (50%)      17 (56.7%)    14 (46.7%)   "              ,
      "&nbsp;                                                    "              ,
      "Female                                                    "              ,
      ""                                                                        ,
      "&nbsp;&nbsp;     15 (50%)      13 (43.3%)    16 (53.3%)   "              ,
      "&nbsp;                                                    "              ,
      "Male                                                      "              ,
      ""                                                                        ,
      "**ethan**                                                          0.178",
      ""                                                                        ,
      "&nbsp;&nbsp;     3             0             0            "              ,
      "&nbsp;                                                    "              ,
      "N-Miss                                                    "              ,
      ""                                                                        ,
      "&nbsp;&nbsp;     17 (63%)      13 (43.3%)    12 (40%)     "              ,
      "&nbsp;                                                    "              ,
      "Ethan                                                     "              ,
      ""                                                                        ,
      "&nbsp;&nbsp;     10 (37%)      17 (56.7%)    18 (60%)     "              ,
      "&nbsp;                                                    "              ,
      "Heinzen                                                   "              ,
      ""                                                                        ,
      "**dt**                                                             0.391",
      ""                                                                        ,
      "&nbsp;&nbsp;     1950-01-07    1951-06-13    1948-09-13   "              ,
      "&nbsp;                                                    "              ,
      "median                                                    "              ,
      ""                                                                        ,
      "&nbsp;&nbsp;     1935-08-15 -  1937-02-08 -  1939-04-01 - "              ,
      "&nbsp;           1968-05-14    1959-09-06    1958-07-30   "              ,
      "Range                                                     "              ,
      ""                                                                        ,
      "------------------------------------------------------------------------"
    )
  )
})


###########################################################################################################
#### Other warnings and tests and things...
###########################################################################################################

test_that("A warning occurs using one-sided formula and na.tableby", {
  expect_warning(tableby(~ ethan, data = mdat, na.action = na.tableby))
})

test_that("The by-variable droplevels is working correctly", {
  expect_identical(
    capture.output(summary(tableby(Group.fac ~ Sex + time + dt, data = mdat[mdat$Group.fac %in% c("High", "Low"), ]), text = TRUE)),
    c("-----------------------------------------------------------------------------",
      "                   High (N=30)    Low (N=30)     Total (N=60)   p value      ",
      "----------------- -------------- -------------- -------------- --------------",
      "Sex                                                                     0.796",
      "   Female         15 (50%)       17 (56.7%)     32 (53.3%)    "               ,
      "   Male           15 (50%)       13 (43.3%)     28 (46.7%)    "               ,
      "time                                                                    0.007",
      "   Mean (SD)      4.57 (1.81)    3.17 (2.04)    3.87 (2.04)   "               ,
      "   Q1, Q3         3.25, 6        1.25, 5        2, 6          "               ,
      "   Range          0 - 7          0 - 6          0 - 7         "               ,
      "dt                                                                      0.574",
      "   median         1950-01-07     1951-06-13     1950-07-02    "               ,
      "   Range          1935-08-15 -   1937-02-08 -   1935-08-15 -  "               ,
      "                  1968-05-14     1959-09-06     1968-05-14    "               ,
      "-----------------------------------------------------------------------------"
    )
  )
})

test_that("Using cat.simplify", {
  expect_identical(
    capture.output(summary(tableby(Group ~ Sex + trt, data = mdat, cat.simplify = TRUE), text = TRUE)),
    c("--------------------------------------------------------------------------------------------",
      "                   High (N=30)    Low (N=30)     Med (N=30)     Total (N=90)   p value      ",
      "----------------- -------------- -------------- -------------- -------------- --------------",
      "Sex               15 (50%)       13 (43.3%)     16 (53.3%)     44 (48.9%)              0.733",
      "Treatment Arm     16 (53.3%)     19 (63.3%)     19 (63.3%)     54 (60%)                0.659",
      "--------------------------------------------------------------------------------------------"
    )
  )
})


test_that("Reordering variables", {
  expect_identical(
    capture.output(summary(tableby(Group ~ Sex + dt + Age, data = mdat)[c(3,1,2)], text = TRUE)),
    c("--------------------------------------------------------------------------------------------",
      "                   High (N=30)    Low (N=30)     Med (N=30)     Total (N=90)   p value      ",
      "----------------- -------------- -------------- -------------- -------------- --------------",
      "Age in Years                                                                           0.906",
      "   Mean (SD)      40 (6.22)      39.6 (3.87)    39.4 (5.57)    39.7 (5.26)   "               ,
      "   Q1, Q3         36, 44.5       37.2, 41.8     35.2, 44       36, 43        "               ,
      "   Range          29 - 53        32 - 48        30 - 52        29 - 53       "               ,
      "Sex                                                                                    0.733",
      "   Female         15 (50%)       17 (56.7%)     14 (46.7%)     46 (51.1%)    "               ,
      "   Male           15 (50%)       13 (43.3%)     16 (53.3%)     44 (48.9%)    "               ,
      "dt                                                                                     0.391",
      "   median         1950-01-07     1951-06-13     1948-09-13     1949-10-07    "               ,
      "   Range          1935-08-15 -   1937-02-08 -   1939-04-01 -   1935-08-15 -  "               ,
      "                  1968-05-14     1959-09-06     1958-07-30     1968-05-14    "               ,
      "--------------------------------------------------------------------------------------------"
    )
  )
})


test_that("Merging tableby objects", {
  expect_error(merge(tableby(Group ~ Sex, data = mdat), tableby(Group.fac ~ Age, data = mdat)))
  expect_identical(
    capture.output(summary(merge(tableby(Group ~ Sex, data = mdat), tableby(Group ~ Age, data = mdat)), text = TRUE)),
    c("--------------------------------------------------------------------------------------------",
      "                   High (N=30)    Low (N=30)     Med (N=30)     Total (N=90)   p value      ",
      "----------------- -------------- -------------- -------------- -------------- --------------",
      "Sex                                                                                    0.733",
      "   Female         15 (50%)       17 (56.7%)     14 (46.7%)     46 (51.1%)    "               ,
      "   Male           15 (50%)       13 (43.3%)     16 (53.3%)     44 (48.9%)    "               ,
      "Age in Years                                                                           0.906",
      "   Mean (SD)      40 (6.22)      39.6 (3.87)    39.4 (5.57)    39.7 (5.26)   "               ,
      "   Q1, Q3         36, 44.5       37.2, 41.8     35.2, 44       36, 43        "               ,
      "   Range          29 - 53        32 - 48        30 - 52        29 - 53       "               ,
      "--------------------------------------------------------------------------------------------"
    )
  )
})


test_that("Changing tests", {
  expect_identical(
    capture.output(summary(tableby(Group ~ fe(Sex) + kwt(Age), data = mdat), text = TRUE)),
    c("--------------------------------------------------------------------------------------------",
      "                   High (N=30)    Low (N=30)     Med (N=30)     Total (N=90)   p value      ",
      "----------------- -------------- -------------- -------------- -------------- --------------",
      "Sex                                                                                    0.806",
      "   Female         15 (50%)       17 (56.7%)     14 (46.7%)     46 (51.1%)    "               ,
      "   Male           15 (50%)       13 (43.3%)     16 (53.3%)     44 (48.9%)    "               ,
      "Age in Years                                                                           0.869",
      "   Mean (SD)      40 (6.22)      39.6 (3.87)    39.4 (5.57)    39.7 (5.26)   "               ,
      "   Q1, Q3         36, 44.5       37.2, 41.8     35.2, 44       36, 43        "               ,
      "   Range          29 - 53        32 - 48        30 - 52        29 - 53       "               ,
      "--------------------------------------------------------------------------------------------"
    )
  )
  
  expect_identical(
    capture.output(summary(tableby(Group ~ Sex + Age, data = mdat, numeric.test = "kwt", cat.test = "fe"), text = TRUE)),
    c("--------------------------------------------------------------------------------------------",
      "                   High (N=30)    Low (N=30)     Med (N=30)     Total (N=90)   p value      ",
      "----------------- -------------- -------------- -------------- -------------- --------------",
      "Sex                                                                                    0.806",
      "   Female         15 (50%)       17 (56.7%)     14 (46.7%)     46 (51.1%)    "               ,
      "   Male           15 (50%)       13 (43.3%)     16 (53.3%)     44 (48.9%)    "               ,
      "Age in Years                                                                           0.869",
      "   Mean (SD)      40 (6.22)      39.6 (3.87)    39.4 (5.57)    39.7 (5.26)   "               ,
      "   Q1, Q3         36, 44.5       37.2, 41.8     35.2, 44       36, 43        "               ,
      "   Range          29 - 53        32 - 48        30 - 52        29 - 53       "               ,
      "--------------------------------------------------------------------------------------------"
    )
  )
})


###########################################################################################################
#### Reported bugs for tableby
###########################################################################################################


