\name{disscenter} 
\alias{disscenter} 
\encoding{latin1} 
\title{Compute distance to the center of a group} 

\description{ 
	Compute the dissimilarity between a set of objects and their group center using a pairwise dissimilarity matrix.
}

\usage{
disscenter(diss, group=NULL, medoids.index=NULL, allcenter = FALSE)
}
\arguments{
	\item{diss}{a dissimilarity matrix such as generated by \code{\link{seqdist}}, or a \code{dist} object (see \code{\link{dist}}}
	\item{group}{if null, only one group is considered, otherwise group to compute center}
%  \item{trim}{percentage of individuals that are considered as outliers and excluded from the computation of distance to center}
	\item{medoids.index}{if NULL, return dissimilarity to center. If equal to "first", return the index of the first encountered most central sequence. One index per group is returned. If equal to "all", all medoids index are returned. If \code{group} is set, one list per group is returned.}
	\item{allcenter}{logical. If \code{TRUE}, returns a \code{data.frame} containing the dissimilarity between each object and its group center, each column corresponding to a group.}
%  \item{max.iter}{maximum number of iteration to trim the outliers}
}
\details{
	This function computes the dissimilarity between given objects and their group center. The group center may not belong to the space formed by the objects (in the same way, the average do not belong to a space formed by discrete measure).
	This distance can also be understood as the contribution to the discrepancy (see \code{\link{dissvar}}).
	The dissimilarity between a given object and its group center may be negative if the dissimilarity measure does not respect the triangle inequality.
  
	It can be shown that this dissimilarity is equal to \cite{Batagelj (1988)}:
		\deqn{d_{x\tilde{g}}=\frac{1}{n}\big(\sum_{i=1}^{n}d_{xi}-SS\big)}{d_(xg)=1/n *(sum d_xi - SS)}
	Where \eqn{SS} is the sum of squares (see \code{\link{dissvar}}).
}
\value{
	A vector with the dissimilarity to center of group for each sequence, or a list of medoid indexes.
}

\references{
	Studer, M., G. Ritschard, A. Gabadinho, and N. S. Mller (2009) 
	Discrepancy	analysis of complex objects using dissimilarities.
	In H. Briand, F. Guillet, G. Ritschard, and D. A. Zighed (Eds.),
	\emph{Advances in Knowledge Discovery and Management},
	Studies in Computational Intelligence. Berlin: Springer.

	Studer, M., G. Ritschard, A. Gabadinho and N. S. Mller (2009)
	Analyse de dissimilarits par arbre d'induction. In EGC 2009,
	\emph{Revue des Nouvelles Technologies de l'Information}, Vol. E-15, pp. 7--18.

	Batagelj, V. (1988) Generalized ward and related clustering problems. In H. Bock (Ed.),
	\emph{Classification and related methods of data analysis}, Amsterdam: North-Holland, pp. 67--74.
 
}

\seealso{
  \code{\link{dissvar}} to compute the pseudo variance from dissimilarities and for a basic introduction to concepts of pseudo variance analysis
  \cr
  \code{\link{dissassoc}} to test association between objects represented by their dissimilarities and a covariate.
  \cr
  \code{\link{disstree}} for an induction tree analyse of objects characterized by a dissimilarity matrix.
  \cr
  \code{\link{dissmfac}} to perform multi-factor analysis of variance from pairwise dissimilarities.
}

\examples{
## Defining a state sequence object
data(mvad)
mvad.seq <- seqdef(mvad[, 17:86])

## Building dissimilarities
mvad.lcs <- seqdist(mvad.seq, method="LCS")

## Compute distance to center according to group gcse5eq
dc <- disscenter(mvad.lcs, group=mvad$gcse5eq)

## Ploting distribution of dissimilarity  to center 
boxplot(dc~mvad$gcse5eq, col="cyan")

## Retrieving index of the first medoids, one per group
dc <- disscenter(mvad.lcs, group=mvad$Grammar, medoids.index="first")
print(dc)

## Retrieving index of all medoids in each group
dc <- disscenter(mvad.lcs, group=mvad$Grammar, medoids.index="all")
print(dc)
}

