######################
# GLOBAL LIBRARY #####
######################

require(DT)
require(shinyjs)

######################
# UI #################
######################

ui=tagList(
  useShinyjs(),
  # !! ONLINE VERSION ####
  # tags$head(includeScript("google-analytics.js")),
  navbarPage(title="ShinyItemAnalysis - TEST AND ITEM ANALYSIS",
             collapsible=TRUE,
             footer=list(
               ############################################
               # !! ONLINE VERSION ####
               # div(class="panel-footer",
               #     p(strong("ShinyItemAnalysis Version 1.1.0")),
               #     p("Download ShinyItemAnalysis R package from ",
               #       a(strong("CRAN"), href = "https://cran.rstudio.com/web/packages/ShinyItemAnalysis/",
               #         target = "_blank"), "to run analysis faster!"),
               #     p("Project was supported by grant funded by Czech Science Foundation under number ",
               #       a("GJ15-15856Y",
               #         href = "http://www.cs.cas.cz/martinkova/psychometrics.html",
               #         target = "_blank")),
               #     p("Copyright 2017  Patricia Martinkova, Adela Drabinova, Ondrej Leder and Jakub Houdek"),
               #     div(
               #       HTML('<p style="font-size: 9pt">
               #            See older versions:
               #            <a href = "https://shiny.cs.cas.cz/ShinyItemAnalysisV01/"> 0.1.0</a>,
               #            <a href = "https://shiny.cs.cas.cz/ShinyItemAnalysisV02/"> 0.2.0</a>,
               #            <a href = "https://shiny.cs.cas.cz/ShinyItemAnalysisV100/"> 1.0.0</a>
               #            </p>')
               #       ),
               #     p(textOutput('counter'))
               #       )
               ############################################
               # !! PACKAGE VERSION ####
               div(class="panel-footer",
                   p(strong("ShinyItemAnalysis Version 1.1.0")),
                   p("You can also try ", code('ShinyItemAnalysis'),
                     a('online!', href = "https://shiny.cs.cas.cz/ShinyItemAnalysis/",
                       target = "_blank")),
                   p("Project was supported by grant funded by Czech Science Foundation under number ",
                     a("GJ15-15856Y",
                       href = "http://www.cs.cas.cz/martinkova/psychometrics.html",
                       target = "_blank")),
                   p("Copyright 2017  Patricia Martinkova, Adela Drabinova, Ondrej Leder and Jakub Houdek"),
                   div(
                     HTML('<p style="font-size: 9pt">
                          See older versions:
                          <a href = "https://shiny.cs.cas.cz/ShinyItemAnalysisV01/"> 0.1.0</a>,
                          <a href = "https://shiny.cs.cas.cz/ShinyItemAnalysisV02/"> 0.2.0</a>,
                          <a href = "https://shiny.cs.cas.cz/ShinyItemAnalysisV100/"> 1.0.0</a>
                          </p>')
                     )
                     )
               ############################################
                   ),
             theme="bootstrap.css",
             ######################
             # MAIN PANEL #########
             ######################
             ######################
             # ABOUT ##############
             ######################
             tabPanel("About",
                      h3('Description'),
                      p(code('ShinyItemAnalysis'), ' provides analysis of educational tests (such as admission tests)
                        and their items including:' ),
                      tags$ul(
                        tags$li("Exploration of total and standard scores on ", strong('Summary'), "page. "),
                        tags$li("Item and distractor analysis on ", strong('Traditional Analysis'), "page. "),
                        tags$li('Item analysis by logistic models on ', strong('Regression'), "page. "),
                        tags$li('Item analysis by item response theory models on ', strong('IRT models'), "page. "),
                        tags$li('Differential item functioning (DIF) and differential distractor functioning (DDF)
                                methods on ', strong('DIF/Fairness'), "page. ")
                        ),
                      p('This application is based on the free statistical software',
                        a('R', href = 'https://cran.r-project.org/', target = "_blank"),
                        ' and its ',
                        a('Shiny', href = 'http://www.rstudio.com/shiny/', target = "_blank"),
                        'package. '),
                      p('For all graphical outputs a download button is provided. Moreover, on ', strong('Reports'),
                        'page HTML or PDF report can be created. Additionaly, all application outputs are
                        complemented by selected R code hence the similar analysis can be run and modified in R.'),
                      ############################################
                      # # !! ONLINE VERSION ####
                      # p('You can also download ', code('ShinyItemAnalysis'), ' package from ',
                      #   a('CRAN', href = 'https://CRAN.R-project.org/package=ShinyItemAnalysis', target = "_blank"),
                      #   'to use it offline or run it faster. '),
                      ############################################
                      # !! PACKAGE VERSION ####
                      p('You can also try ', code('ShinyItemAnalysis'), ' application ',
                        a('online!', href = 'https://shiny.cs.cas.cz/ShinyItemAnalysis/', target = "_blank")),
                      ############################################
                      h4('Data'),
                      p('For demonstration purposes, by default, 20-item dataset', code('GMAT'),'
                        from R ', code('difNLR'),' package is used. Other three datasets are available: ',
                        code('GMAT2'), 'and', code('Medical 20 DIF'), 'from', code('difNLR'), 'package and ',
                        code('Medical 100'), 'from', code('ShinyItemAnalysis'), 'package.
                        You can change the dataset (and try your own one) on page', strong('Data.')),

                      h4('Version'),
                      p('Current version of ', code('ShinyItemAnalysis'), ' is 1.1.0'),
                      div(
                        HTML('<p>
                             See also older versions:
                             <a href = "https://shiny.cs.cas.cz/ShinyItemAnalysisV01/"> 0.1.0</a>,
                             <a href = "https://shiny.cs.cas.cz/ShinyItemAnalysisV02/"> 0.2.0</a>,
                             <a href = "https://shiny.cs.cas.cz/ShinyItemAnalysisV100/"> 1.0.0</a>
                             </p>')
                        ),

                      h4('List of packages used'),
                      code('library(corrplot)'), br(),
                      code('library(CTT)'), br(),
                      code('library(deltaPlotR)'), br(),
                      code('library(DT)'), br(),
                      code('library(difNLR)'), br(),
                      code('library(difR)'), br(),
                      code('library(ggplot2)'), br(),
                      code('library(grid)'), br(),
                      code('library(gridExtra)'), br(),
                      code('library(latticeExtra)'), br(),
                      code('library(ltm)'), br(),
                      code('library(mirt)'), br(),
                      code('library(moments)'), br(),
                      code('library(msm)'), br(),
                      code('library(nnet)'), br(),
                      code('library(psych)'), br(),
                      code('library(psychometric)'), br(),
                      code('library(reshape2)'), br(),
                      code('library(rmarkdown)'), br(),
                      code('library(shiny)'), br(),
                      code('library(shinyjs)'), br(),
                      code('library(stringr)'), br(),
                      code('library(WrightMap)'), br(),

                      h4('Authors'),
                      img(src = 'patricia.jpg', width = 70),
                      p(a("Patricia Martinkova, Institute of Computer Science, Czech Academy of Sciences",
                          href = "http://www.cs.cas.cz/martinkova/", target = "_blank")),
                      img(src = 'adela.jpg', width = 70),
                      p("Adela Drabinova"),
                      img(src = 'leder.png', width = 70),
                      p(a("Ondrej Leder", href = "https://www.linkedin.com/in/ond%C5%99ej-leder-3864b1119",
                          target = "_blank")),
                      img(src = 'jakub.jpg', width = 70),
                      p("Jakub Houdek"),

                      h4('Bug reports'),
                      p("If you discover a problem with this application please contact the project maintainer
                        at martinkova(at)cs.cas.cz or use ", a("GitHub.", href = "https://github.com/patriciamar/ShinyItemAnalysis/issues",
                                                               target = "_blank")),

                      h4('Acknowledgments'),
                      p(" Project was supported by grant funded by Czech Science foundation under number ",
                        a("GJ15-15856Y.", href = "http://www.cs.cas.cz/martinkova/psychometrics.html",
                          target = "_blank")),

                      h4('License'),
                      p("Copyright 2016  Patricia Martinkova, Adela Drabinova, Ondrej Leder and Jakub Houdek"),
                      p("This program is free software you can redistribute it and or modify it under the terms of the GNU
                        General Public License as published by the Free Software Foundation either version 3 of the License or
                        at your option any later version."),
                      p("This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
                        even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
                        Public License for more details." ),
                      br(),
                      br()
                      ),
             ###################
             # DATA ############
             ###################
             tabPanel("Data",
                      h3("Data"),
                      p('For demonstration purposes, 20-item dataset ' , code("GMAT"),'
                         and dataset', code("GMATkey"),' from R ', code('difNLR'),' package are used.
                         On this page, you may select one of four datasets offered from ', code('difNLR'),
                        ' and ', code('ShinyItemAnalysis'), 'packages or you may upload your own dataset
                         (see below). To return to demonstration dataset,
                         refresh this page in your browser' , strong("(F5)"), '.'),
                      p('Used dataset ', code("GMAT"), ' is generated based on parameters of real Graduate Management
                        Admission Test (GMAT) data set (Kingston et al., 1985). However, first two items were
                        generated to function differently in uniform and non-uniform way respectively.
                        The data set represents responses of 2,000 subjects (1,000 males, 1,000 females) to
                        multiple-choice test of 20 items. The distribution of total scores is the same for both groups. '),
                      p('Dataset ', code("GMAT2"), ' is also generated based on parameters of GMAT (Kingston et
                         al., 1985) from R ', code('difNLR'),' package . Again, first two items were generated
                         to function differently in uniform and non-uniform way respectively. The data set
                         represents responses of 1,000 subjects (500 males, 500 females) to multiple-choice test
                         of 20 items. '),
                      p('Dataset ', code("Medical 20 DIF"), ' is a subset of real admission test to medical
                         school from R ', code('difNLR'),' package. First item was previously detected as
                         functioning differently. The data set represents responses of 1,407 subjects (484 males,
                         923 females) to multiple-choice test of 20 items. For more details of item selection
                         see Drabinova & Martinkova (2016).'),
                      p('Dataset ', code("Medical 100"), ' is a real data set of admission test to medical school
                         from R ', code('ShinyItemAnalysis'),' package . The data set represents responses of
                         3,204 subjects to multiple-choice test of 100 items. There is no group membership
                         variable in the data set hence it is not possible to run DIF or DDF detection procedures. '),
                      br(),
                      selectInput("dataSelect", "Select dataset",
                                  c("GMAT" = "GMAT_difNLR",
                                    "GMAT2" = "GMAT2_difNLR",
                                    "Medical 20 DIF" = "difMedical_difNLR",
                                    "Medical 100" = "dataMedical_ShinyItemAnalysis"
                                  ),
                                  selected="GMAT_difNLR"),
                      h4("Upload your own datasets"),
                      p('Main dataset should contain responses of individual students (rows) to given items
                        (columns). Header may contain item names, no row names should be included. If responses
                        are in unscored ABCD format, the key provides correct response for each item. If responses are
                        scored 0-1, key is vector of 1s. Group is 0-1 vector, where 0 represents reference group
                        and 1 represents focal group. Its length need to be the same as number of individual
                        students in main dataset. If the group is not provided then it wont be possible to run DIF and DDF
                        detection procedures. In all data sets header should be either included or excluded. '),
                      fluidRow(
                        column(3, offset = 0, fileInput(
                          'data', 'Choose data (csv file)',
                          accept = c('text/csv',
                                     'text/comma-separated-values',
                                     'text/tab-separated-values',
                                     'text/plain',
                                     '.csv',
                                     '.tsv'
                          )
                        )
                        ),
                        column(3, fileInput(
                          'key', 'Choose key (csv file)',
                          accept = c('text/csv',
                                     'text/comma-separated-values',
                                     'text/tab-separated-values',
                                     'text/plain',
                                     '.csv',
                                     '.tsv'
                          )
                        )
                        ),
                        column(3, fileInput(
                          'groups', 'Choose groups for DIF (optional)',
                          accept = c('text/csv',
                                     'text/comma-separated-values',
                                     'text/tab-separated-values',
                                     'text/plain',
                                     '.csv',
                                     '.tsv'
                          )
                        )
                        ),
                        column(3, offset = 1, actionButton(inputId = "submitButton", label = "Submit Data"))
                      ),
                      tags$hr(),
                      h4("Data specification"),
                      fluidRow(
                        column(1, offset = 0, checkboxInput('header', 'Header', TRUE)),
                        column(3, offset = 1, radioButtons('sep', 'Separator',
                                                           c(Comma = ',',
                                                             Semicolon = ';',
                                                             Tab = '\t'
                                                           ),
                                                           ','
                        )
                        ),
                        column (3, offset = 0, radioButtons('quote', 'Quote',
                                                            c(None = '',
                                                              'Double Quote' = '"',
                                                              'Single Quote' = "'"
                                                            ),
                                                            '"'
                        )
                        )
                      ),
                      tags$hr(),
                      h4("Data check"),
                      DT::dataTableOutput('headdata'),
                      h4("Key (correct answers)"),
                      DT::dataTableOutput('key'),
                      h4("Scored test"),
                      DT::dataTableOutput('sc01'),
                      h4("Group vector"),
                      DT::dataTableOutput('group')
                      ),
             ########################
             # SUMMARY ##############
             ########################
             navbarMenu("Summary",
                        # * TOTAL SCORES ####
                        tabPanel("Total scores",
                                 h3("Analysis of total scores"),
                                 h4("Summary table"),
                                 tableOutput('results'),
                                 h4("Histogram of total score"),
                                 sliderInput("inSlider2", "Cut-Score", min = 0, max = 10,
                                             value = 1, step = 1),
                                 p('For selected cut-score, blue part of histogram shows students with total score
                                   above the cut-score, grey column shows students with Total Score equal
                                   to cut-score and red part of histogram shows students below the cut-score.'),
                                 plotOutput('histogram_totalscores'),
                                 downloadButton("DP_histogram_totalscores", label = "Download figure"),
                                 br(),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data  <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     br(),
                                     code('score <- apply(data, 1, sum) # Total score'),
                                     br(),
                                     br(),
                                     code('# Summary of total score'),
                                     br(),
                                     code('summary(score)'),
                                     br(),
                                     code('# Histogram'),
                                     br(),
                                     code('hist(score, breaks = 0:ncol(data)) ')),
                                 br()
                                 ),
                        # * STANDARD SCORES ####
                        tabPanel("Standard scores",
                                 h3('Standard scores'),
                                 p(strong('Total Score'), 'also known as raw score is a total number of correct
                                   answers. It can be used to compare individual score to a norm group, e.g. if the mean
                                   is 12, then individual score can be compared to see if it is below or above this average. ', br(),
                                   strong('Percentile'), 'indicates the value below which a percentage of observations
                                   falls, e.g. a individual score at the 80th percentile means that the individual score
                                   is the same or higher than the scores of 80% of all respondents. ', br(),
                                   strong('Success Rate'), 'is the percentage of success, e.g. if the maximum points of test
                                   is equal to 20 and individual score is 12 then success rate is 12/20 = 0.6, i.e. 60%.', br(),
                                   strong('Z-score'), 'or also standardized score is a linear transformation of total
                                   score with a mean of 0 and with variance of 1. If X is total score, M its mean and SD its
                                   standard deviation then Z-score = (X - M) / SD. ', br(),
                                   strong('T-score'), 'is transformed Z-score with a mean of 50 and standard deviation
                                   of 10. If Z is Z-score then T-score = (Z * 10) + 50. '),
                                 h4("Table by score"),
                                 tableOutput('percentile'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data  <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     br(),
                                     code('score <- apply(data, 1, sum) # Total score'),
                                     br(),
                                     code('tosc <- sort(unique(score)) # Levels of total score'),
                                     br(),
                                     code('perc <- cumsum(prop.table(table(score))) # Percentiles'),
                                     br(),
                                     code('sura <- 100 * (tosc / max(score)) # Success rate'),
                                     br(),
                                     code('zsco <- sort(unique(scale(score))) # Z-score'),
                                     br(),
                                     code('tsco <- 50 + 10 * zsco # T-score')),
                                 br()
                        )
                        ),
             ############################
             # CORRELATION STRUCTURE ####
             ############################
             tabPanel("Correlation structure",
                      h3("Correlation structure"),
                      h4("Polychoric correlation heat map"),
                      p('Polychoric correlation heat map is a correlation plot which displays a polychoric
                                     correlations of items. The size and shade of circles indicate how much the
                                     items are correlated (larger and darker
                                     circle means larger correlation). The color of circles indicates in which way the
                                     items are correlated - blue color shows possitive correlation and red color shows
                                     negative correlation.'),
                      plotOutput('corr_plot'),
                      downloadButton("DP_corr_plot", label = "Download figure"),
                      br(),
                      h4("Scree plot"),
                      p('A scree plot displays the eigenvalues associated with an component or a factor in descending order
                                    versus the number of the component or factor. '),
                                 plotOutput('scree_plot'),
                      downloadButton("DP_scree_plot", label = "Download figure"),
                      h4("Selected R code"),
                      div(code('library(corrplot)'),
                          br(),
                          code('library(difNLR)'),
                          br(),
                          code('library(psych)'),
                          br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data  <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     br(),
                                     code('# Correlation plot'),
                                     br(),
                                     code('corP <- polychoric(data)'),
                                     br(),
                                     code('corrplot(corP$rho)'),
                                     br(),
                                     code('corP$rho # Correlation matrix'),
                                     br(),
                                     br(),
                                     code('# Scree plot'),
                                     br(),
                                     code('plot(1:length(eigen(corP$rho)$values), eigen(corP$rho)$values,
                                          ylab = "Eigen value", xlab = "Component Number")'),
                                     br(),
                                     code('lines(1:length(eigen(corP$rho)$values), eigen(corP$rho)$values)'),
                                     br(),
                                     code('eigen(corP$rho) # Eigen values and vectors')),
                                 br()
                        ),
             ###########################
             # TRADITIONAL ANALYSIS ####
             ###########################
             navbarMenu('Traditional analysis',
                        # * ITEM ANALYSIS ####
                        tabPanel("Item analysis",
                                 h3("Traditional item analysis"),
                                 p('Traditional item analysis uses proportions of correct answers or correlations to estimate item properties.'),
                                 h4("Item difficulty/discrimination graph"),
                                 p("Displayed is difficulty (red) and discrimination (blue)
                                   for all items. Items are ordered by difficulty. ", br(),
                                   strong("Difficulty"),' of items is estimated as percent of students who
                                   answered correctly to that item.', br(),
                                   strong("Discrimination"),' is described by difference of percent correct
                                   in upper and lower third of students (Upper-Lower Index, ULI). By rule of
                                   thumb it should not be lower than 0.2 (borderline in the plot), except for
                                   very easy or very difficult items.'),
                                 plotOutput('difplot'),
                                 downloadButton("DP_difplot", label = "Download figure"),
                                 h4("Cronbach's alpha"),
                                 p("Chronbach's alpha is an estimate of the reliability of a psychometric test. It is a function
                                   of the number of items in a test, the average covariance between item-pairs, and the variance
                                   of the total score (Cronbach, 1951)."),
                                 tableOutput('cronbachalpha'),
                                 h4("Traditional item analysis table"),
                                 p(strong('Explanation: Difficulty'), ' - Difficulty of item is estimated as percent
                                   of students who answered correctly to that item. ', strong('SD'),' - standard deviation, ',
                                   strong('RIT'), ' - Pearson correlation between item and Total score, ', strong('RIR'),'
                                   - Pearson correlation between item and rest of items, ', strong('ULI'),'
                                   - Upper-Lower Index, ', strong('Alpha Drop'),' - Cronbach\'s alpha of test without given item.'),
                                 tableOutput('itemexam'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(psych)'),
                                     br(),
                                     code('library(psychometric)'),
                                     br(),
                                     code('library(ShinyItemAnalysis)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data  <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     br(),
                                     code('# Difficulty and discrimination plot'),
                                     br(),
                                     code('DDplot(data)'),
                                     br(),
                                     br(),
                                     code('# Cronbach alpha'),
                                     br(),
                                     code('psych::alpha(data)'),
                                     br(),
                                     br(),
                                     code('# Table'),
                                     br(),
                                     code('tab <- round(data.frame(item.exam(data, discr = TRUE)[, c(4, 1, 5, 2, 3)],
                                          psych::alpha(data)$alpha.drop[, 1]), 2)'),
                                     br(),
                                     code('colnames(tab) <- c("Difficulty", "SD", "Dsicrimination ULI", "Discrimination RIT", "Discrimination RIR", "Alpha Drop")'),
                                     br(),
                                     code('tab')),
                                 br()
                                 ),
                        # * DISTRACTORS ####
                        tabPanel("Distractors",
                                 h3("Distractor analysis"),
                                 p('In distractor analysis, we are interested in how test takers select
                                   the correct answer and how the distractors (wrong answers) were able
                                   to function effectively by drawing the test takers away from the correct answer.'),
                                 sliderInput('gr','Number of groups:',
                                             min   = 1,
                                             max   = 5,
                                             value = 3
                                 ),
                                 htmlOutput("text_distractor"),
                                 h4("Distractors plot"),
                                 radioButtons('type_combinations_distractor', 'Type',
                                              list("Combinations", "Distractors")
                                 ),
                                 sliderInput("distractorSlider", "Item Slider", min=1, value=1, max=10,
                                             step=1, animate=TRUE),
                                 plotOutput('graf'),
                                 downloadButton("DP_graf", label = "Download figure"),
                                 br(),
                                 h4("Table with counts"),
                                 fluidRow(column(12, align = "center", tableOutput('tab_counts_distractor'))),
                                 h4("Table with proportions"),
                                 fluidRow(column(12, align = "center", tableOutput('tab_props_distractor'))),
                                 br(),
                                 h4('Histogram of total scores'),
                                 plotOutput('hist_distractor_by_group'),
                                 downloadButton("DP_hist_distractor_by_group", label = "Download figure"),
                                 br(),
                                 h4('Table of total scores by groups'),
                                 fluidRow(column(12, align = "center", tableOutput('tab_distractor_by_group'))),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(ShinyItemAnalysis)'),
                                     br(),
                                     code('data(GMATtest)'),
                                     br(),
                                     code('data  <- GMATtest[, colnames(GMATtest) != "group"]'),
                                     br(),
                                     code('data(GMATkey)'),
                                     br(),
                                     code('key  <- GMATkey'),
                                     br(),
                                     br(),
                                     code('# Combinations - plot for item 1 and 3 groups'),
                                     br(),
                                     code('plotDistractorAnalysis(data, key, num.group = 3, item = 1,
                                          multiple.answers = T)'),
                                     br(),
                                     code('# Distractors - plot for item 1 and 3 groups'),
                                     br(),
                                     code('plotDistractorAnalysis(data, key, num.group = 3, item = 1,
                                          multiple.answers = F)'),
                                     br(),
                                     code('# Table with counts and margins - item 1 and 3 groups'),
                                     br(),
                                     code('DA <- DistractorAnalysis(data, key, num.groups = 3)[[1]]'),
                                     br(),
                                     code('dcast(as.data.frame(DA), response ~ score.level, sum, margins = T, value.var = "Freq")'),
                                     br(),
                                     code('# Table with proportions - item 1 and 3 groups'),
                                     br(),
                                     code('DistractorAnalysis(data, key, num.groups = 3, p.table = T)[[1]]'),
                                     br(),
                                     code('tab')),
                                 br()
                                 )
                        ),

             #####################
             # REGRESSION ########
             #####################
             navbarMenu("Regression",
                        # * LOGISTIC ####
                        tabPanel("Logistic",
                                 h3("Logistic regression on total scores"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Logistic regression'),'can model dependency of probability of correct answer on total score by
                                   s-shaped logistic curve. Parameter', strong( "b0"),' describes horizontal position of the fitted curve,
                                   parameter ', strong( 'b1'),' describes its slope.'),
                                 br(),
                                 h4("Plot with estimated logistic curve"),
                                 p('Points represent proportion of correct answer with respect to total score.
                                   Their size is determined by count of respondents who answered item correctly.'),
                                 sliderInput("logregSlider", "Item Slider", min=1, value=1, max=10,
                                             step=1, animate=TRUE),
                                 plotOutput('logreg'),
                                 downloadButton("DP_logref", label = "Download figure"),
                                 h4("Equation"),
                                 withMathJax(),
                                 ('$$\\mathrm{P}(Y = 1|X, b_0, b_1) = \\mathrm{E}(Y|X, b_0, b_1) = \\frac{e^{\\left( b_{0} + b_1 X\\right)}}{1+e^{\\left( b_{0} + b_1 X\\right) }} $$'),

                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('logregtab'))),
                                 htmlOutput("logisticint"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data  <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     code('score  <- apply(data, 1, sum)'),
                                     br(),
                                     br(),
                                     code('# Logistic model for item 1'),
                                     br(),
                                     code('fit <- glm(data[, 1] ~ score, family = binomial)'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)'),
                                     br(),
                                     code('# Function for plot'),
                                     br(),
                                     code('fun <- function(x, b0, b1){exp(b0 + b1 * x) / (1 + exp(b0 + b1 * x))}'),
                                     br(),
                                     code('# Plot of estimated curve'),
                                     br(),
                                     code('curve(fun(x, b0 = coef(fit)[1], b1 = coef(fit)[2]), 0, 20,
                                          xlab = "Total score",
                                          ylab = "Probability of correct answer",
                                          ylim = c(0, 1))')),
                                 br()
                                 ),
                        # * LOGISTIC Z ####
                        tabPanel("Logistic Z",
                                 h3("Logistic regression on standardized total scores"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Logistic regression'), 'can model dependency of probability of correct answer on
                                   standardized total score (Z-score) by s-shaped logistic curve. Parameter ', strong( 'b0'), ' describes
                                   horizontal position of the fitted curve (difficulty), parameter ', strong('b1'),' describes its slope at
                                   inflection point (discrimination). '),
                                 br(),
                                 h4("Plot with estimated logistic curve"),
                                 p('Points represent proportion of correct answer with respect to standardized
                                   total score. Their size is determined by count of respondents who answered item correctly.'),
                                 sliderInput("zlogregSlider", "Item Slider", min=1, value=1, max=10,
                                             step=1, animate=TRUE),
                                 plotOutput('zlogreg'),
                                 downloadButton("DP_zlogreg", label = "Download figure"),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}(Y = 1|Z, b_0, b_1) = \\mathrm{E}(Y|Z, b_0, b_1) = \\frac{e^{\\left( b_{0} + b_1 Z\\right) }}{1+e^{\\left( b_{0} + b_1 Z\\right) }} $$'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('zlogregtab'))),
                                 htmlOutput("zlogisticint"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data  <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     code('stand.score  <- scale(apply(data, 1, sum))'),
                                     br(),
                                     br(),
                                     code('# Logistic model for item 1'),
                                     br(),
                                     code('fit <- glm(data[, 1] ~ stand.score, family = binomial)'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)'),
                                     br(),
                                     code('# Function for plot'),
                                     br(),
                                     code('fun <- function(x, b0, b1){exp(b0 + b1 * x) / (1 + exp(b0 + b1 * x))}'),
                                     br(),
                                     code('# Plot of estimated curve'),
                                     br(),
                                     code('curve(fun(x, b0 = coef(fit)[1], b1 = coef(fit)[2]), -3, 3,
                                          xlab = "Standardized total score",
                                          ylab = "Probability of correct answer",
                                          ylim = c(0, 1))')),
                                 br()
                                 ),
                        # * LOGISTIC IRT Z ####
                        tabPanel("Logistic IRT Z",
                                 h3("Logistic regression on standardized total scores with IRT parameterization"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Logistic regression'), 'can model dependency of probability of correct answer on
                                   standardized total score (Z-score) by s-shaped logistic curve. Note change in parametrization - the IRT parametrization
                                   used here corresponds to the parametrization used in IRT models.
                                   Parameter', strong('b') , 'describes horizontal position of the fitted curve (difficulty),
                                   parameter' , strong('a') , ' describes its slope at inflection point (discrimination). '),
                                 br(),
                                 h4("Plot with estimated logistic curve"),
                                 p('Points represent proportion of correct answer with respect to standardized
                                   total score. Their size is determined by count of respondents who answered item correctly.'),
                                 sliderInput("zlogreg_irtSlider", "Item Slider", min=1, value=1, max=10,
                                             step=1, animate=TRUE),
                                 plotOutput('zlogreg_irt'),
                                 downloadButton("DP_zlogreg_irt", label = "Download figure"),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}(Y = 1|Z, a, b) = \\mathrm{E}(Y|Z, a, b) = \\frac{e^{ a\\left(Z - b\\right) }}{1+e^{a\\left(Z - b\\right)}} $$'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('zlogregtab_irt'))),
                                 htmlOutput("zlogisticint_irt"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data  <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     code('stand.score  <- scale(apply(data, 1, sum))'),
                                     br(),
                                     br(),
                                     code('# Logistic model for item 1'),
                                     br(),
                                     code('fit <- glm(data[, 1] ~ stand.score, family = binomial)'),
                                     br(),
                                     code('# Coefficients - tranformation'),
                                     br(),
                                     code('coef <- c(a = coef(fit)[2], b = - coef(fit)[1] / coef(fit)[2])'),
                                     br(),
                                     code('coef'),
                                     br(),
                                     code('# Function for plot'),
                                     br(),
                                     code('fun <- function(x, a, b){exp(a * (x - b)) / (1 + exp(a * (x - b)))}'),
                                     br(),
                                     code('# Plot of estimated curve'),
                                     br(),
                                     code('curve(fun(x, a = coef[1], b = coef[2]), -3, 3,
                                          xlab = "Standardized total score",
                                          ylab = "Probability of correct answer",
                                          ylim = c(0, 1))')),
                                 br()
                                 ),

                        # * NONLINEAR Z ####
                        tabPanel("Nonlinear IRT Z",
                                 h3("Nonlinear regression on standardized total scores"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Nonlinear regression'), 'can model dependency of probability of correct answer on
                                   standardized total score (Z-score) by s-shaped logistic curve. The IRT parametrization used here corresponds
                                   to the parametrization used in IRT models. Parameter ', strong( 'b'),' describes horizontal position of the fitted curve (difficulty),
                                   parameter ',strong( 'a'), ' describes its slope at inflection point (discrimination). This model allows for nonzero lower left asymptote ',strong( 'c'),'
                                   (pseudo-guessing). '),
                                 br(),
                                 h4("Plot with estimated nonlinear curve"),
                                 p('Points represent proportion of correct answer with respect to standardized
                                   total score. Their size is determined by count of respondents who answered item correctly.'),
                                 sliderInput("nlsSlider", "Item Slider", min=1, value=1, max=10,
                                             step=1, animate=TRUE),
                                 plotOutput('nlsplot'),
                                 downloadButton("DP_nlsplot", label = "Download figure"),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}(Y = 1|Z, b_0, b_1, c) = \\mathrm{E}(Y|Z, b_0, b_1, c) = c + \\left( 1-c \\right) \\cdot \\frac{e^{a\\left(Z-b\\right) }}{1+e^{a\\left(Z-b\\right) }} $$'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('nonlinearztab'))),
                                 htmlOutput("nonlinearint"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('Data  <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     code('stand.score  <- scale(apply(Data, 1, sum))'),
                                     br(),
                                     br(),
                                     code('# NLR model for item 1'),
                                     br(),
                                     code('fun <- function(x, a, b, c){c + (1 - c) * exp(a * (x - b)) / (1 + exp(a * (x - b)))}'),
                                     br(),
                                     code('fit <- nls(data[, 1] ~ fun(stand.score, a, b, c), algorithm = "port",
                                          start = startNLR(data, GMAT[, "group"], model = "3PLcg")[1, 1:3])'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)'),
                                     br(),
                                     code('# Plot of estimated curve'),
                                     br(),
                                     code('curve(fun(x, a = coef(fit)[1], b = coef(fit)[2], c = coef(fit)[3]), -3, 3,
                                          xlab = "Standardized total score",
                                          ylab = "Probability of correct answer",
                                          ylim = c(0, 1))')),
                                 br()
                                 ),
                        "----",
                        # * MODELS COMPARISON ####
                        tabPanel("Model comparison",
                                 h3("Logistic regression model selection"),
                                 p('Here you can compare classic 2PL logistic regression model to non-linear model
                                   item by item using some information criterions: '),
                                 tags$ul(
                                   tags$li(strong('AIC'), 'is the Akaike information criterion (Akaike, 1974), '),
                                   tags$li(strong('BIC'), 'is the Bayesian information criterion (Schwarz, 1978)')
                                 ),
                                 p('Another approach to nested models can be likelihood ratio chi-squared test.
                                   Significance level is set to 0.05. As tests are performed item by item, it is
                                   possible to use multiple comparison correction method. '),
                                 selectInput("correction_method_regrmodels", "Correction method",
                                             c("BH" = "BH",
                                               "Holm" = "holm",
                                               "Hochberg" = "hochberg",
                                               "Hommel" = "hommel",
                                               "BY" = "BY",
                                               "FDR" = "fdr",
                                               "none" = "none"),
                                             selected="BH"),
                                 h4("Table of comparison statistics"),
                                 p('Rows ', strong('BEST'), 'indicate which model has the lowest value of criterion, or is the largest
                                   significant model by likelihood ratio test.'),
                                 DT::dataTableOutput('regr_comp_table'),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('Data  <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     code('stand.score  <- scale(apply(Data, 1, sum))'),
                                     br(),
                                     br(),
                                     code('# Fitting models'),
                                     br(),
                                     code('fun <- function(x, a, b, c){c + (1 - c) * exp(a * (x - b)) / (1 + exp(a * (x - b)))}'),
                                     br(),
                                     code('# 2PL model for item 1'),
                                     br(),
                                     code('fit2PL <- nls(Data[, 1] ~ fun(stand.score, a, b, c = 0), algorithm = "port",
                                          start = startNLR(Data, GMAT[, "group"], model = "3PLcg")[1, 1:2])'),
                                     br(),
                                     code('# 3PL model for item 1'),
                                     br(),
                                     code('fit3PL <- nls(Data[, 1] ~ fun(stand.score, a, b, c), algorithm = "port",
                                          start = startNLR(Data, GMAT[, "group"], model = "3PLcg")[1, 1:3])'),
                                     br(),
                                     br(),
                                     code('# Comparison'),
                                     br(),
                                     code('AIC(fit2PL); AIC(fit3PL)'),
                                     br(),
                                     code('BIC(fit2PL); BIC(fit3PL)'),
                                     br(),
                                     code('LRstat <- -2 * (sapply(fit2PL, logLik) - sapply(fit3PL, logLik))'),
                                     br(),
                                     code('LRdf <- 1'),
                                     br(),
                                     code('LRpval <- 1 - pchisq(LRstat, LRdf)'),
                                     br(),
                                     code('LRpval <- p.adjust(LRpval, method = "BH")'),
                                     br()),
                                 br()
                                 ),
                        "----",
                        # * MULTINOMIAL ####
                        tabPanel("Multinomial",
                                 h3("Multinomial regression on standardized total scores"),
                                 p('Various regression models may be fitted to describe
                                   item properties in more detail.',
                                   strong('Multinomial regression'),'allows for simultaneous modelling of probability of choosing
                                   given distractors on standardized total score (Z-score).'),
                                 br(),
                                 h4("Plot with estimated curves of multinomial regression"),
                                 p('Points represent proportion of selected option with respect to standardized
                                   total score. Their size is determined by count of respondents who selected given option.'),
                                 sliderInput("multiSlider", "Item Slider", min=1, value=1, max=10,
                                             step=1, animate=TRUE),
                                 plotOutput('multiplot'),
                                 downloadButton("DP_multiplot", label = "Download figure"),
                                 h4("Equation"),
                                 uiOutput('multieq'),
                                 h4("Table of parameters"),
                                 fluidRow(column(12, align = "center", tableOutput('multitab'))),
                                 strong("Interpretation:"),
                                 htmlOutput("multiint"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(nnet)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data.scored  <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     code('stand.score  <- scale(apply(data, 1, sum))'),
                                     br(),
                                     code('data(GMATtest)'),
                                     br(),
                                     code('data  <- GMATtest[, colnames(GMATtest) != "group"]'),
                                     br(),
                                     code('data(GMATkey)'),
                                     br(),
                                     code('key  <- GMATkey'),

                                     br(),
                                     br(),
                                     code('# multinomial model for item 1'),
                                     br(),
                                     code('fit <- multinom(relevel(data[, 1], ref = paste(key[1])) ~ stand.score)'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)')),
                                 br()
                                 )
                        ),
             ###################
             # IRT MODELS ######
             ###################
             navbarMenu("IRT models",
                        # * 1PL (RASCH) ####
                        tabPanel("1PL (Rasch)",
                                 h3("One parameter Item Response Theory model"),
                                 p('Item Response Theory (IRT) models are mixed-effect regression models in which
                                   student ability (theta) is assumed to be a random effect and is estimated together with item
                                   paramters. Ability (theta) is often assumed to follow normal distibution.'),
                                 p('In',
                                   strong('1PL IRT model,'), 'all items are assumed to have the same slope in inflection point – the
                                   same discrimination', strong('a.'), 'Items can differ in location of their inflection point – in item difficulty',
                                   strong('b.'), 'More restricted version of this model, the
                                   ',strong('Rasch model,'),'assumes discrimination', strong('a'), 'is equal to 1.'),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}\\left(Y_{ij} = 1\\vert \\theta_{i}, a, b_{j} \\right) =  \\frac{e^{a\\left(\\theta_{i}-b_{j}\\right) }}{1+e^{a\\left(\\theta_{i}-b_{j}\\right) }} $$'),
                                 h4("Item characteristic curves"),
                                 plotOutput('rasch'),
                                 downloadButton("DP_rasch", label = "Download figure"),
                                 h4("Item information curves"),
                                 plotOutput('raschiic'),
                                 downloadButton("DP_raschiic", label = "Download figure"),
                                 h4("Test information function"),
                                 plotOutput('raschtif'),
                                 downloadButton("DP_raschtif", label = "Download figure"),
                                 h4("Table of parameters"),
                                 tableOutput('raschcoef'),
                                 h4('Scatter plot of factor scores and standardized total scores'),
                                 plotOutput('raschFactor'),
                                 downloadButton("DP_raschFactor", label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(ltm)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- rasch(data)'),
                                     br(),
                                     code('# for Rasch model use'),
                                     br(),
                                     code('# fit <- rasch(data, constraint = cbind(ncol(data) + 1, 1))'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "IIC")'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, items = 0, type = "IIC")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('df1  <- ltm::factor.scores(fit, return.MIvalues = T)$score.dat'),
                                     br(),
                                     code('FS   <- as.vector(df1[, "z1"])'),
                                     br(),
                                     code('df2  <- df1'),
                                     br(),
                                     code('df2$Obs <- df2$Exp <- df2$z1 <- df2$se.z1 <- NULL'),
                                     br(),
                                     code('STS <- as.vector(scale(apply(df2, 1, sum)))'),
                                     br(),
                                     code('df  <- data.frame(FS, STS)'),
                                     br(),
                                     code('plot(FS ~ STS, data = df,
                                          xlab = "Standardized total score",
                                          ylab = "Factor score")')),
                                 br()
                                 ),
                        # * 2PL ####
                        tabPanel("2PL ",
                                 h3("Two parameter Item Response Theory model"),
                                 p('Item Response Theory (IRT) models are mixed-effect regression models in which
                                   student ability (theta) is assumed to be a random effect and is estimated together with item
                                   paramters. Ability (theta) is often assumed to follow normal distibution.'),
                                 p(strong('2PL IRT model,'), 'allows for different slopes in inflection point – different
                                   discriminations', strong('a.'), 'Items can also differ in location of their inflection point – in item difficulty',
                                   strong('b.')),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}\\left(Y_{ij} = 1\\vert \\theta_{i}, a_{j}, b_{j}\\right) =  \\frac{e^{a_{j}\\left(\\theta_{i}-b_{j}\\right) }}{1+e^{a_{j}\\left(\\theta_{i}-b_{j}\\right) }} $$'),

                                 h4("Item characteristic curves"),
                                 plotOutput('twoparam'),
                                 downloadButton("DP_twoparam", label = "Download figure"),
                                 h4("Item information curves"),
                                 plotOutput("twoparamiic"),
                                 downloadButton("DP_twoparamiic", label = "Download figure"),
                                 h4("Test information function"),
                                 plotOutput("twoparamtif"),
                                 downloadButton("DP_twoparamtif", label = "Download figure"),
                                 h4("Table of parameters"),
                                 tableOutput('twoparamcoef'),
                                 h4('Scatter plot of factor scores and standardized total scores'),
                                 plotOutput('twoFactor'),
                                 downloadButton("DP_twoFactor", label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(ltm)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- ltm(data ~ z1, IRT.param = TRUE)'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "IIC")'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, items = 0, type = "IIC")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('df1  <- ltm::factor.scores(fit, return.MIvalues = T)$score.dat'),
                                     br(),
                                     code('FS   <- as.vector(df1[, "z1"])'),
                                     br(),
                                     code('df2  <- df1'),
                                     br(),
                                     code('df2$Obs <- df2$Exp <- df2$z1 <- df2$se.z1 <- NULL'),
                                     br(),
                                     code('STS <- as.vector(scale(apply(df2, 1, sum)))'),
                                     br(),
                                     code('df  <- data.frame(FS, STS)'),
                                     br(),
                                     code('plot(FS ~ STS, data = df,
                                          xlab = "Standardized total score",
                                          ylab = "Factor score")')),
                                 br()
                                 ),
                        # * 3PL ####
                        tabPanel("3PL ",
                                 h3("Three parameter Item Response Theory model"),
                                 p('Item Response Theory (IRT) models are mixed-effect regression models in which
                                   student ability (theta) is assumed to be a random effect and is estimated together with item
                                   paramters. Ability (theta) is often assumed to follow normal distibution.'),
                                 p(strong('3PL IRT model,'), 'allows for different discriminations of items', strong('a,'),
                                   'different item difficulties',
                                   strong('b,'), 'and allows also for nonzero left asymptote – pseudo-guessing', strong('c.')),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}\\left(Y_{ij} = 1\\vert \\theta_{i}, a_{j}, b_{j}, c_{j} \\right) = c_{j} + \\left(1 - c_{j}\\right) \\cdot \\frac{e^{a_{j}\\left(\\theta_{i}-b_{j}\\right) }}{1+e^{a_{j}\\left(\\theta_{i}-b_{j}\\right) }} $$'),

                                 h4("Item characterisic curves"),
                                 plotOutput('threeparam'),
                                 downloadButton("DP_threeparam", label = "Download figure"),
                                 h4("Item information curves"),
                                 plotOutput("threeparamiic"),
                                 downloadButton("DP_threeparamiic", label = "Download figure"),
                                 h4("Test information function"),
                                 plotOutput("threeparamtif"),
                                 downloadButton("DP_threeparamtif", label = "Download figure"),
                                 h4("Table of parameters"),
                                 tableOutput("threeparamcoef"),
                                 h4('Scatter plot of factor scores and standardized total scores'),
                                 plotOutput('threeFactor'),
                                 downloadButton("DP_threeFactor", label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(ltm)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- tpm(data, IRT.param = TRUE)'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "IIC")'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, items = 0, type = "IIC")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('df1  <- ltm::factor.scores(fit, return.MIvalues = T)$score.dat'),
                                     br(),
                                     code('FS   <- as.vector(df1[, "z1"])'),
                                     br(),
                                     code('df2  <- df1'),
                                     br(),
                                     code('df2$Obs <- df2$Exp <- df2$z1 <- df2$se.z1 <- NULL'),
                                     br(),
                                     code('STS <- as.vector(scale(apply(df2, 1, sum)))'),
                                     br(),
                                     code('df  <- data.frame(FS, STS)'),
                                     br(),
                                     code('plot(FS ~ STS, data = df,
                                          xlab = "Standardized total score",
                                          ylab = "Factor score")')),
                                 br()
                                 )
                                 ),
             #############################
             # IRT MODELS WITH MIRT ######
             #############################
             navbarMenu("IRT models with mirt",
                        # * RASCH ####
                        tabPanel("Rasch",
                                 h3("Rasch Item Response Theory model"),
                                 p('Item Response Theory (IRT) models are mixed-effect regression models in which
                                   student ability (theta) is assumed to be a random effect and is estimated together with item
                                   paramters. Ability (theta) is often assumed to follow normal distibution.'),
                                 p('In',
                                   strong('Rasch IRT model,'), '(Rasch, 1960) all items are assumed to have the same slope in inflection point – the
                                   same discrimination', strong('a'), 'which is fixed to value of 1. Items can differ in location of their inflection point – in item difficulty',
                                   strong('b.')),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}\\left(Y_{ij} = 1\\vert \\theta_{i}, b_{j} \\right) =  \\frac{e^{\\left(\\theta_{i}-b_{j}\\right) }}{1+e^{\\left(\\theta_{i}-b_{j}\\right) }} $$'),
                                 h4("Item characteristic curves"),
                                 plotOutput('rasch_mirt'),
                                 downloadButton("DP_rasch_mirt", label = "Download figure"),
                                 h4("Item information curves"),
                                 plotOutput('raschiic_mirt'),
                                 downloadButton("DP_raschiic_mirt", label = "Download figure"),
                                 h4("Test information function"),
                                 plotOutput('raschtif_mirt'),
                                 downloadButton("DP_raschtif_mirt", label = "Download figure"),
                                 h4("Table of parameters with item fit statistics"),
                                 p('Estimates of parameters are completed by SX2 item fit statistics (Ames & Penfield, 2015)'),
                                 tableOutput('raschcoef_mirt'),
                                 h4('Scatter plot of factor scores and standardized total scores'),
                                 textOutput('raschFactorCor_mirt'),
                                 plotOutput('raschFactor_mirt'),
                                 downloadButton("DP_raschFactor_mirt", label = "Download figure"),
                                 h4('Wright map'),
                                 p('Wright map (Wilson, 2005; Wright & Stone, 1979), also called item-person map, is a graphical tool
                                   to display person estimates and item parameters. The person side
                                   (left) represents histogram of estimated knowledge of students.
                                   The item side (right) displays estimates of difficulty of particular items. '),
                                 plotOutput('raschWrightMap_mirt'),
                                 downloadButton('DP_raschWM_mirt', label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(mirt)'),
                                     br(),
                                     code('library(WrightMap)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- mirt(data, model = 1, itemtype = "Rasch", SE = T)'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit, type = "trace", facet_items = F)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "infotrace", facet_items = F)'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, type = "infoSE")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit, simplify = TRUE)'),
                                     br(),
                                     code('coef(fit, IRTpars = TRUE, simplify = TRUE)'),
                                     br(),
                                     code('# Item fit statistics'),
                                     br(),
                                     code('itemfit(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('fs <- as.vector(fscores(fit))'),
                                     br(),
                                     code('sts <- as.vector(scale(apply(data, 1, sum)))'),
                                     br(),
                                     code('plot(fs ~ sts)'),
                                     br(),
                                     br(),
                                     code('# Wright Map'),
                                     br(),
                                     code('b <- sapply(1:ncol(data), function(i) coef(fit)[[i]][, "d"])'),
                                     br(),
                                     code('wrightMap(fs, b, item.side = itemClassic)')),
                                 br()
                                 ),
                        # * 1 PL IRT ####
                        tabPanel("1PL",
                                 h3("One parameter Item Response Theory model"),
                                 p('Item Response Theory (IRT) models are mixed-effect regression models in which
                                   student ability (theta) is assumed to be a random effect and is estimated together with item
                                   paramters. Ability (theta) is often assumed to follow normal distibution.'),
                                 p('In',
                                   strong('1PL IRT model,'), 'all items are assumed to have the same slope in inflection point – the
                                   same discrimination', strong('a'), '. Items can differ in location of their inflection point – in item difficulty',
                                   strong('b.')),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}\\left(Y_{ij} = 1\\vert \\theta_{i}, a, b_{j} \\right) =  \\frac{e^{a\\left(\\theta_{i}-b_{j}\\right) }}{1+e^{a\\left(\\theta_{i}-b_{j}\\right) }} $$'),
                                 h4("Item characteristic curves"),
                                 plotOutput('oneparamirt_mirt'),
                                 downloadButton("DP_oneparamirt_mirt", label = "Download figure"),
                                 h4("Item information curves"),
                                 plotOutput('oneparamirtiic_mirt'),
                                 downloadButton("DP_oneparamirtiic_mirt", label = "Download figure"),
                                 h4("Test information function"),
                                 plotOutput('oneparamirttif_mirt'),
                                 downloadButton("DP_oneparamirttif_mirt", label = "Download figure"),
                                 h4("Table of parameters with item fit statistics"),
                                 p('Estimates of parameters are completed by SX2 item fit statistics
                                   (Ames & Penfield, 2015)'),
                                 tableOutput('oneparamirtcoef_mirt'),
                                 h4('Scatter plot of factor scores and standardized total scores'),
                                 textOutput('oneparamirtFactorCor_mirt'),
                                 plotOutput('oneparamirtFactor_mirt'),
                                 downloadButton("DP_oneparamirtFactor_mirt", label = "Download figure"),
                                 h4('Wright map'),
                                 p('Wright map (Wilson, 2005; Wright & Stone, 1979), also called item-person map, is a graphical tool
                                   to display person estimates and item parameters. The person side
                                   (left) represents histogram of estimated knowledge of students.
                                   The item side (right) displays estimates of difficulty of particular items. '),
                                 plotOutput('oneparamirtWrightMap_mirt'),
                                 downloadButton('DP_oneparamirtWM_mirt', label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(mirt)'),
                                     br(),
                                     code('library(WrightMap)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- mirt(data, model = 1, itemtype = "2PL", constrain = list((1:ncol(data)) + seq(0, (ncol(data) - 1)*3, 3)), SE = T)'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit, type = "trace", facet_items = F)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "infotrace", facet_items = F)'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, type = "infoSE")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit, simplify = TRUE)'),
                                     br(),
                                     code('coef(fit, IRTpars = TRUE, simplify = TRUE)'),
                                     br(),
                                     code('# Item fit statistics'),
                                     br(),
                                     code('itemfit(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('fs <- as.vector(fscores(fit))'),
                                     br(),
                                     code('sts <- as.vector(scale(apply(data, 1, sum)))'),
                                     br(),
                                     code('plot(fs ~ sts)'),
                                     br(),
                                     br(),
                                     code('# Wright Map'),
                                     br(),
                                     code('b <- sapply(1:ncol(data), function(i) coef(fit)[[i]][, "d"])'),
                                     br(),
                                     code('wrightMap(fs, b, item.side = itemClassic)')),
                                 br()
                                 ),
                        # * 2PL ####
                        tabPanel("2PL ",
                                 h3("Two parameter Item Response Theory model"),
                                 p('Item Response Theory (IRT) models are mixed-effect regression models in which
                                   student ability (theta) is assumed to be a random effect and is estimated together with item
                                   paramters. Ability (theta) is often assumed to follow normal distibution.'),
                                 p(strong('2PL IRT model,'), 'allows for different slopes in inflection point – different
                                   discriminations', strong('a.'), 'Items can also differ in location of their inflection point – in item difficulty',
                                   strong('b.')),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}\\left(Y_{ij} = 1\\vert \\theta_{i}, a_{j}, b_{j}\\right) =  \\frac{e^{a_{j}\\left(\\theta_{i}-b_{j}\\right) }}{1+e^{a_{j}\\left(\\theta_{i}-b_{j}\\right) }} $$'),
                                 h4("Item characteristic curves"),
                                 plotOutput('twoparamirt_mirt'),
                                 downloadButton("DP_twoparamirt_mirt", label = "Download figure"),
                                 h4("Item information curves"),
                                 plotOutput('twoparamirtiic_mirt'),
                                 downloadButton("DP_twoparamirtiic_mirt", label = "Download figure"),
                                 h4("Test information function"),
                                 plotOutput('twoparamirttif_mirt'),
                                 downloadButton("DP_twoparamirttif_mirt", label = "Download figure"),
                                 h4("Table of parameters with item fit statistics"),
                                 p('Estimates of parameters are completed by SX2 item fit statistics (Ames & Penfield, 2015)'),
                                 tableOutput('twoparamirtcoef_mirt'),
                                 h4('Scatter plot of factor scores and standardized total scores'),
                                 textOutput('twoparamirtFactorCor_mirt'),
                                 plotOutput('twoparamirtFactor_mirt'),
                                 downloadButton("DP_twoparamirtFactor_mirt", label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(mirt)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- mirt(data, model = 1, itemtype = "2PL", SE = T)'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit, type = "trace", facet_items = F)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "infotrace", facet_items = F)'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, type = "infoSE")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit, simplify = TRUE)'),
                                     br(),
                                     code('coef(fit, IRTpars = TRUE, simplify = TRUE)'),
                                     br(),
                                     code('# Item fit statistics'),
                                     br(),
                                     code('itemfit(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('fs <- as.vector(fscores(fit))'),
                                     br(),
                                     code('sts <- as.vector(scale(apply(data, 1, sum)))'),
                                     br(),
                                     code('plot(fs ~ sts)')),
                                 br()
                                 ),
                        # * 3PL ####
                        tabPanel("3PL ",
                                 h3("Three parameter Item Response Theory model"),
                                 p('Item Response Theory (IRT) models are mixed-effect regression models in which
                                   student ability (theta) is assumed to be a random effect and is estimated together with item
                                   paramters. Ability (theta) is often assumed to follow normal distibution.'),
                                 p(strong('3PL IRT model,'), 'allows for different discriminations of items', strong('a,'),
                                   'different item difficulties',
                                   strong('b,'), 'and allows also for nonzero left asymptote – pseudo-guessing', strong('c.')),
                                 h4("Equation"),
                                 ('$$\\mathrm{P}\\left(Y_{ij} = 1\\vert \\theta_{i}, a_{j}, b_{j}, c_{j} \\right) = c_{j} + \\left(1 - c_{j}\\right) \\cdot \\frac{e^{a_{j}\\left(\\theta_{i}-b_{j}\\right) }}{1+e^{a_{j}\\left(\\theta_{i}-b_{j}\\right) }} $$'),
                                 h4("Item characteristic curves"),
                                 plotOutput('threeparamirt_mirt'),
                                 downloadButton("DP_threeparamirt_mirt", label = "Download figure"),
                                 h4("Item information curves"),
                                 plotOutput('threeparamirtiic_mirt'),
                                 downloadButton("DP_threeparamirtiic_mirt", label = "Download figure"),
                                 h4("Test information function"),
                                 plotOutput('threeparamirttif_mirt'),
                                 downloadButton("DP_threeparamirttif_mirt", label = "Download figure"),
                                 h4("Table of parameters with item fit statistics"),
                                 p('Estimates of parameters are completed by SX2 item fit statistics (Ames & Penfield, 2015)'),
                                 tableOutput('threeparamirtcoef_mirt'),
                                 h4('Scatter plot of factor scores and standardized total scores'),
                                 textOutput('threeparamirtFactorCor_mirt'),
                                 plotOutput('threeparamirtFactor_mirt'),
                                 downloadButton("DP_threeparamirtFactor_mirt", label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(mirt)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- mirt(data, model = 1, itemtype = "3PL", SE = T)'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit, type = "trace", facet_items = F)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "infotrace", facet_items = F)'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, type = "infoSE")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit, simplify = TRUE)'),
                                     br(),
                                     code('coef(fit, IRTpars = TRUE, simplify = TRUE)'),
                                     br(),
                                     code('# Item fit statistics'),
                                     br(),
                                     code('itemfit(fit)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('fs <- as.vector(fscores(fit))'),
                                     br(),
                                     code('sts <- as.vector(scale(apply(data, 1, sum)))'),
                                     br(),
                                     code('plot(fs ~ sts)')),
                                 br()
                                 ),
                        "----",
                        # * MODEL COMPARISON ####
                        tabPanel("Model comparison ",
                                 h3("Item Response Theory model selection"),
                                 p('Item Response Theory (IRT) models are mixed-effect regression models in which
                                   student ability (theta) is assumed to be a random effect and is estimated together with item
                                   paramters. Ability (theta) is often assumed to follow normal distibution.'),
                                 p('IRT models can be compared by several information criterions: '),
                                 tags$ul(
                                   tags$li(strong('AIC'), 'is the Akaike information criterion (Akaike, 1974), '),
                                   tags$li(strong('AICc'), 'is AIC with a correction for finite sample size, '),
                                   tags$li(strong('SABIC'), 'is the Sample-sized adjusted BIC criterion, '),
                                   tags$li(strong('BIC'), 'is the Bayesian information criterion (Schwarz, 1978).')
                                 ),
                                 p('Another approach to compare IRT models can be likelihood ratio chi-squared test.
                                   Significance level is set to 0.05.'),
                                 h4("Table of comparison statistics"),
                                 p('Row ', strong('BEST'), 'indicates which model has the lowest value of criterion, or is the largest
                                   significant model by likelihood ratio test.'),
                                 tableOutput('irtcomparison'),
                                 tags$style(type = "text/css", "#irtcomparison tr:last-child {font-weight:bold;}"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(mirt)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     br(),
                                     code('# 1PL IRT model'),
                                     br(),
                                     code('fit1PL <- mirt(data, model = 1, itemtype = "3PL", SE = T)'),
                                     br(),
                                     code('# 2PL IRT model'),
                                     br(),
                                     code('fit2PL <- mirt(data, model = 1, itemtype = "2PL")'),
                                     br(),
                                     code('# 3PL IRT model'),
                                     br(),
                                     code('fit3PL <- mirt(data, model = 1, itemtype = "3PL")'),
                                     br(),
                                     br(),
                                     code('# Comparison'),
                                     br(),
                                     code('anova(fit1PL, fit2PL)'),
                                     br(),
                                     code('anova(fit2PL, fit3PL)')),
                                 br()
                                 ),
                        "----",
                        # * BOCK'S NOMINAL MODEL ####
                        tabPanel("Bock's nominal model",
                                 h3("Bock's nominal Item Response Theory model"),
                                 p('The nominal response model (NRM) was introduced by Bock (1972) as a way to model
                                   responses to items with two or more nominal categories. This model is suitable for
                                   multiple-choice items with no particular ordering of distractors. The correct answer
                                   represent the highest category, in terms of the measured latent trait. '),
                                 h4('Equation'),
                                 withMathJax('For ', strong('K'), ' possible test choices is the probability of the choice ', strong('k'), ' for
                                   person ', strong('i'), ' with latent trait', strong('\\(\\theta\\)'), ' in item ', strong('j'),
                                   'given by the following equation: '),
                                 ('$$\\mathrm{P}(Y_{ij} = k|\\theta_i, a_{j1}, al_{j(l-1)}, d_{j(l-1)}, l = 1, \\dots, K) =
                                             \\frac{e^{(ak_{j(k-1)} * a_{j1} * \\theta_i + d_{j(k-1)})}}{\\sum_l e^{(al_{j(l-1)} * a_{j1} * \\theta_i + d_{j(l-1)})}}$$'),
                                 br(),
                                 h4("Item characteristic curves"),
                                 plotOutput('bock_CC'),
                                 downloadButton("DP_bock_CC", label = "Download figure"),
                                 h4("Item information curves"),
                                 plotOutput('bock_IIC'),
                                 downloadButton("DP_bock_IIC", label = "Download figure"),
                                 h4("Test information function"),
                                 plotOutput('bock_TIF'),
                                 downloadButton("DP_bock_TIF", label = "Download figure"),
                                 h4("Table of parameters"),
                                 textOutput("bock_coef_warning"),
                                 tableOutput('bock_coef'),
                                 h4('Scatter plot of factor scores and standardized total scores'),
                                 plotOutput('bock_factor'),
                                 downloadButton("DP_bock_factor", label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('library(mirt)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, colnames(GMAT) != "group"]'),
                                     br(),
                                     br(),
                                     code('# Model'),
                                     br(),
                                     code('fit <- mirt(data, model = 1, itemtype = "nominal")'),
                                     br(),
                                     code('# Item Characteristic Curves'),
                                     br(),
                                     code('plot(fit, type = "trace", facet_items = F)'),
                                     br(),
                                     code('# Item Information Curves'),
                                     br(),
                                     code('plot(fit, type = "infotrace", facet_items = F)'),
                                     br(),
                                     code('# Test Information Function'),
                                     br(),
                                     code('plot(fit, type = "infoSE")'),
                                     br(),
                                     code('# Coefficients'),
                                     br(),
                                     code('coef(fit, simplify = TRUE)'),
                                     br(),
                                     code('coef(fit, IRTpars = TRUE, simplify = TRUE)'),
                                     br(),
                                     code('# Factor scores vs Standardized total scores'),
                                     br(),
                                     code('fs <- as.vector(fscores(fit))'),
                                     br(),
                                     code('sts <- as.vector(scale(apply(data, 1, sum)))'),
                                     br(),
                                     code('plot(fs ~ sts)')),
                                 br()
                                 )
                                 ),
             ###################
             # DIF/FAIRNESS ####
             ###################
             navbarMenu("DIF/Fairness",
                        "Used methods",
                        # * TOTAL SCORES ####
                        tabPanel("Total scores",
                                 h3("Total scores"),
                                 p('DIF is not about total scores! Two groups may have the same distribution of total scores, yet,
                                   some item may function differently for two groups. Also, one of the groups may have signifficantly
                                   lower total score, yet, it may happen that there is no DIF item!'),
                                 h4("Summary of total scores for groups"),
                                 tableOutput('resultsgroup'),
                                 h4("Histograms of total scores for groups"),
                                 sliderInput("inSlider2group", "Cut-Score", min=1, value=1, max=10,
                                             step=1, animate=TRUE),
                                 p('For selected cut-score, blue part of histogram shows students with total score
                                   above the cut-score, grey column shows students with Total Score equal
                                   to cut-score and red part of histogram shows students below the cut-score.'),
                                 plotOutput('histbyscoregroup0'),
                                 downloadButton("DP_histbyscoregroup0", label = "Download figure"),
                                 plotOutput('histbyscoregroup1'),
                                 downloadButton("DP_histbyscoregroup1", label = "Download figure"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(difNLR)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),
                                     code('# Summary table'),
                                     br(),
                                     code('sc_zero <- apply(data[group == 0, ], 1, sum); summary(sc_zero) # total scores of reference group'),
                                     br(),
                                     code('sc_one  <- apply(data[group == 1, ], 1, sum); summary(sc_one)  # total scores of focal group'),
                                     br(),
                                     code('# Histograms'),
                                     br(),
                                     code('hist(sc_zero, breaks = 0:20)'),
                                     br(),
                                     code('hist(sc_one, breaks = 0:20)')),
                                 br()
                                 ),
                        # * DELTA PLOTS ####
                        tabPanel("Delta plots",
                                 h3("Delta plot"),
                                 p('Delta plot (Angoff & Ford, 1973) compares the proportions of correct answers per
                                   item in the two groups. It displays non-linear transformation of these proportions using
                                   quantiles of standard normal distributions (so called delta scores) for each item for the two
                                   genders in a scatterplot called diagonal plot or delta plot (see Figure). Item is under
                                   suspicion of DIF if the delta point considerably departs from the diagonal. The detection
                                   threshold is either fixed to value 1.5 or based on bivariate normal approximation (Magis &
                                   Facon, 2012).'),

                                 radioButtons('type_threshold', 'Threshold',
                                              list("Fixed", "Normal")
                                 ),

                                 plotOutput('deltaplot'),
                                 downloadButton("DP_deltaplot", label = "Download figure"),
                                 br(),
                                 br(),
                                 verbatimTextOutput("dp_text_normal"),
                                 br(),
                                 h4("Selected R code"),
                                 div(code('library(deltaPlotR)'),
                                     br(),
                                     code('library(difNLR)'),
                                     br(),
                                     code('data(GMAT)'),
                                     br(),
                                     code('data <- GMAT[, 1:20]'),
                                     br(),
                                     code('group <- GMAT[, "group"]'),
                                     br(),
                                     br(),
                                     code('# Delta scores with fixed threshold'),
                                     br(),
                                     code('deltascores <- deltaPlot(data.frame(data, group), group = "group",
                                          focal.name = 1, thr = 1.5)'),
                                     br(),
                                     code('deltascores'),
                                     br(),
                                     code('# Delta plot'),
                                     br(),
                                     code('diagPlot(deltascores, thr.draw = T)'),
                                     br(),
                                     br(),
                                     code('# Delta scores with normal threshold'),
                                     br(),
                                     code('deltascores <- deltaPlot(data.frame(data, group), group = "group",
                                          focal.name = 1, thr = "norm")'),
                                     br(),
                                     code('deltascores'),
                                     br(),
                                     code('# Delta plot'),
                                     br(),
                                     code('diagPlot(deltascores, thr.draw = T)')),
                                 br()
                                 ),
                        # * MANTEL-HAENSZEL ####
                        tabPanel("Mantel-Haenszel",
                                 tabsetPanel(
                                   # Summary
                                   tabPanel("Summary",
                                            h3("Mantel-Haenszel test"),
                                            p('Mantel-Haenszel test is DIF detection method based on contingency
                                              tables that are calculated for each level of total score (Mantel &
                                              Haenszel, 1959).'),
                                            selectInput("correction_method_MZ_print", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"
                                                        ),
                                                        selected="BH"),
                                            verbatimTextOutput("print_DIF_MH"),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# Mantel-Haenszel test'),
                                                br(),
                                                code('fit <- difMH(Data = data, group = group, focal.name = 1,
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('fit')),
                                            br()
                                            ),
                                   tabPanel('Items',
                                            h3("Mantel-Haenszel test"),
                                            p('Mantel-Haenszel test is DIF detection method based on contingency
                                              tables that are calculated for each level of total score (Mantel &
                                              Haenszel, 1959).'),
                                            h4('Contingency tables and odds ratio calculation'),
                                            sliderInput("difMHSlider_item", "Item", animate = TRUE,
                                                        min = 1, max = 10, value = 1, step = 1),
                                            sliderInput("difMHSlider_score", "Cut-Score", min = 0, max = 10,
                                                        value = 1, step = 1),
                                            fluidRow(column(12, align = "center", tableOutput('table_DIF_MH'))),
                                            uiOutput('ORcalculation'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# Contingency table for item 1 and score 12'),
                                                br(),
                                                code('df <- data.frame(data[, 1], group)'),
                                                br(),
                                                code('colnames(df) <- c("Answer", "Group")'),
                                                br(),
                                                code('df$Answer <- relevel(factor(df$Answer, labels = c("Incorrect", "Correct")), "Correct")'),
                                                br(),
                                                code('df$Group <- factor(df$Group, labels = c("Reference Group", "Focal Group"))'),
                                                br(),
                                                code('score <- apply(data, 1, sum)'),
                                                br(),

                                                code('df <- df[score == 12, ]'),
                                                br(),

                                                code('tab <- dcast(data.frame(xtabs(~ Group + Answer, data = df)),
                                                     Group ~ Answer,
                                                     value.var = "Freq",
                                                     margins = T,
                                                     fun = sum)'),
                                                br(),
                                                code('tab'),
                                                br(),

                                                code('# Mantel-Haenszel estimate of OR'),
                                                br(),
                                                code('fit <- difMH(Data = data, group = group, focal.name = 1,
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('fit$alphaMH')),
                                            br()
                                            )
                        )
                        ),
                        # * LOGISTIC ####
                        tabPanel("Logistic",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Logistic regression on total scores'),
                                            p('Logistic regression allows for detection of uniform and non-uniform DIF (Swaminathan & Rogers, 1990) by adding a group
                                              specific intercept', strong('b2'), '(uniform DIF) and group specific interaction', strong('b3'), '(non-uniform DIF) into model and
                                              by testing for their significance.'),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | X_i, G_i, b_0, b_1, b_2, b_3\\right) = \\frac{e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}}{1+e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}} $$'),
                                            radioButtons('type_print_DIF_logistic', 'Type',
                                                         c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                           "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                           "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'
                                                         ),
                                                         'both'
                                            ),
                                            selectInput("correction_method_logSummary", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"
                                                        ),
                                                        selected="BH"),
                                            verbatimTextOutput('print_DIF_logistic'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),

                                                code('# Logistic regression DIF detection method'),
                                                br(),
                                                code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                                     type = "both",
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('fit')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Logistic regression on total scores'),
                                            p('Logistic regression allows for detection of uniform and non-uniform DIF by adding a group
                                              specific intercept', strong('b2'), '(uniform DIF) and group specific interaction', strong('b3'), '(non-uniform DIF) into model and
                                              by testing for their significance.'),
                                            h4("Plot with estimated DIF logistic curve"),
                                            p('Points represent proportion of correct answer with respect to standardized
                                              total score. Their size is determined by count of respondents who answered item correctly.'),
                                            p('NOTE: Plots and tables are based on DIF logistic procedure without any correction method. '),
                                            radioButtons('type_plot_DIF_logistic', 'Type',
                                                         c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                           "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                           "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'
                                                         ),
                                                         'both'
                                            ),
                                            selectInput("correction_method_logItems", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"),
                                                        selected="BH"),
                                            sliderInput("diflogSlider", "Item Slider", min=1, value=1, max=10,
                                                        step=1, animate=TRUE),
                                            plotOutput('plot_DIF_logistic'),
                                            downloadButton("DP_plot_DIF_logistic", label = "Download figure"),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | X_i, G_i, b_0, b_1, b_2, b_3\\right) = \\frac{e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}}{1+e^{b_0 + b_1 X_i + b_2 G_i + b_3 X_i G_i}} $$'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_logistic'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),

                                                code('# Logistic regression DIF detection method'),
                                                br(),
                                                code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                                     type = "both",
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('fit'),
                                                br(),

                                                code('# Plot of characteristic curve for item 1'),
                                                br(),
                                                code('plotDIFLogistic(data, group,
                                                     type = "both",
                                                     item =  1,
                                                     IRT = F,
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('# Coefficients'),
                                                br(),
                                                code('fit$logitPar')),
                                            br()
                                            )
                                   )
                                 ),

                        # * LOGISTIC Z ####
                        tabPanel("Logistic IRT Z",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Logistic regression on standardized total scores with IRT parameterization'),
                                            p('Logistic regression allows for detection of uniform and non-uniform DIF (Swaminathan & Rogers, 1990) by adding a group
                                              specific intercept', strong('bDIF'), '(uniform DIF) and group specific interaction', strong('aDIF'), '(non-uniform DIF) into model and
                                              by testing for their significance.'),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) = \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)
                                             \\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}{1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)
                                             \\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
                                            radioButtons('type_print_DIF_logistic_IRT_Z', 'Type',
                                                         c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                           "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                           "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'
                                                         ),

                                                         'both'
                                            ),
                                            selectInput("correction_method_logZSummary", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"),
                                                        selected="BH"),
                                            verbatimTextOutput('print_DIF_logistic_IRT_Z'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                code('scaled.score <- scale(score)'),
                                                br(),
                                                br(),
                                                code('# Logistic regression DIF detection method'),
                                                br(),
                                                code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                                     type = "both",
                                                     match = scaled.score,
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('fit')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Logistic regression on standardized total scores with IRT parameterization'),
                                            p('Logistic regression allows for detection of uniform and non-uniform DIF by adding a group
                                              specific intercept', strong('bDIF'), '(uniform DIF) and group specific interaction', strong('aDIF'), '(non-uniform DIF) into model and
                                              by testing for their significance.'),
                                            h4("Plot with estimated DIF logistic curve"),
                                            p('Points represent proportion of correct answer with respect to standardized
                                              total score. Their size is determined by count of respondents who answered item correctly.'),
                                            p('NOTE: Plots and tables are based on DIF logistic procedure without any correction method. '),
                                            radioButtons('type_plot_DIF_logistic_IRT_Z', 'Type',
                                                         c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                           "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                           "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'
                                                         ),
                                                         'both'
                                            ),
                                            selectInput("correction_method_logZItems", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"),
                                                        selected="BH"),
                                            sliderInput("diflog_irtSlider", "Item Slider", min=1, value=1, max=10,
                                                        step=1, animate=TRUE),
                                            plotOutput('plot_DIF_logistic_IRT_Z'),
                                            downloadButton("DP_plot_DIF_logistic_IRT_Z", label = "Download figure"),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) =
                                             \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}
                                             {1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_logistic_IRT_Z'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                code('scaled.score <- scale(score)'),
                                                br(),
                                                br(),
                                                code('# Logistic regression DIF detection method'),
                                                br(),
                                                code('fit <- difLogistic(Data = data, group = group, focal.name = 1,
                                                     type = "both",
                                                     match = scaled.score,
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('fit'),
                                                br(),

                                                code('# Plot of characteristic curve for item 1'),
                                                br(),
                                                code('plotDIFLogistic(data, group,
                                                     type = "both",
                                                     item =  1,
                                                     IRT = T,
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('# Coefficients for item 1 - recalculation'),
                                                br(),
                                                code('coef_old <- fit$logitPar[1, ]'),
                                                br(),
                                                code('coef <- c()'),
                                                br(),
                                                code('# a = b1, b = -b0/b1, adif = b3, bdif = -(b1b2-b0b3)/(b1(b1+b3))'),
                                                br(),
                                                code('coef[1] <- coef_old[2]'),
                                                br(),
                                                code('coef[2] <- -(coef_old[1] / coef_old[2])'),
                                                br(),
                                                code('coef[3] <- coef_old[4]'),
                                                br(),
                                                code('coef[4] <- -(coef_old[2] * coef_old[3] + coef_old[1] * coef_old[4] ) /
                                                     (coef_old[2] * (coef_old[2] + coef_old[4]))')),
                                            br()
                                                )
                                            )
                                   ),
                        # * NONLINEAR Z ####
                        tabPanel("Nonlinear Z",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Nonlinear regression on standardized total scores'),
                                            p('Nonlinear regression model allows for nonzero lower asymptote - pseudoguessing',
                                              strong('c.'), 'Similarly to logistic regression, also nonlinear regression allows for
                                              detection of uniform and non-uniform DIF by adding a group specific intercept',
                                              strong('bDIF'), '(uniform DIF) and group specific interaction', strong('aDIF'),
                                              '(non-uniform DIF) into the model and by testing for their significance.'),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, c_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) =
                                             c_j + \\left(1 - c_j\\right) \\cdot \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}
                                             {1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
                                            radioButtons('type_print_DIF_NLR', 'Type',
                                                         c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                           "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                           "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'
                                                         ),
                                                         'both'
                                            ),
                                            selectInput("correction_method_nlrSummary", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"),
                                                        selected="BH"),
                                            verbatimTextOutput('print_DIF_NLR'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('Data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# Nonlinear regression DIF method'),
                                                br(),
                                                code('fit <- difNLR(Data = Data, group = group, focal.name = 1,
                                                     model = "3PLcg", type = "both", p.adjust.method = "BH")'),
                                                br(),
                                                code('fit')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Nonlinear regression on standardized total scores'),
                                            p('Nonlinear regression model allows for nonzero lower asymptote - pseudoguessing',
                                              strong('c.'), 'Similarly to logistic regression, also nonlinear regression allows
                                              for detection of uniform and non-uniform DIF (Drabinova & Martinkova, 2016) by
                                              adding a group specific intercept', strong('bDIF'), '(uniform DIF) and group specific
                                              interaction', strong('aDIF'), '(non-uniform DIF) into the model and by testing for
                                              their significance.'),
                                            h4("Plot with estimated DIF nonlinear curve"),
                                            p('Points represent proportion of correct answer with respect to standardized
                                              total score. Their size is determined by count of respondents who answered item correctly.'),
                                            radioButtons('type_plot_DIF_NLR', 'Type',
                                                         c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                           "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                           "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'
                                                         ),
                                                         'both'
                                            ),
                                            selectInput("correction_method_nlrItems", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"),
                                                        selected="BH"),
                                            sliderInput("difnlrSlider", "Item Slider", min=1, value=1, max=10,
                                                        step=1, animate=TRUE),
                                            plotOutput('plot_DIF_NLR'),
                                            downloadButton("DP_plot_DIF_NLR", label = "Download figure"),
                                            h4("Equation"),
                                            ('$$\\mathrm{P}\\left(Y_{ij} = 1 | Z_i, G_i, a_j, b_j, c_j, a_{\\text{DIF}j}, b_{\\text{DIF}j}\\right) =
                                             c_j + \\left(1 - c_j\\right) \\cdot \\frac{e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}}
                                             {1+e^{\\left(a_j + a_{\\text{DIF}j} G_i\\right)\\left(Z_i -\\left(b_j + b_{\\text{DIF}j} G_i\\right)\\right)}} $$'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_NLR'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('Data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# Nonlinear regression DIF method'),
                                                br(),
                                                code('fit <- difNLR(Data = Data, group = group, focal.name = 1,
                                                     model = "3PLcg", type = "both", p.adjust.method = "BH")'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plot(fit, item = 1)'),
                                                br(),
                                                code('# Coefficients'),
                                                br(),
                                                code('fit$nlrPAR')),
                                            br()
                                            )
                                            )
                        ),
                        # * IRT LORD ####
                        tabPanel("IRT Lord",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Lord test for IRT models'),
                                            p('Lord test (Lord, 1980) is based on IRT model
                                              (1PL, 2PL, or 3PL with the same guessing). It uses the
                                              difference between item parameters for the two groups
                                              to detect DIF. In statistical terms, Lord statistic is
                                              equal to Wald statistic.'),
                                            br(),
                                            fluidRow(
                                              column(6, align="center",
                                                     img(src = 'lord_udif.png', width = 380)),
                                              column(6, align="center",
                                                     img(src = 'lord_nudif.png', width = 380))
                                            ),
                                            br(),
                                            radioButtons('type_print_DIF_IRT_lord', 'Model',
                                                         c("1PL" = '1PL',
                                                           "2PL" = '2PL',
                                                           "3PL" = '3PL'
                                                         ),
                                                         '2PL'
                                            ),
                                            selectInput("correction_method_DIF_IRT_lordSummary", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"),
                                                        selected="BH"),
                                            verbatimTextOutput('print_DIF_IRT_Lord'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# 2PL IRT MODEL'),
                                                br(),
                                                code('fit <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "2PL",
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('fit')),
                                            br()
                                            ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Lord test for IRT models'),
                                            p('Lord test (Lord, 1980) is based on IRT model
                                              (1PL, 2PL, or 3PL with the same guessing). It uses the
                                              difference between item parameters for the two groups
                                              to detect DIF. In statistical terms, Lord statistic is
                                              equal to Wald statistic.'),
                                            br(),
                                            h4('Plot with estimated DIF characteristic curve'),
                                            p('NOTE: Plots and tables are based on larger DIF IRT model. '),
                                            radioButtons('type_plot_DIF_IRT_lord', 'Model',
                                                         c("1PL" = '1PL',
                                                           "2PL" = '2PL',
                                                           "3PL" = '3PL'
                                                         ),
                                                         '2PL'
                                            ),
                                            selectInput("correction_method_DIF_IRT_lordItems", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"),
                                                        selected="BH"),
                                            sliderInput("difirt_lord_itemSlider", "Item Slider", min=1, value=1, max=10,
                                                        step=1, animate=TRUE),
                                            plotOutput('plot_DIF_IRT_Lord'),
                                            downloadButton("DP_plot_DIF_IRT_Lord", label = "Download figure"),
                                            h4("Equation"),
                                            uiOutput('irtint_lord'),
                                            uiOutput('irteq_lord'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_IRT_Lord'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# 2PL IRT MODEL'),
                                                br(),
                                                code('fit <- difLord(Data = data, group = group, focal.name = 1,
                                                     model = "2PL",
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('fit'),
                                                br(),
                                                code('# Coefficients for item 1'),
                                                br(),
                                                code('tab_coef <- fit$itemParInit[c(1, ncol(data) + 1), 1:2]'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef, item = 1)')),
                                            br()
                                            )
                                   )
                                   ),
                        # * IRT RAJU ####
                        tabPanel("IRT Raju",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Raju test for IRT models'),
                                            p('Raju test (Raju, 1988, 1990) is based on IRT
                                              model (1PL, 2PL, or 3PL with the same guessing). It
                                              uses the area between the item charateristic curves
                                              for the two groups to detect DIF.'),
                                            br(),
                                            fluidRow(
                                              column(6, align="center",
                                                     img(src = 'raju_udif.png', width = 380)),
                                              column(6, align="center",
                                                     img(src = 'raju_nudif.png', width = 380))
                                            ),
                                            br(),
                                            radioButtons('type_print_DIF_IRT_raju', 'Model',
                                                         c("1PL" = '1PL',
                                                           "2PL" = '2PL',
                                                           "3PL" = '3PL'
                                                         ),
                                                         '2PL'
                                            ),
                                            selectInput("correction_method_DIF_IRT_rajuSummary", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"),
                                                        selected="BH"),
                                            verbatimTextOutput('print_DIF_IRT_Raju'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# 2PL IRT MODEL'),
                                                br(),
                                                code('fit <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "2PL",
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('fit')),
                                            br()),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Raju test for IRT models'),
                                            p('Raju test (Raju, 1988, 1990) is based on IRT
                                              model (1PL, 2PL, or 3PL with the same guessing). It
                                              uses the area between the item charateristic curves
                                              for the two groups to detect DIF.'),
                                            br(),
                                            h4('Plot with estimated DIF characteristic curve'),
                                            p('NOTE: Plots and tables are based on larger DIF IRT model. '),
                                            radioButtons('type_plot_DIF_IRT_raju', 'Model',
                                                         c("1PL" = '1PL',
                                                           "2PL" = '2PL',
                                                           "3PL" = '3PL'
                                                         ),
                                                         '2PL'
                                            ),
                                            selectInput("correction_method_DIF_IRT_rajuItems", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"),
                                                        selected="BH"),
                                            sliderInput("difirt_raju_itemSlider", "Item Slider", min=1, value=1, max=10,
                                                        step=1, animate=TRUE),
                                            plotOutput('plot_DIF_IRT_Raju'),
                                            downloadButton("DP_plot_DIF_IRT_Raju", label = "Download figure"),
                                            h4("Equation"),
                                            uiOutput('irtint_raju'),
                                            uiOutput('irteq_raju'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DIF_IRT_Raju'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('library(difR)'),
                                                br(),
                                                code('data(GMAT)'),
                                                br(),
                                                code('data <- GMAT[, 1:20]'),
                                                br(),
                                                code('group <- GMAT[, "group"]'),
                                                br(),
                                                br(),
                                                code('# 2PL IRT MODEL'),
                                                br(),
                                                code('fit <- difRaju(Data = data, group = group, focal.name = 1,
                                                     model = "2PL",
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('fit'),
                                                br(),
                                                code('# Coefficients for item 1'),
                                                br(),
                                                code('tab_coef <- fit$itemParInit[c(1, ncol(data) + 1), 1:2]'),
                                                br(),
                                                code('# Plot of characteristic curve of item 1'),
                                                br(),
                                                code('plotDIFirt(parameters = tab_coef, item = 1, test = "Raju")')),
                                            br())
                                   )
                                   ),
                        # * DDF ####
                        tabPanel("DDF",
                                 tabsetPanel(
                                   # ** Summary ####
                                   tabPanel('Summary',
                                            h3('Differential Distractor Functioning with multinomial log-linear regression Model'),
                                            p('Differential Distractor Functioning (DDF) occurs when people from different
                                              groups but with the same knowledge have different probability of selecting
                                              at least one distractor choice. DDF is here examined by Multinomial Log-linear
                                              Regression model with Z-score and group membership as covariates. '),
                                            h4('Equation'),
                                            p('For ', strong('K'), ' possible test choices is the probability of the correct answer for
                                               person ', strong('i'), ' with standardized total score ', strong('Z'), ' and group
                                              membership ', strong('G'),' in item ', strong('j'), 'given by the following equation: '),
                                            ('$$\\mathrm{P}(Y_{ij} = K|Z_i, G_i, b_{jl0}, b_{jl1}, b_{jl2}, b_{jl3}, l = 1, \\dots, K-1) =
                                             \\frac{1}{1 + \\sum_l e^{\\left( b_{il0} + b_{il1} Z + b_{il2} G + b_{il3} Z:G\\right)}}$$'),
                                            p('The probability of choosing distractor ', strong('k'), ' is then given by: '),
                                            ('$$\\mathrm{P}(Y_{ij} = k|Z_i, G_i, b_{jl0}, b_{jl1}, b_{jl2}, b_{jl3}, l = 1, \\dots, K-1) =
                                             \\frac{e^{\\left( b_{jk0} + b_{jk1} Z_i + b_{jk2} G_i + b_{jk3} Z_i:G_i\\right)}}
                                                   {1 + \\sum_l e^{\\left( b_{jl0} + b_{jl1} Z_i + b_{jl2} G_i + b_{jl3} Z_i:G_i\\right)}}$$'),
                                            br(),
                                            radioButtons('type_print_DDF', 'Type',
                                                         c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                           "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                           "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'
                                                         ),
                                                         'both'
                                            ),
                                            selectInput("correction_method_print_DDF", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"),
                                                        selected = "BH"),
                                            verbatimTextOutput('print_DDF'),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('data(GMATtest, GMATkey)'),
                                                br(),
                                                code('Data <- GMATtest[, 1:20]'),
                                                br(),
                                                code('group <- GMATtest[, "group"]'),
                                                br(),
                                                code('key <- GMATkey'),
                                                br(),
                                                br(),
                                                code('# DDF with difNLR package'),
                                                br(),
                                                code('fit <- ddfMLR(Data, group, focal.name = 1, key, type = "both",
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('fit')),
                                            br()
                                   ),
                                   # ** Items ####
                                   tabPanel('Items',
                                            h3('Differential Distractor Functioning with multinomial log-linear regression model'),
                                            p('Differential Distractor Functioning (DDF) occurs when people from different
                                              groups but with the same knowledge have different probability of selecting
                                              at least one distractor choice. DDF is here examined by Multinomial Log-linear
                                              Regression model with Z-score and group membership as covariates. '),
                                            h4("Plot with estimated DDF curves"),
                                            p('Points represent proportion of selected answer with respect to standardized
                                              total score. Size of points is determined by count of respondents who chose particular
                                              answer.'),
                                            radioButtons('type_plot_DDF', 'Type',
                                                         c("H0: Any DIF vs. H1: No DIF" = 'both',
                                                           "H0: Uniform DIF vs. H1: No DIF" = 'udif',
                                                           "H0: Non-Uniform DIF vs. H1: Uniform DIF" = 'nudif'
                                                         ),
                                                         'both'
                                            ),
                                            selectInput("correction_method_plot_DDF", "Correction method",
                                                        c("BH" = "BH",
                                                          "Holm" = "holm",
                                                          "Hochberg" = "hochberg",
                                                          "Hommel" = "hommel",
                                                          "BY" = "BY",
                                                          "FDR" = "fdr",
                                                          "none" = "none"),
                                                        selected = "BH"),
                                            sliderInput("ddfSlider", "Item Slider", min = 1, value = 1, max = 10,
                                                        step = 1, animate = TRUE),
                                            plotOutput('plot_DDF'),
                                            downloadButton("DP_plot_DDF", label = "Download figure"),
                                            h4("Equation"),
                                            uiOutput('DDFeq'),
                                            h4("Table of parameters"),
                                            fluidRow(column(12, align = "center", tableOutput('tab_coef_DDF'))),
                                            br(),
                                            h4("Selected R code"),
                                            div(code('library(difNLR)'),
                                                br(),
                                                code('data(GMATtest, GMATkey)'),
                                                br(),
                                                code('Data <- GMATtest[, 1:20]'),
                                                br(),
                                                code('group <- GMATtest[, "group"]'),
                                                br(),
                                                code('key <- GMATkey'),
                                                br(),
                                                br(),
                                                code('# DDF with difNLR package'),
                                                br(),
                                                code('fit <- ddfMLR(Data, group, focal.name = 1, key, type = "both",
                                                     p.adjust.method = "BH")'),
                                                br(),
                                                code('# Estimated coefficients of item 1'),
                                                br(),
                                                code('fit$mlrPAR[[1]]')),
                                            br()
                                            )
                                 )
                                 ),

                        "----",
                        # * SUMMARY ####
                        "Description",
                        tabPanel('About DIF',
                                 h3('Differential Item Functioning / Item Fairness'),
                                 p('Differential item functioning (DIF) occurs when people from different
                                   groups (commonly gender or ethnicity) with the same underlying true
                                   ability have a different probability of answering the item correctly.
                                   If item functions differently for two groups, it is potentially unfair.
                                   In general, two type of DIF can be recognized: if the item has different
                                   difficulty for given two groups with the same discrimination, ',
                                   strong('uniform'), 'DIF is present (left figure). If the item has different
                                   discrimination and possibly also different difficulty for given two groups, ',
                                   strong('non-uniform'), 'DIF is present (right figure)'),
                                 br(),
                                 fluidRow(
                                   column(6, align="center",
                                          img(src = 'fig_NLR_uniformDIF.png', width = 380)),
                                   column(6, align="center",
                                          img(src = 'fig_NLR_nonuniformDIF.png', width = 380))
                                 ),
                                 br(),
                                 br()
                        )
                        ),
                ########################
                # REPORTS ##############
                ########################
                tabPanel("Reports",
                         h3("Download report"),
                         p("This shiny app also offers an option to download a report in HTML or PDF format."),
                         p("PDF report creation requires latest version of",
                           a("MiKTeX", href = "https://miktex.org/howto/install-miktex", target = "_blank"),
                           "(or other TeX distribution). If you don't have the latest installation, please, use the HTML report."),
                         radioButtons("report_format", "Format of Report",
                                     c("HTML" = "html",
                                       "PDF" = "pdf")),
                         radioButtons("irt_type_report", "IRT Model selection",
                                      c("Rasch" = "rasch",
                                        "1PL" = "1pl",
                                        "2PL" = "2pl",
                                        "3PL" = "3pl"),
                                      selected = "1pl"),
                         downloadButton("report", "Generate Report"),
                         p(strong("Warning"), ": download of Reports takes some time. Please, be patient.")
                         ),

             ########################
             # REFERENCES ###########
             ########################
             tabPanel("References",
                      h3('References'),
                      p('Akaike, H. (1974). A New Look at the Statistical Model Identification. IEEE Transactions
                        on Automatic Control, 19(6), 716-723.', a('See online.',
                                                                  href = "http://ieeexplore.ieee.org/abstract/document/1100705/",
                                                                  target = "_blank")),
                      p('Ames, A. J., & Penfield, R. D. (2015). An NCME Instructional Module on Item-Fit
                        Statistics for Item Response Theory Models. Educational Measurement: Issues and
                        Practice, 34(3), 39-48.', a("See online.",
                                                    href = "http://onlinelibrary.wiley.com/doi/10.1111/emip.12067/full",
                                                    target = "_blank")),
                      p('Angoff, W. H., & Ford, S. F. (1973). Item-Race Interaction on a Test of
                        Scholastic Aptitude. Journal of Educational Measurement, 10(2), 95-105.'),
                      p('Bock, R. D. (1972). Estimating Item Parameters and Latent Ability when
                        Responses Are Scored in Two or More Nominal Categories. Psychometrika,
                        37(1), 29-51.', a("See online.",
                                          href = "http://link.springer.com/article/10.1007/BF02291411",
                                          target = "_blank")),
                      p('Cronbach, L. J. (1951). Coefficient alpha and the internal structure
                        of tests. psychometrika, 16(3), 297-334.'),
                      p("Drabinova, A., & Martinkova, P. (2016). Detection of Differential Item
                        Functioning Based on Non-Linear Regression. Technical Report",
                        a("V-1229", href = "https://goo.gl/R3dpJ5", target = "_blank"), "."),
                      p("Lord, F. M. (1980). Applications of Item Response Theory to Practical Testing Problems.
                        Routledge."),
                      p("Magis, D., & Facon, B. (2012). Angoff's delta method revisited:
                        Improving DIF detection under small samples. British Journal of
                        Mathematical and Statistical Psychology, 65(2), 302-321."),
                      p("Mantel, N., & Haenszel, W. (1959). Statistical Aspects of the Analysis of Data from
                        Retrospective Studies. Journal of the National Cancer Institute, 22 (4), 719-748."),
                      p("Swaminathan, H., & Rogers, H. J. (1990). Detecting Differential Item
                        Functioning Using Logistic Regression Procedures. Journal of Educational
                        Measurement, 27(4), 361-370."),
                      p("Raju, N. S. (1988). The Area between Two Item Characteristic Curves. Psychometrika,
                        53 (4), 495-502."),
                      p("Raju, N. S. (1990). Determining the Significance of Estimated Signed and Unsigned Areas
                        between Two Item Response Functions. Applied Psychological Measurement, 14 (2), 197-207."),
                      p('Rasch, G. (1960) Probabilistic Models for Some Intelligence and Attainment Tests.
                        Copenhagen: Paedagogiske Institute.'),
                      p('Schwarz, G. (1978). Estimating the Dimension of a Model. The Annals of Statistics,
                        6(2), 461-464.', a('See online.',
                                           href = "https://projecteuclid.org/euclid.aos/1176344136",
                                           target = "_blank")),
                      p("Wilson, M. (2005). Constructing Measures: An Item Response Modeling Approach."),
                      p("Wright, B. D., & Stone, M. H. (1979). Best Test Design. Chicago: Mesa Press."),
                      br(),
                      br()
                      )
                      )
                      )
