#' One step analysis function
#'
#' Sequentially calls a function to test all accessible cellularities for all mutations in the samples,then cluster them, and finally draws 
#' phylogenetic trees based on the uncovered cellularities
#' @param SNV_list A list of dataframes (one for each sample), with as columns : (for the first column of the first sample the name of the sample), 
#' the chromosome "Chr",the position of the mutation "Start", the number of reads supporting the mutation "Alt", the depth of coverage at this locus "Depth",
#' and if the output from FREEC for the samples are not associated, the genotype "Genotype". 
#' @param FREEC_list list of dataframes from FREEC for each samples (usually named Sample_ratio.txt), in the same order as SNV_list
#' @param timepoints a numeric vector indicating if the samples are from different timepoints or tumors (e.g. one tumor and metastates) If NULL, 
#' all samples are considered from the same tumor.
#' @param nclone_range A number or range of clusters that should be used for clustering
#' @param clone_priors List of vectors with the putated position of clones
#' @param prior_weight Numeric with the proportion mutations in each clone
#' @param preclustering Boolean: should a kmeans be performed on A and AB sites to determine priors
#' @param maxit Number of initial conditions to be tested for the EM algorithm
#' @param simulated Should be TRUE if the data has been been generated by the QuantumCat algorithm
#' @param contamination Numeric vector describind the contamination in all samples (ranging from 0 to 1). Default is 0.
#' @param save_plot Should the plots be saved? Default is TRUE
#' @param ncores Number of cores to be used during EM algorithm
#' @param output_directory Path to output directory
#' @param epsilon Stop value: maximal admitted value of the difference in cluster position and weights between two optimization steps.
#' @keywords Clonal inference Cancer phylogeny
#' @export
#' @examples 
#' cat("Generate data : 2 clones ; 10 observed mutations; diploid tumor ; 
#'                          1000x sequencing depth; 2 samples; no contamination")
#' Mutations<-QuantumCat(number_of_clones=2,number_of_mutations=10,ploidy="AB",depth=1000,
#'                       number_of_samples=2,Random_clones=FALSE,contamination=c(0,0))
#'  for(i in 1:2){
#'  Mutations[[i]]<-cbind(rep(paste("Example_",i,sep=""),times=10),Mutations[[i]])
#'  colnames(Mutations[[i]])[1]<-"Sample"
#' }
#' print("The data should look like this:")
#' print(head(Mutations[[1]][,c(1:4,8,9)]))
#' 
#' cat("Cluster data: will try to cluster between 2 and 3 clones, with 1 maximum search each time, 
#'       and will use priors from preclustering (e.g. k-medoids on A and AB sites)")
#' print("The genotype is provided in the list frame, and 
#'           there is no associated data from FREEC to get genotype from.")
#' print("The computation will run on a single CPU.")
#' Clustering_output<-QuantumClone(SNV_list = Mutations,
#' FREEC_list = NULL,contamination = c(0,0),nclone_range = 2:3,
#' clone_priors = NULL,prior_weight = NULL , 
#' maxit = 1,preclustering = TRUE, simulated = TRUE,
#' save_plot = TRUE,ncores=1,output_directory="Example")
#' print("The data can be accessed by Clustering_output$filtered_data")
#' print("All plots are now saved in the working directory") 
#' 
QuantumClone<-function(SNV_list,FREEC_list=NULL,contamination,
                       nclone_range=2:5,clone_priors=NULL,prior_weight=NULL,
                       simulated=F,
                       save_plot=T, epsilon = 5*(10**(-3)),
                       maxit=8,preclustering=T,timepoints=NULL,ncores=1,output_directory=NULL){
  r<-One_step_clustering(SNV_list = SNV_list,FREEC_list = FREEC_list,contamination = contamination,nclone_range = nclone_range,
                         clone_priors = clone_priors,prior_weight =prior_weight ,
                         maxit = maxit,preclustering = preclustering,
                         simulated = simulated,
                         save_plot = save_plot,ncores=ncores,output_directory=output_directory)
  
#   t<-Tree_generation(Clone_cellularities = r$pamobject$medoids,timepoints = timepoints)
#   
#   if(save_plot){
#     pdf(paste(as.character(SNV_list[[1]][1,1]),'trees.pdf',sep='_'))
#     multiplot_trees(result_list =t,d = dim(r$pamobject$medoids)[1],cex = 0.8)
#     dev.off()
#   }
  return(r)
}

#' Cellularity clustering
#'
#' Wrap up function that clusters cellularities. This is based on the most likely possibility for each mutation, give ints frequency and genotype.
#' @param SNV_list A list of dataframes (one for each sample), with as columns : (for the first column of the first sample the name of the sample), 
#' the chromosome "Chr",the position of the mutation "Start", the number of reads supporting variant "Alt", as well as the total number of 
#' reads overlapping position "Depth",
#' and if the output from FREEC for the samples are not associated, the genotype "Genotype". 
#' @param FREEC_list list of dataframes from FREEC for each samples (usually named Sample_ratio.txt), in the same order as SNV_list
#' @param plot_3D_before_clustering Should a 3D plot be realized before clustering, can be useful to guess priors.
#' @param plot_3D Should a 3D density of probability be plotted? This option is only available with 2 samples. Default is FALSE
#' @param simulated Should be TRUE if the data has been been generated by the QuantumCat algorithm
#' @param save_plot Should the plots be saved? Default is TRUE
#' @param contamination Numeric vector giving the proportion of normal cells in each samples
#' @param maxit Number of initial conditions to be tested for the EM algorithm
#' @param preclustering Boolean: should a kmeans be performed on A and AB sites to determine priors
#' @param epsilon Stop value: maximal admitted value of the difference in cluster position and weights between two optimization steps.
#' @param ncores Number of cores to be used during EM algorithm
#' @param clone_priors List of vectors with the putated position of clones
#' @param prior_weight Numeric with the proportion mutations in each clone
#' @param nclone_range A number or range of clusters that should be used for clustering
#' @param restrict.to.AB Boolean: Should the analysis keep only sites located in A and AB sites in all samples?
#' @param output_directory Directory in which to save results
#' @keywords Clonal inference
#' @export
#' @examples
#' cat("Generate data : 2 clones ; 10 observed mutation;diploid tumor ; 
#'                          1000x sequencing depth; 1 sample; no contamination")
#' Mutations<-QuantumCat(number_of_clones=2,number_of_mutations=10,ploidy="AB",depth=1000,
#'                       number_of_samples=1,Random_clones=FALSE,contamination=c(0,0))
#'  Mutations[[1]]<-cbind(rep(paste("Example_",1,sep=""),times=100),Mutations[[1]])
#'  colnames(Mutations[[1]])[1]<-"Sample"

#' print("The data should look like this:")
#' print(head(Mutations[[1]][,c(1:4,8,9)]))
#' 
#' cat("Cluster data: will try to cluster between 2 and 3 clones, with 1 maximum search each time, 
#'       and will use priors from preclustering (e.g. k-medoids on A and AB sites)")
#' print("The genotype is provided in the list frame, and 
#'           there is no associated data from FREEC to get genotype from.")
#' print("The computation will run on a single CPU.")
#' Clustering_output<-QuantumClone(SNV_list = Mutations,
#' FREEC_list = NULL,contamination = 0,nclone_range = 2:3,
#' clone_priors = NULL,prior_weight = NULL , 
#' maxit = 1,preclustering = TRUE, simulated = TRUE,
#' save_plot = FALSE,ncores=1,output_directory="Example")
#' print("The data can be accessed by Clustering_output$filtered_data")
#' print("All plots are now saved in the working directory") 
#' 
One_step_clustering<-function(SNV_list,FREEC_list=NULL,
                              contamination,nclone_range=2:5,
                              clone_priors=NULL,prior_weight=NULL,maxit=8,preclustering=T,
                              simulated = F, epsilon = 5*(10**(-3)), 
                              save_plot = T,ncores=1,plot_3D = F, plot_3D_before_clustering = F,
                              restrict.to.AB = F,output_directory=NULL){  
  Sample_name<-as.character(SNV_list[[1]][1,1])
  Sample_names<-lapply(SNV_list,FUN = function(z) z[1,1])
  if(is.null(FREEC_list)){
    print("FREEC_list is empty. Checking that there is a genotype column in all samples...")
    check<-TRUE
    for(i in 1:length(SNV_list)){
      if(is.null(SNV_list[[i]][,"Genotype"])){
        print(paste("Sample",i,"does not have a Genotype column"))
        check<-FALSE
      }
    }
    if(!check){
      return(NA)
    }
    else{
      print("Genotype is provided.")
      Genotype_provided<-TRUE
    }
  }
  else{
    print("Checking that SNV_list and FREEC_list have the same number of samples...")
    if(length(SNV_list)!=length(FREEC_list)){
      print("Incorrect input: number of samples different for SNV and FREEC")
      return(NA)
    }
    print("Passed")
    Genotype_provided<-FALSE
  }
  print(paste("Checking all possibilities for",Sample_name))
  
  Cell<-From_freq_to_cell(SNV_list = SNV_list, 
                          FREEC_list = FREEC_list, 
                          Sample_names = Sample_names, 
                          contamination = contamination,
                          Genotype_provided = Genotype_provided, 
                          save_plot = save_plot,restrict.to.AB = restrict.to.AB,output_directory=output_directory)
  
  print("Starting clustering...")
  r<-Cluster_plot_from_cell(Cell = Cell,nclone_range = nclone_range,epsilon = epsilon,
                            Sample_names = Sample_names, preclustering = preclustering,
                            clone_priors = clone_priors,prior_weight = prior_weight,maxit = maxit,
                            simulated = simulated,save_plot=save_plot,contamination = contamination,
                            ncores=ncores,plot_3D_before_clustering =plot_3D_before_clustering,
                            output_directory=output_directory)
  if(plot_3D){
    print("Clustering done... Computing 3D structure")
    ThreeD_plot(r$filtered.data,contamination)
  }
  if(length(SNV_list)==1 & save_plot){
    q<-One_D_plot(EM_out = r,contamination = contamination)
    if(is.null(output_directory)){
      ggplot2::ggsave(plot = q, filename = paste(Sample_name,'/', 'Density', Sample_name,'.png',sep=''),width = 6.04,height = 6.04)
    }
    else{
      ggplot2::ggsave(plot = q, filename = paste(output_directory,'/', 'Density', Sample_name,'.png',sep=''),width = 6.04,height = 6.04)
    }
  }
    print("post-processing output...")
  for(i in 1:length(r$filtered.data)){
    to_bind<-data.frame()
    if(Genotype_provided){
        commonCols<-c("Chr","Start","Depth","Alt","Genotype")
    }else{
        commonCols<-c("Chr","Start","Depth","Alt")
    }
    for(chr in unique(r$filtered.data[[i]]$Chr)){
        spare<-SNV_list[[i]][SNV_list[[i]]$Chr==chr,]
    	spare<-spare[spare$Start %in% r$filtered.data[[i]]$Start,]
	to_bind<-rbind(to_bind,spare) 
    }
    r$filtered.data[[i]]<-merge(r$filtered.data[[i]], to_bind,by = commonCols)
  }
  return(r)
}

#' Wrap-up function
#'
#' Function that computes the most likely position for each mutation based on the genotype
#' @param SNV_list A list of dataframes (one for each sample), with as columns : (for the first column of the first sample the name of the sample), 
#' the chromosome "Chr",the position of the mutation "Start", the number of reads supporting the mutation "Alt", the depth of coverage at this locus "Depth",
#' and if the output from FREEC for the samples are not associated, the genotype "Genotype". 
#' @param FREEC_list list of dataframes from FREEC for each samples (usually named Sample_ratio.txt), in the same order as SNV_list
#' @param Genotype_provided If the FREEC_list is provided, then should be FALSE (default), otherwise TRUE
#' @param save_plot Should the plots be saved? Default is TRUE
#' @param Sample_names Name of the samples
#' @param contamination Numeric vector giving the contamination by normal cells
#' @param ncores Number of cores to be used during EM algorithm
#' @param restrict.to.AB Should the analysis keep only sites located in A and AB sites in all samples?
#' @param output_directory Directory in which to save results
#' @keywords Clonal inference
#' 
From_freq_to_cell<-function(SNV_list,FREEC_list=NULL,Sample_names,Genotype_provided=F,save_plot=T,
                            contamination,ncores = 4, restrict.to.AB = F,output_directory=NULL){
  if(save_plot){
    if(is.null(output_directory)){
      dir.create(path = paste(Sample_names[1]), showWarnings = FALSE)
    }
    else{
      dir.create(path=output_directory,showWarnings = FALSE)
    }
  }
  if(Genotype_provided){
    Schrod_out<-Patient_schrodinger_cellularities(SNV_list = SNV_list,Genotype_provided =T,
                                                  contamination = contamination, restrict.to.AB = restrict.to.AB)
  }
  else{
    Schrod_out<-Patient_schrodinger_cellularities(SNV_list = SNV_list, FREEC_list = FREEC_list,
                                                  contamination = contamination, restrict.to.AB = restrict.to.AB)
  }
  
  if(save_plot){
    plot_cell_from_Return_out(Schrod_out,Sample_names,output_directory)
  }
  
  return(Schrod_out)
}

#' Patient Schrodinger Cellularities
#'
#' Computes all possible cellularities for all mutations across all samples. Calls CellularitiesFromFreq on all mutations to evaluate all possibilities
#' @param SNV_list A list of dataframes (one for each sample), with as columns : (for the first column of the first sample the name of the sample), 
#' the chromosome "Chr",the position of the mutation "Start", the number of reads supporting the mutation "Alt", the depth of coverage at this locus "Depth", 
#' and if the output from FREEC for the samples are not associated, the genotype "Genotype". 
#' @param FREEC_list list of dataframes from FREEC for each samples (usually named Sample_ratio.txt), in the same order as SNV_list
#' @param Genotype_provided If the FREEC_list is provided, then should be FALSE (default), otherwise TRUE
#' @param contamination Numeric vector describind the contamination in all samples (ranging from 0 to 1). Default is 0.
#' @param restrict.to.AB Should the analysis keep only sites located in A and AB sites in all samples?
#' @keywords Clonal inference

Patient_schrodinger_cellularities<-function(SNV_list,FREEC_list=NULL,Genotype_provided=F,contamination, restrict.to.AB = F){
  result<-list()
  count<-0
  id<-1
  chr<-SNV_list[[1]][,"Chr"]
  chr_ante<-0
  for(i in 1:dim(SNV_list[[1]])[1]){ ##Exploring all possibilities for each mutation
    Cell<-list()
    test<-T
    for(k in 1:length(SNV_list)){ 
      if(test){ ## Do not look at position if it is invalid in another sample previously explored
        if(!is.null(SNV_list[[k]]$subclone.genotype)){
          if(is.na(SNV_list[[k]]$subclone.genotype)){
            subclone.geno<-NULL
          }
        }
        else{
          subclone.geno<-SNV_list[[k]][i,"subclone.genotype"]
        }
        if(Genotype_provided){
          Cell[[k]]<-cbind(CellularitiesFromFreq(Genotype= as.character(SNV_list[[k]][i,'Genotype']),Alt = SNV_list[[k]][i,"Alt"],
                                                 Depth = SNV_list[[k]][i,"Depth"],subclone.genotype = subclone.geno,
                                                 subclone.cell = SNV_list[[k]][i,"subclone.cell"],
                                                 chr = SNV_list[[k]][i,'Chr'], position = SNV_list[[k]][i,'Start'],
                                                 contamination = contamination[k],restrict.to.AB = restrict.to.AB),
                           id)
        }
        else{
          if(chr[i]!=chr_ante){
            CHR_FREEC<-lapply(FREEC_list,function(z) z[as.character(z[,"Chromosome"])==strsplit(as.character(chr[i]),split = "r")[[1]][2],])
            chr_ante<-chr[i]
          }
          Cell[[k]]<-cbind(CellularitiesFromFreq(Freec_ratio = CHR_FREEC[[k]],
                                                 Alt = SNV_list[[k]][i,"Alt"],Depth = SNV_list[[k]][i,"Depth"],
                                                 subclone.genotype = subclone.geno,
                                                 subclone.cell = SNV_list[[k]][i,"subclone.cell"],
                                                 chr = SNV_list[[k]][i,'Chr'], position = SNV_list[[k]][i,'Start'],
                                                 contamination = contamination[k],restrict.to.AB = restrict.to.AB),
                           id)
        }
        if(sum(is.na(Cell[[k]]))>0){
          test<-F
          count<-count+1
        }
      }
    }
    if(test){ ## Checking that genotype is available
      L<-list()
      for(r in 1:length(Cell)){
        L[[r]]<-1:(dim(Cell[[r]])[1])
      }
      U<-expand.grid(L) ##Table with all Cell row combinations
         for(k in 1:(dim(U)[2])){
              if(id==1){
                result[[k]]<-Cell[[k]][U[,k],]
              }
              else{
                result[[k]]<-rbind(result[[k]],Cell[[k]][U[,k],])
              }
          }
        id<-id+1
      }
  }
  if(count>0){
    warning(paste(count,'mutations exluded due to missing genotype or normalization issues'))
  }
  return(result)
}

#' Cellularities from allele frequency
#'
#' Creates all possibilities for one mutation in one sample (given a genotype), then computes 
#' the cellularity associated with each possibility and finally the probability of each possibility
#' @param chr The chromosome on which is the position (numeric value, not chr1 as in BED files)
#' @param position The genomic position of the mutation
#' @param Freec_ratio The FREEC output associated with the sample of interest
#' @param Genotype If the FREEC output is not given, the genotype associated with the locus (for example AAB)
#' @param Alt Number of reads supporting the variation
#' @param Depth Number of reads mapped at the position
#' @param subclone.genotype If existing, the genotype of the subclone. Else NULL
#' @param subclone.cell The cellular prevalence of the subclone which has a different Copy Number at this site
#' @param contamination The fraction of normal cells in the sample
#' @param restrict.to.AB Should the analysis keep only sites located in A and AB sites in all samples?
#' @keywords Clonal inference

CellularitiesFromFreq<-function(chr, position,Alt,Depth, 
                                Freec_ratio=NULL, Genotype=NULL,subclone.genotype=NULL,
                                subclone.cell=NULL,contamination, restrict.to.AB = F){##For 1 mutation
  if(!is.null(Freec_ratio)){
    if(grepl(pattern = "chr",x = chr,ignore.case = T)){
      FChr<-sapply(X = 'chr',FUN = paste, Freec_ratio[,'Chromosome'],sep='')
    }
    else{
      FChr<-Freec_ratio[,'Chromosome']
    }
    Genotype<-as.character(tail(Freec_ratio[FChr==chr & Freec_ratio[,'Start']<position,'Genotype'],1))
  }
  alpha<-double()
  if(length(Genotype)==0){
    print(paste("Genotype is not available at position",chr,position))
    return(NA)
  }
  else if(is.na(Genotype)){
    print(paste("Genotype is not available at position",chr,position))
    return(NA)
  }
  else if(Genotype==-1){
    print(paste("Genotype is not available at position",chr,position))
    return(NA)
  }
  else if(restrict.to.AB & (Genotype!='A' & Genotype!='AB')){
    print(paste("Position",chr,position,"is not in an A or AB site. Genotype:",Genotype))
    return(NA)
  }
  else if (is.null(subclone.genotype) | is.null(subclone.cell)){
    result<-data.frame()
    As<-strcount(x = Genotype, pattern = 'A',split = '')
    Ns<-nchar(Genotype)
    for(i in 1:As){
      Cellularity<-as.numeric(Alt/Depth*Ns/(i*1-contamination))
      spare<-data.frame(chr,position,Cellularity, Genotype,Alt,Depth,i,Ns)
      colnames(spare)<-c('Chr','Start','Cellularity','Genotype',"Alt","Depth","NC","NCh")
      result<-rbind(result,spare)
      alpha<-c(alpha,choose(As,i)+choose(Ns-As,i))
    }
  }
  else{ ## Two possibilities: belong to clone or subclone
    result<-data.frame()
    As<-strcount(x = Genotype, pattern = 'A',split = '')
    Ns<-nchar(Genotype)
    for(i in 1:As){
      Cellularity<-as.numeric(frequency/100*Ns/i*(1/(1-contamination)))
      spare<-data.frame(chr,position,Cellularity, Genotype,Alt,Depth,i,Ns)
      colnames(spare)<-c('Chr','Start','Cellularity','Genotype',"Alt","Depth","NC","NCh")
      result<-rbind(result,spare)
      alpha<-c(alpha,(1-subclone.cell)*(choose(As,i)+choose(Ns-As,i)))
    }
    A.sub<-strcount(x=subclone.genotype,pattern = "A",split = '')
    N.sub<-nchar(subclone.genotype)
    for(j in A.sub){ 
      ### Keep only possibilities that have a cellularity lower than the subclonal cellularity
      
      Cellularity<-as.numeric(frequency/100*N.sub/j*(1/(1-contamination)))
      if(Cellularity<=subclone.cell){
        spare<-data.frame(chr,position,Cellularity, subclone.genotype,Alt,Depth,j,N.sub)
        colnames(spare)<-c('Chr','Start','Cellularity','Genotype',"Alt","Depth","NC","NCh")
        result<-rbind(result,spare)
        alpha<-c(alpha,(subclone.cell)*(choose(A.sub,j)+choose(N.sub-A.sub,j)))
      }
    }
  }
  if(sum(is.nan(alpha))>0){
    
    return(NA)
  }
  else{
    alpha<-alpha/(sum(alpha))
  }
  result<-data.frame(result,alpha)
  colnames(result)<-c('Chr','Start','Cellularity','Genotype',"Alt","Depth","NC","NCh",'alpha')
  return(result)
}

#'String count
#'
#' Counting the number of characters for each element of a vector
#'
#' @param x The vector from which elements should be counted
#' @param pattern Pattern to be recognized. Default is ''
#' @param split Pattern used to split elements of the vector. Default is ''
#' @keywords Text handling

strcount <- function(x, pattern='', split=''){
  
  unlist(lapply(strsplit(x, split),function(z) na.omit(length(grep(pattern, z)))))  
}
#' Cellularity clustering
#'
#' Clustering cellularities based on  the most likely presence of a clone, using the pamk algorithm (fpc package). Clustering can be guided by toggling manual_clustering on and/or giving a range of number of clusters.
#' @param Cell Output from Return_one_cell_by_mut, list of cellularities (one list-element per sample)
#' @param Sample_names  Name of the sample
#' @param simulated Was the data generated by QuantumCat?
#' @param save_plot Should the clustering plots be saved? Default is True
#' @param contamination The fraction of normal cells in the samples
#' @param clone_priors If known a list of priors (cell prevalence) to be used in the clustering
#' @param prior_weight If known a list of priors (fraction of mutations in a clone) to be used in the clustering
#' @param nclone_range Number of clusters to look for
#' @param maxit Maximal number of independant initial condition tests to be tried
#' @param preclustering Boolean: should a kmeans be performed on A and AB sites to determine priors
#' @param epsilon Stop value: maximal admitted value of the difference in cluster position and weights between two optimization steps.
#' @param ncores Number of CPUs to be used
#' @param plot_3D_before_clustering Should a 3D plot be realized before clustering, can be useful to guess priors.
#' @param output_directory Directory in which to save results
#' @import fpc
#' @keywords Clonal inference
#' 
Cluster_plot_from_cell<-function(Cell,Sample_names,simulated,save_plot=T,
                                 contamination, clone_priors,prior_weight,nclone_range,maxit,preclustering=T,
                                 epsilon=5*(10**(-3)),ncores = 2,plot_3D_before_clustering = F,output_directory=NULL){
  preclustering_success<-F
  if(plot_3D_before_clustering){
    ThreeD_plot(Cell,contamination = contamination)
  }
  if(preclustering){
    for(i in 1:length(Cell)){
      if(i==1){
        select<-Cell[[1]][,"Genotype"]=='A'| Cell[[1]][,"Genotype"]=='AB'
        Spare<-Cell[[1]][,'Cellularity']
      }
      else{
        select<- select & (Cell[[i]][,"Genotype"]=='A'| Cell[[i]][,"Genotype"]=='AB')
        Spare<-cbind(Spare,Cell[[i]][,'Cellularity'])
      }
    }
    if(length(Cell)==1){
      Spare<-Spare[select]
      if(length(Spare)==0){
        warning("No A and AB sites to do preclustering")
        p_clone<-clone_priors
        p_weight<-prior_weight
      }
    }
    else{
      Spare<-Spare[select,] ##restriction to A AB sites
    }
    if(is.null(dim(Spare))){
      warning("No A and AB sites to do preclustering")
      p_clone<-clone_priors
      p_weight<-prior_weight
    }
    else{
      if(dim(Spare)[1]<=max(nclone_range)){
        warning("Too few mutations to cluster. Will use priors / random initial conditions")
        p_clone<-clone_priors
        p_weight<-prior_weight
      }
      else{
        Spare[Spare>1]<-1
        kmeans<-fpc::pamk(Spare,krange = nclone_range,usepam = F)
        p_clone<-list()
        p_weight<-numeric()
        create_prior_weight<- dim(Spare)[1]>=50
        for(j in 1:(dim(kmeans$pamobject$medoids)[2])){
          p_clone[[j]]<-as.numeric(kmeans$pamobject$medoids[,j])
          if(create_prior_weight){
            p_weight[j]<-sum(kmeans$pamobject$clustering==j)/length(kmeans$pamobject$clustering)
          }
        }
        if(!create_prior_weight){
          p_weight<-prior_weight
        }
        preclustering_success<-T
      }
    }
  }
  else{
    p_clone<-clone_priors
    p_weight<-prior_weight
  }
  if(preclustering_success){
    result<-EM_clustering(Schrod = Cell,contamination = contamination,epsilon = epsilon,
                          prior_weight = p_weight,clone_priors = p_clone,maxit = 1,nclone_range = nclone_range,ncores = ncores)
  }
  else{
    result<-EM_clustering(Schrod = Cell,contamination = contamination,epsilon = epsilon,
                          prior_weight = p_weight,clone_priors = p_clone,maxit = maxit,nclone_range = nclone_range,ncores = ncores)
  }
  for(i in 1:length(Cell)){
    if(i==1){
      Spare<-result$filtered.data[[1]]$Cellularity
    }
    else{
      Spare<-cbind(Spare,result$filtered.data[[i]]$Cellularity)
    }
  }
  M<-max(as.numeric(as.character(result$cluster)))
  cluster<-factor(result$cluster)
  if(length(Cell)>1){
    U<-expand.grid(1:length(Cell),1:length(Cell))
    U<-U[U[,1]<U[,2],]
    for(i in 1:dim(U)[1]){
      if(save_plot){
        if(!simulated){
          q<-ggplot2::qplot(x=Spare[,U[i,1]],y=Spare[,U[i,2]] , asp = 1,main=paste('Cellular prevalence',Sample_names[[U[i,1]]],Sample_names[[U[i,2]]]),
                            xlab=paste('Cellular prevalence',Sample_names[[U[i,1]]]),ylab=paste('Cellular prevalence',Sample_names[[U[i,2]]]), 
                            colour = cluster)+ggplot2::scale_colour_discrete(name='Clone')+ggplot2::coord_cartesian(xlim=c(0,1),ylim=c(0,1))+ggplot2::theme_bw()
          if(is.null(output_directory)){
            ggplot2::ggsave(plot = q, filename = paste(Sample_names[1],'/', 'Cellularity_clustered',"_", Sample_names[[U[i,1]]],"_",Sample_names[[U[i,2]]],"_",U[i,1],'_',U[i,2], '.png',sep=''),width = 6.04,height = 6.04)
          }
          else{
            ggplot2::ggsave(plot = q, filename = paste(output_directory,'/', 'Cellularity_clustered',"_", Sample_names[[U[i,1]]],"_",Sample_names[[U[i,2]]],"_",U[i,1],'_',U[i,2], '.png',sep=''),width = 6.04,height = 6.04)
            
          }
          
        }
        else{
          q<-ggplot2::qplot(x=Spare[,U[i,1]],y=Spare[,U[i,2]] , asp = 1,main=paste('Cellular prevalence plot',Sample_names[[U[i,1]]],Sample_names[[U[i,2]]]),
                            xlab=paste('Cellular prevalence',Sample_names[[U[i,1]]]),ylab=paste('Cellular prevalence',Sample_names[[U[i,2]]]),
                             colour = cluster,
                             shape=factor(result$filtered.data[[1]]$Chr))+ggplot2::theme_bw()+ggplot2::scale_shape_discrete(factor(1:max(Cell[[1]][,'Chr'])),
                                                                                                   name='Clone \n(simulated)')+ggplot2::scale_colour_discrete(name='Cluster')+ggplot2::coord_cartesian(xlim=c(0,1),ylim=c(0,1))+ggplot2::theme_bw()
          if(is.null(output_directory)){
            ggplot2::ggsave(plot = q, filename = paste(Sample_names[1],'/', 'Cellularity_clustered', Sample_names[[U[i,1]]],"_",Sample_names[[U[i,2]]],"_",U[i,1],'_',U[i,2], '.png',sep=''),width = 6.04,height = 6.04)
          }
          else{
            ggplot2::ggsave(plot = q, filename = paste(output_directory,'/', 'Cellularity_clustered', Sample_names[[U[i,1]]],"_",Sample_names[[U[i,2]]],"_",U[i,1],'_',U[i,2], '.png',sep=''),width = 6.04,height = 6.04)
          }
        }
      }
    }
  }
  else{
    if(save_plot){
      if(!simulated){
        q<-ggplot2::qplot(x=Spare, y=jitter(rep(0.5,times=length(Spare)),factor = 5) , asp = 1,main=paste('Cellular prevalence',Cell[[1]][1,'Sample']),
                          xlab=paste('cellularity',Sample_names),ylab='', 
                          colour = cluster)+ggplot2::scale_colour_discrete(name='Clone')+ggplot2::coord_cartesian(xlim=c(0,1),ylim=c(0,1))+ggplot2::theme_bw()+ggplot2::theme(axis.line.y=ggplot2::element_blank(),
                                                                                                                                          axis.ticks.y=ggplot2::element_blank(),
                                                                                                                                          panel.background  = ggplot2::element_blank(),
                                                                                                                                          axis.text.y = ggplot2::element_blank())
        if(is.null(output_directory)){
          ggplot2::ggsave(plot = q, filename = paste(Sample_names[1],'/', 'Cellularity_clustered', Sample_names[1], '.png',sep=''),width = 6.04,height = 6.04)
        } 
        else{
          ggplot2::ggsave(plot = q, filename = paste(output_directory,'/', 'Cellularity_clustered', Sample_names[1], '.png',sep=''),width = 6.04,height = 6.04)
          
        }
      }
      else{
        q<-ggplot2::qplot(x=Spare,y=jitter(rep(0.5,times=length(Spare)),factor = 5), asp = 1,main=paste('Cellular prevalence',Cell[[1]][1,'Sample']),
                          xlab=paste('Cellular prevalence',Sample_names),ylab='',
                          colour = cluster)+ggplot2::scale_colour_discrete(name='Cluster')+ggplot2::coord_cartesian(xlim=c(0,1),ylim=c(0,1))+ggplot2::scale_shape_discrete(factor(1:max(Cell[[1]][,'Chr'])),
                                                                                                                                                name='Clone \n(simulated)')+ggplot2::theme_bw()+ggplot2::theme(axis.line.y=ggplot2::element_blank(),
                                                                                                                                                                                  axis.ticks.y=ggplot2::element_blank(),
                                                                                                                                                                                  panel.background  = ggplot2::element_blank(),
                                                                                                                                                                                  axis.text.y = ggplot2::element_blank())
        if(is.null(output_directory)){
          ggplot2::ggsave(plot = q, filename = paste(Sample_names[1],'/', 'Cellularity_clustered', Sample_names[1],'.png',sep=''),width = 6.04,height = 6.04)
        }
        else{
          ggplot2::ggsave(plot = q, filename = paste(output_directory,'/', 'Cellularity_clustered', Sample_names[1],'.png',sep=''),width = 6.04,height = 6.04)
          
        }
      }
    }
  }
  return(result)
}

#' Plot cellularity
#'
#' 2D plot of cellularity based on the output of the EM
#' @param lis Output from Return_one_cell_by_mut, list of cellularities (one list-element per sample)
#' @param Sample_names Name of the samples.
#' @param output_dir Directory in which to save plots
#' @keywords Clonal inference plot
#'
plot_cell_from_Return_out<-function(lis,Sample_names,output_dir=NULL){
  if(length(lis)>1){
    U<-expand.grid(1:length(lis),1:length(lis))
    U<-U[U[,1]<U[,2],]
    Sample_names<-lapply(lis,FUN = function(z) z[1,1])
    for(i in 1:dim(U)[1]){
      d<-ggplot2::qplot(x = lis[[U[i,1]]][,'Cellularity'], y = lis[[U[i,2]]][,'Cellularity'],asp = 1,
               xlab=paste('Cellular prevalence',Sample_names[[U[i,1]]]),ylab=paste('Cellular prevalence',Sample_names[[U[i,2]]]),
               main=paste('Cellular prevalence of all possibilities'))+ggplot2::coord_cartesian(xlim=c(0,1),ylim=c(0,1))+ggplot2::theme_bw()
      if(is.null(output_dir)){
        ggplot2::ggsave(filename = paste(Sample_names[[1]],'/', 'Cellularity', Sample_names[[U[i,1]]],"_",Sample_names[[U[i,2]]],'.png',sep=''),plot = d,width = 6.04,height = 6.04)
      }
      else{
        ggplot2::ggsave(filename = paste(output_dir,'/', 'Cellularity', Sample_names[[U[i,1]]],"_",Sample_names[[U[i,2]]],'.png',sep=''),plot = d,width = 6.04,height = 6.04)
        
      }
    }
  }
  else{
    d<-ggplot2::qplot(x = lis[[1]][,'Cellularity'], y = jitter(rep(0.5,times=length(lis[[1]][,'Cellularity'])),factor = 10),asp = 1,
             xlab=paste('Cellular prevalence'),ylab='',
             main=paste(Sample_names,'Cellular prevalence of all possibilities'))
    d<-d+ggplot2::coord_cartesian(xlim=c(0,1),ylim=c(0,1))+ggplot2::theme_bw()+ggplot2::theme(axis.line.y=ggplot2::element_blank(),
                                                                                axis.ticks.y=ggplot2::element_blank(),
                                                                                axis.text.y = ggplot2::element_blank())
    if(is.null(output_dir)){
      ggplot2::ggsave(filename = paste(Sample_names[1],'/', 'Cellularity', Sample_names,'.png',sep=''),plot = d,width = 6.04,height = 6.04)
    }
    else{
      ggplot2::ggsave(filename = paste(output_dir,'/', 'Cellularity', Sample_names,'.png',sep=''),plot = d,width = 6.04,height = 6.04)
      
    }
  }
}

#' Plots
#' 
#' Creates density plot when only one sample is given
#' @param EM_out output from the EM algorithm
#' @param contamination Numeric vector giving the proportion of normal cells in each samples
#' @keywords Clonal inference phylogeny

One_D_plot<-function(EM_out,contamination){
  theta=seq(from = 0,to = 1,by = 0.0001)
  p<-outer(theta,EM_out$filtered.data[[1]]$NC*(1-contamination)/ EM_out$filtered.data[[1]]$NCh)
  P<-choose(EM_out$filtered.data[[1]]$Depth,EM_out$filtered.data[[1]]$Alt)*p**(EM_out$filtered.data[[1]]$Alt)*(1-p)**(EM_out$filtered.data[[1]]$Depth-EM_out$filtered.data[[1]]$Alt)
  y<-apply(X = P,MARGIN = 1,FUN = sum)
  y<-y/sum(y)
  r<-ggplot2::qplot(x=theta,y=y,geom = "line",main="Density of probability of presence of a clone",xlab="Cell fraction",ylab="density")+ggplot2::theme_bw()
  return(r)
}
#' 3D Plot
#' 
#' Creates density plot when two samples are given
#' @param Schrod List of 2 dataframes, output of the Schrodinger function or the EM algorithm
#' @param contamination Numeric vector giving the proportion of normal cells in each samples
#' @keywords Clonal inference phylogeny
#' @export
#' @examples 
#' print("Generating data...")
#' QC<-QuantumCat(number_of_clones = 4,number_of_mutations = 200, 
#'                depth= 100, ploidy = "AB")
#' print("Formating data so it can be used as input")
#' alpha<-rep(1,time=200)
#' for(i in 1:2){
#' print(head(2*QC[[i]]$Alt/QC[[i]]$Depth))
#' QC[[i]][,4]<-2*QC[[i]]$Alt/QC[[i]]$Depth
#' QC[[i]]<-cbind(QC[[i]],alpha)
#' }
#'  print("Creating 3D plot")                     
#' ThreeD_plot(QC,c(0,0))
ThreeD_plot<-function(Schrod,contamination){
  Z<-matrix(nrow=101,ncol=101)
  alpha<-Schrod[[1]]$alpha*Schrod[[2]]$alpha
  for(i in 0:100){
    Z[,i+1]<-sapply(X=0:100,function(y) {
        if(i<100){
          S<-sum((Schrod[[1]]$Cellularity>=floor(x = i/10)/10 & Schrod[[1]]$Cellularity<floor(x = i/10+1)/10 & Schrod[[2]]$Cellularity>=floor(x = y/10)/10 & Schrod[[2]]$Cellularity<floor(x = y/10 +1)/10)*alpha)
        }
        else{
          S<-sum((Schrod[[1]]$Cellularity>=floor(x = i/10)/10 & Schrod[[2]]$Cellularity>=floor(x = y/10)/10 & Schrod[[2]]$Cellularity<floor(x = y/10 +1)/10)*alpha)
        }
        if(is.na(S)){
          return(0)
        }
        else{
          return(S)
        }
      }
    )
  }
  rgl::open3d()
  palette <- colorRampPalette(c('royalblue',"blue",'cyan','grey','yellow','orange','red'))
  col.table <- palette(1024)
  col.ind <- cut(Z, 1024)
  rgl::persp3d(x=(0:100)/100,y=(0:100)/100,z=Z,xlab = 'Cellularity diag',ylab = 'Cellularity relapse',
               zlab = 'weighted number of possibilities',col=col.table[col.ind],xlim=c(0,1),ylim=c(0,1)) 
} 


#' Phylogenetic tree
#'
#' Generates a list of possible trees based on the cellularity of each clone, and the spatial and temporal distribution of the samples.
#' Assumption is made the different clones are on different lines of the matrix
#' @param Clone_cellularities A dataframe with cellularities (ranging from 0 to 1) of each clone (rows) in each sample (columns)
#' @param timepoints A numeric vector giving the spatial and/or temporal distribution of the samples
#' @keywords Clonal inference phylogeny
# Clone_cell<-data.frame(cbind(c(1,0.5,0.3,0.8),c(1,0.05,0.6,0.7)))
# print("Using clone cellularities:")
# print(Clone_cell)
# Tree_generation(Clone_cell)
Tree_generation<-function(Clone_cellularities,timepoints=NULL){
  if(is.data.frame(Clone_cellularities)){
    Clone_cellularities<-as.matrix(Clone_cellularities)
  }
  #Inclusion detection
  if(is.null(timepoints)){
    timepoints<-rep(1,times=dim(Clone_cellularities)[2])
  }
  else if(length(timepoints)!=dim(Clone_cellularities)[2]){
    warning('Temporal vector and number of samples are of different sizes')
  }
  nr<-dim(Clone_cellularities)[1]
  nc<-dim(Clone_cellularities)[2]
  print(Clone_cellularities)
  
  ####Creating network matrix (inclusion), 1 at line i col j means i is included in j
  M<-matrix(0,nr,nr)
  for(i in 1:nr){
    for(j in 1:nr){
      if(i!=j & is_included(Clone_cellularities[i,],Clone_cellularities[j,])){
        M[j,i]<-1
      }
    }
  }
  ###Re-ordering so that you go from root to leaves
  print("M:")
  print(M)
  
  S<-apply(M,1,sum)
  print("S:")
  print(S)
  Clone_cellularities<-Clone_cellularities[order(S,decreasing = F),]
  S<-S[order(S,decreasing = F)]
  
  connexion_list<-matrix(0,nrow=(2*nr-1),ncol = (2*nr-1))

  if(is.matrix(Clone_cellularities)){
    ###adding non mutated clones
    connexion_list<-cbind(connexion_list,rbind(Clone_cellularities,matrix(0,nrow=nr-1,ncol=nc)))
  }
  else{
    connexion_list<-cbind(connexion_list,c(Clone_cellularities,rep(0,times=nc-1)))
  }
  start<-min(which(S>0))
  for(i in (start-1):nr){
    print(paste("i=",i))
    remove<-0
    if(i==(start-1)){
      connexion_list<-list(list(connexion_list,1))
    }
    else{
      for(k in 1:length(connexion_list)){
        if(is.matrix(Clone_cellularities)){
          print("adding")
          t<-add_leaf_list(leaf = Clone_cellularities[i,],connexion_list = connexion_list[[k]],
                           timepoints = timepoints,d = nr,selector_position = i)
          print(t)
        }
        else{          
          t<-add_leaf_list(leaf = Clone_cellularities[i],connexion_list = connexion_list[[k]],
                           timepoints = timepoints,d=nr,selector_position = i)
        }
        if(!is.list(t)){
          if(remove==0){
            remove<-k
          }
          else{
            remove<-c(remove,k)
          }
        }

        else if(length(t)==2 & !is.list(t[[1]])){
          connexion_list[[k]]<-t
        }
        else{          
          connexion_list[[k]]<-t[[1]]
          connexion_list<-c(connexion_list,t[-1])
        }
      }
    }
    if(sum(remove)>0){
      connexion_list<-connexion_list[-remove]
    }
  }
  return(connexion_list)
}

#' Phylogenetic tree leaf
#'
#' Adds a leaf to an already built tree. Output is a list of all possibilities.
#' @param leaf A vector of cellularities (ranging from 0 to 1)
#' @param connexion_list List containing 1. An interaction matrix concatenated with the cellularity of each cluster (one line per cluster)
#' @param timepoints A numeric vector giving the spatial and/or temporal distribution of the samples
#' @param d The initial number of clusters
#' @param selector_position The row of the studied leaf in the data frame.
#' @keywords Clonal inference phylogeny

add_leaf_list<-function(leaf,connexion_list,timepoints,d,selector_position){
  print("Connexion list:")
  print(connexion_list)
  Exclude<-apply(X = connexion_list[[1]][,1:(2*d-1)],MARGIN = 1,FUN = sum)==2
  
  if(dim(connexion_list[[1]])[2]-2*d>0){
    Inclusion<-apply(X = connexion_list[[1]][,(2*d):(dim(connexion_list[[1]])[2])],MARGIN = 1,FUN = is_included,leaf)
  }
  else{
    Inclusion<-sapply(X = connexion_list[[1]][,(2*d)],FUN = is_included,leaf)
  }
  Inclusion[selector_position]<-F

  S<-sum(Inclusion & !Exclude)
  if(S==0){
    return(NA)
  }
  else if(S>=1){
    w<-which(Inclusion & !Exclude)
    result<-list()
    t<-min(timepoints[leaf>0])
    if(dim(connexion_list[[1]])[2]-2*d>0){
    size_at_t<-apply(X = connexion_list[[1]][,(2*d):(dim(connexion_list[[1]])[2])],MARGIN = 1,FUN = sum)
    }
    else{
      size_at_t<-sapply(X = connexion_list[[1]][,(2*d)],FUN = sum)
    }
    prob_at_t<-size_at_t/(sum(size_at_t[w]))
    if(length(w)==1){
        spare<-connexion_list[[1]]
        spare[w,c(selector_position,selector_position+d-1)]<-1
        spare[selector_position+d-1,(2*d):(dim(connexion_list[[1]])[2])]<-spare[w,(2*d):(dim(connexion_list[[1]])[2])]-leaf
        prob<-prob_at_t[w]*connexion_list[[2]]
        result<-list(list(spare,prob))
    }
    else{
      for(i in w){
        spare<-connexion_list[[1]]
        print(spare)
        print(i)
        print(w)
        spare[i,c(selector_position,selector_position+d-1)]<-1
        spare[selector_position+d-1,(2*d):(dim(connexion_list[[1]])[2])]<-spare[i,(2*d):(dim(connexion_list[[1]])[2])]-leaf
        prob<-prob_at_t[i]*connexion_list[[2]]
        result<-c(result,list(list(spare,prob)))
      }
    }
  }
  return(result)
}

#' Length 
#'
#' Computes the length from the clone on the n-th row of the matrix, to the most ancestral clone
#' @param matrix The interaction matrix of the tree (1 on the i-th row j-th column means "clone j is the progeny of clone i")
#' @param n Index of the clone in the matrix
#' @keywords Clonal inference phylogeny

longueur<-function(matrix,n){
  if(sum(matrix[,n])==0){
    return(0)
  }
  else{
    return(longueur(matrix,which(matrix[,n]==1))+1)
  }
}

#' Plots multiple trees 
#'
#' Plots all trees created by the function Tree_generation. The red line means that mutations occured.
#' @param result_list List of lists (tree generated and the probability associated with each tree)
#' @param d Initial number of clones
#' @param cex Coefficient of expansion for the texts in phylogenetic tree plots. Default is 0.8
#' @keywords Clonal inference phylogeny

multiplot_trees<-function(result_list,d,cex=0.8){
  if(length(result_list)%%2==0){
    L<-length(result_list)%/%2
  }
  else{
    L<-length(result_list)%/%2+1
  }
  if(L>1){
    op<-par(mfrow = c(2,L),mar = rep(2, 4))
  }
  for(i in 1:length(result_list)){
    manual_plot_trees(result_list[[i]][[1]],d,cex,result_list[[i]][[2]])
  }
}

#' Plot tree 
#'
#' Creates a visual output for the phylogeny created by Tree_generation()
#' @param connexion_list Data frame of the concatenation of the interaction matrix and the cellularity of each clone at different time points.
#' @param d Initial number of clones
#' @param cex Coefficient of expansion for the texts in phylogenetic tree plots. Default is 0.8
#' @param p Probability of a tree
#' @keywords Clonal inference phylogeny
manual_plot_trees<-function(connexion_list,d,cex=0.8,p){
  s<-dim(connexion_list[[1]][2])
  V<-numeric()
  X<-numeric()
  for(i in 1:(2*d-1)){
    V[i]<-longueur(connexion_list[1:(2*d-1),1:(2*d-1)],i)
    X[i]<-find_x_position(connexion_list[1:(2*d-1),1:(2*d-1)],i,d)
  }
  Y<-1-V/(max(V))
  plot(x=X,y=Y,xlim=c(-1,1),ylim=c(min(Y),1),cex=0, axes = F,xlab='',ylab='',main = paste('p = ',round(p,digits=5)))
  for(i in which(apply(X = connexion_list[1:(2*d-1),1:(2*d-1)],MARGIN = 1,FUN = sum)==2)){
    segments(x0=X[i],x1=X[i],y0=Y[i],y1=Y[i]-1/(max(V)))
    segments(x0=X[which(connexion_list[i,]==1)[1]],x1=X[i],y0=Y[i]-1/(max(V)),y1=Y[i]-1/(max(V)),col='red')
    segments(x0=X[i],x1=X[which(connexion_list[i,]==1)[2]],y0=Y[i]-1/(max(V)),y1=Y[i]-1/(max(V))) 
  }
  if(2*d<dim(connexion_list)[2]){
    LABELS<-apply(X = apply(X = connexion_list[1:(2*d-1),(2*d):(dim(connexion_list)[2])],2,FUN = round,digit=3),1,paste,collapse='\n')
    text(x=X,y=Y,labels = LABELS,pos = 3,cex = cex)
  }
  else{
    LABELS<-sapply(X = connexion_list[1:(2*d-1),(2*d)],FUN = round,digit=3)
    text(x=X,y=Y,labels = LABELS,pos = 3,cex = cex)
  }
}

#' Graphic position 
#'
#' Computes the position of a node on the graph, based on the interaction matrix.
#' @param matrix The interaction matrix of the tree (1 on the i-th row j-th column means "clone j is the progeny of clone i")
#' @param d Initial number of clones
#' @param n Index of the clone of interest in the matrix
#' @keywords Clonal inference phylogeny

find_x_position<-function(matrix,n,d){
  if(sum(matrix[,n])==0){
    return(0)
  }
  else if(n<=d){
    return(find_x_position(matrix,which(matrix[,n]==1),d)-1/2**longueur(matrix,n))
  }
  else{
    return(find_x_position(matrix,which(matrix[,n]==1),d)+1/2**longueur(matrix,n))
  }
}

#' Group theory
#'
#' Clone2 is included in Clone1 if all values of Clone2 are lower or equal to the ones in Clone1 at the same position. Returns TRUE is Clone2 is included in Clone1.
#' @param Clone1 Numeric vector, representing the cellularity of Clone1 in different samples
#' @param Clone2 Numeric vector, representing the cellularity of Clone2 in different samples
#' @keywords Clonal inference phylogeny

is_included<-function(Clone1,Clone2){#Returns True if Clone2 is included in Clone1
  for(i in 1:length(Clone1)){
    if(Clone1[i]<Clone2[i]){
      return(F)
    }
  }
  return(T)
}

#' Probability
#'
#' Returns dataframe with all informations about mutation (Number of copies, Cellularity, etc.) and probability to belong to a clone
#' @param SNV_list A list of dataframes (one for each sample), with as columns : (for the first column of the first sample the name of the sample), 
#' the chromosome "Chr",the position of the mutation "Start", the number of reads supporting the mutation "Alt", the depth of coverage at this locus "Depth", 
#' and if the output from FREEC for the samples are not associated, the genotype "Genotype".  
#' @param clone_prevalence List of numeric vectors giving the cellular prevalence of each clone in each sample
#' @param contamination Numeric vector giving the contamination by normal cells
#' @param clone_weights Numeric vector giving the proportion of mutations in each clone
#' @keywords Clonal inference phylogeny
#' @export
#' @examples 
#' SNVs<-QuantumCat(number_of_clones = 2,number_of_mutations = 50,number_of_samples = 1)
#' Probability.to.belong.to.clone(SNV_list=SNVs,
#' clone_prevalence=list(c(0.5,1),c(0.5,1)),contamination=c(0,0))

Probability.to.belong.to.clone<-function(SNV_list,clone_prevalence,contamination,clone_weights=NULL){
  if(is.null(clone_weights)){
    clone_weights<-rep(1/length(SNV_list),times = length(SNV_list))
  }
  if(is.null(SNV_list[[1]]$NC)){
    Schrod<-Patient_schrodinger_cellularities(SNV_list = SNV_list,Genotype_provided = T,contamination = contamination)
    result<-Schrod[[1]]
    if(length(Schrod)>1){
      for(i in 2:length(Schrod)){
        result<-cbind(result,Schrod[[i]]$Cell,Schrod[[i]][,c('Cellularity','Genotype',"Alt","Depth","NC")])
      }
    }
    result<-cbind(result,proba = eval.fik(Schrod = Schrod,centers = clone_prevalence,alpha= rep(1,times=length(Schrod[[1]]$NC)),
                                   weights= clone_weights,keep.all.poss = TRUE,
                                   adj.factor = Compute.adj.fact(Schrod = Schrod,contamination = contamination)))
  }
  else{
    result<-SNV_list[[1]]
    if(length(SNV_list)>1){
      for(i in 2:length(SNV_list)){
        result<-cbind(result,SNV_list[[i]]$Cell,SNV_list[[i]][,c('Cellularity','Genotype',"Alt","Depth","NC")])
      }
    }
    if(is.null(SNV_list[[1]]$alpha)){
      for(a in 1:length(SNV_list)){
        SNV_list[[a]]<-rep(1,times=length(SNV_list[[a]]$NC))
      }
    }
    adj.fact<-Compute.adj.fact(SNV_list,contamination)
    result<-eval.fik(Schrod = SNV_list,centers = clone_prevalence,weights =clone_weights,
                     keep.all.poss = TRUE,adj.factor = adj.fact)
  }
  return(result)
}

#'Plot with margin densities
#'
#'Adapted from http://stackoverflow.com/questions/11883844/inserting-a-table-under-the-legend-in-a-ggplot2-histogram
#'Uses gridExtra package
#' @param QClone_Output Output from QuantumClone algorithm
#' @keywords Plot Densities
#' @export
#' @examples 
#' require(ggplot2)
#' require(gridExtra)
#' print("Creating data formated as QuantumClone analysis")
#' print("Cluster 1: coordinates (0.8,0.7)")
#' print("Cluster 2: coordinates (0.3,0.4)")
#' QC_out<-list(cluster=rep(c(1,2),each=50),
#'        filtered.data=list(data.frame(id=1:100,Cellularity=c(rnorm(n=50,mean=0.8,sd=0.1),
#'        rnorm(n=50,0.3,0.08))),
#'        data.frame(id=1:100,Cellularity=c(rnorm(n=50,mean=0.700,
#'        sd=0.095),rnorm(n=50,mean=0.400,sd=0.085)))))
#' plot_with_margins_densities(QC_out)
#' 
#' 
#' 
plot_with_margins_densities<-function(QClone_Output){
  sq<-floor(sqrt(max(QClone_Output$cluster)))+1
  
  
  main<-ggplot2::qplot(x=QClone_Output$filtered.data[[1]]$Cellularity,y=QClone_Output$filtered.data[[2]]$Cellularity,color=as.character(QClone_Output$cluster),
              xlab="Cellularity diag",ylab="Cellulariy relapse",xlim=c(0,1),ylim=c(0,1))+ggplot2::theme_bw()+ggplot2::scale_color_discrete(guide = ggplot2::guide_legend(title="Cluster",ncol=sq))


  top<-ggplot2::ggplot(QClone_Output$filtered.data[[1]],
                       ggplot2::aes_string("Cellularity"))+ggplot2::geom_density(alpha=.5)+ggplot2::theme_bw()+ggplot2::theme(legend.position="none",
                                                                                                                     axis.title.x=ggplot2::element_blank())
  
  right<-ggplot2::ggplot(QClone_Output$filtered.data[[2]],
                         ggplot2::aes_string("Cellularity"))+ggplot2::geom_density(alpha=.5)+ggplot2::coord_flip()+ggplot2::theme_bw()+ggplot2::theme(legend.position="none",
                                                                                                                                axis.title.y=ggplot2::element_blank())
  
  tmp <- ggplot2::ggplot_gtable(ggplot2::ggplot_build(main)) 
  leg <- which(sapply(tmp$grobs, function(x) x$name) == "guide-box") 
  legend <- tmp$grobs[[leg]] 
  
  return(gridExtra::grid.arrange(top,
               legend,
               main+ggplot2::theme(legend.position="none"),
               right,
               ncol=2, nrow=2, widths=c(4, 1), heights=c(1, 4)))
  
}
