% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MultivariateMoments.R
\name{StructuredMoments}
\alias{StructuredMoments}
\alias{M2.struct}
\alias{M3.struct}
\alias{M4.struct}
\title{Functions for calculating structured comoments of financial time series}
\usage{
M2.struct(R, struct = c("Indep", "IndepId", "observedfactor", "CC"), f = NULL)

M3.struct(
  R,
  struct = c("Indep", "IndepId", "observedfactor", "CC", "latent1factor", "CS"),
  f = NULL,
  unbiasedMarg = FALSE,
  as.mat = TRUE
)

M4.struct(
  R,
  struct = c("Indep", "IndepId", "observedfactor", "CC"),
  f = NULL,
  as.mat = TRUE
)
}
\arguments{
\item{R}{an xts, vector, matrix, data frame, timeSeries or zoo object of
asset returns}

\item{struct}{string containing the preferred method. See Details.}

\item{f}{vector or matrix with observations of the factor, to be used with 'observedfactor'. See Details.}

\item{unbiasedMarg}{TRUE/FALSE whether to use a correction to have an unbiased
estimator for the marginal skewness values, in case of 'Indep' or 'IndepId', default FALSE}

\item{as.mat}{TRUE/FALSE whether to return the full moment matrix or only
the vector with the unique elements (the latter is advised for speed), default
TRUE}
}
\description{
calculates covariance, coskewness and cokurtosis matrices as structured estimators
}
\details{
The coskewness and cokurtosis matrices are defined as the matrices of dimension 
p x p^2 and p x p^3 containing the third and fourth order central moments. They
are useful for measuring nonlinear dependence between different assets of the 
portfolio and computing modified VaR and modified ES of a portfolio.

Structured estimation is based on the assumption that the underlying data-generating
process is known, or at least resembles the assumption. The first four structured estimators correspond to the models 'independent marginals', 
'independent and identical marginals', 'observed multi-factor model' and 'constant correlation'. 
Coskewness estimation includes an additional model based on the latent 1-factor model
proposed in Simaan (1993).

The constant correlation and 1-factor coskewness and cokurtosis matrices can be found in 
Martellini and Ziemann (2010). If f is a matrix containing multiple factors, 
then the multi-factor model of Boudt, Lu and Peeters (2915) is used. For information
about the other structured matrices, we refer to Boudt, Cornilly and Verdonck (2017)
}
\examples{

data(edhec)

# structured estimation with constant correlation model
# 'as.mat = F' would speed up calculations in higher dimensions
sigma <- M2.struct(edhec, "CC")
m3 <- M3.struct(edhec, "CC")
m4 <- M4.struct(edhec, "CC")

# compute equal-weighted portfolio modified ES
mu <- colMeans(edhec)
p <- length(mu)
ES(p = 0.95, portfolio_method = "component", weights = rep(1 / p, p), mu = mu, 
    sigma = sigma, m3 = m3, m4 = m4)

# compare to sample method
sigma <- cov(edhec)
m3 <- M3.MM(edhec)
m4 <- M4.MM(edhec)
ES(p = 0.95, portfolio_method = "component", weights = rep(1 / p, p), mu = mu, 
    sigma = sigma, m3 = m3, m4 = m4)

}
\references{
Boudt, Kris, Lu, Wanbo and Peeters, Benedict. 2015. Higher order comoments of multifactor 
models and asset allocation. Finance Research Letters, 13, 225-233.

Boudt, Kris, Brian G. Peterson, and Christophe Croux. 2008.
Estimation and Decomposition of Downside Risk for Portfolios with Non-Normal
Returns. Journal of Risk. Winter.

Boudt, Kris, Cornilly, Dries and Verdonck, Tim. 2017. A Coskewness Shrinkage 
Approach for Estimating the Skewness of Linear Combinations of Random Variables. 
Submitted. Available at SSRN: https://ssrn.com/abstract=2839781

Ledoit, Olivier and Wolf, Michael. 2003. Improved estimation of the covariance matrix 
of stock returns with an application to portfolio selection. Journal of empirical 
finance, 10(5), 603-621.

Martellini, Lionel and Ziemann, V\"olker. 2010. Improved estimates of higher-order 
comoments and implications for portfolio selection. Review of Financial 
Studies, 23(4), 1467-1502.

Simaan, Yusif. 1993. Portfolio selection and asset pricing: three-parameter framework. 
Management Science, 39(5), 68-577.
}
\seealso{
\code{\link{CoMoments}} \cr \code{\link{ShrinkageMoments}} \cr \code{\link{EWMAMoments}} \cr \code{\link{MCA}}
}
\author{
Dries Cornilly
}
\concept{co-moments}
\concept{moments}
