% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FitMCMC.R
\name{FitMCMC}
\alias{FitMCMC}
\title{MCMC/Bayesian estimation.}
\usage{
FitMCMC(spec, data, ctr = list())
}
\arguments{
\item{spec}{Model specification of class \code{MSGARCH_SPEC}
created with \code{\link{CreateSpec}}.}

\item{data}{Vector (of size T) of observations.}

\item{ctr}{A list of control parameters:
\itemize{
\item \code{par0}: Vector (of size d) where d must have
the same length as the default parameters of the specification.
It is the starting value for the chain (if empty the
the method automatically set starting parameters; see *Details*).
\item \code{n.burn} (integer >= 0): Number of discarded draws.
(Default: \code{n.burn = 5000L})
\item \code{n.mcmc} (integer > 0): Number of draws.
(Default: \code{n.mcmc = 10000L})
\item \code{n.thin} (integer > 0): Thinning factor (every \code{N.thin}
draws are kept). (Default: \code{n.thin = 10L})
\item \code{SamplerFUN}: Custom MCMC sampler (see *Details*).
}}
}
\value{
A list of class \code{MSGARCH_MCMC_FIT} with the following elements:
\itemize{
\item \code{par}: The MCMC chain (matrix from the \R package
\code{coda} (Plummer et al., 2006) of size \code{N.mcmc} / \code{N.thin} x d).
\item \code{accept}: Acceptation rate of the sampler.
\item \code{spec}:  Model specification of class \code{MSGARCH_SPEC}
created with \code{\link{CreateSpec}}.
\item \code{data}:  Vector (of size T) of observations.
\item \code{ctr}: \code{list} of the control used for the fit.
}
The \code{MSGARCH_MCMC_FIT} with the following methods:
\itemize{
\item \code{\link{AIC}}: Compute Akaike information criterion (AIC).
\item \code{\link{BIC}}: Compute Bayesian information criterion (BIC).
\item \code{\link{DIC}}: Compute Deviance Information Criterion (DIC).
\item \code{\link{Volatility}}: In-sample conditional volatility filterting of the overall process.
\item \code{\link{Forecast}}: Forecast of the conditional volatility of the overall process.
\item \code{\link{UncVol}}: Unconditional volatility in each regime and the overall process.
\item \code{\link{Pred}}: Predictive method.
\item \code{\link{PIT}}: Probability integral transform.
\item \code{\link{Risk}}: Value-at-Risk And Expected-Shortfall methods.
\item \code{\link{Sim}}: Simulation method.
\item \code{\link{State}}: State probabilities methods.
\item \code{\link{ExtractStateFit}}: Single-regime model extractor.
\item \code{summary}: Summary of the fit.
}
}
\description{
Method that performs MCMC/Bayesian estimation of
a \code{MSGARCH_SPEC} object on a set of observations.
}
\details{
The total number of draws is equal to \code{n.mcmc / n.thin}.
The MCMC/Bayesian estimation relies on the package \pkg{adaptMCMC} (Andreas, 2012) which
implements the adaptive sampler of Vihola (2012).
Starting values when \code{par0} is not provided are chosen automatically
before sampling (see Ardia et al. (2016) for more details).\cr
\code{SamplerFUN} allows for a custom sampler to be used. The function
must take the form: \cr \code{function(f_posterior, data, spec, par0, ctr)}, \cr
where  \code{f_posterior} is the function to optimize, \code{data} is
the data, \code{spec} is the specification,
\code{par0} are the starting parameters, and \code{ctr} are the control
parameters. The inputs \code{spec} and \code{data},
must be passed as inputs in the sampler (see *Examples*).
The custom sampler must output a matrix containing the MCMC chain. \cr
}
\examples{
# load data
data("SMI", package = "MSGARCH")

# create model specification
# MS(2)-GARCH(1,1)-Normal (default)
spec <- CreateSpec()

# fit the model on the data by MCMC
set.seed(123)
fit <- FitMCMC(spec = spec, data = SMI, ctr = list(n.burn = 500L, n.mcmc = 500L, n.thin = 1L,
                                              do.adapt = TRUE, acc.rate = 0.4))
summary(fit)

# custom sampler example
\dontrun{
library("mcmc")
f_MCMC <- function(f_posterior, data, spec, par0, ctr){
  par <- mcmc::metrop(f_posterior, initial = par0, nbatch = ctr$n.mcmc + ctr$n.burn,
                        data = data, spec = spec)$batch
  return(par)
}

set.seed(123)
fit <- FitMCMC(spec, data = SMI, ctr  = list(SamplerFun = f_MCMC,
                                             n.burn = 500L, n.mcmc = 500L))
summary(fit)
}
}
\references{
Andreas, S. (2012).
\code{adaptMCMC}: Implementation of a generic adaptive Monte Carlo Markov chain sampler.
\url{https://cran.r-project.org/package=adaptMCMC}

Ardia, D. Bluteau, K. Boudt, K. Catania, L. & Trottier, D.-A. (2016).
Markov-switching GARCH models in \R: The MSGARCH package.
\url{https://ssrn.com/abstract=2845809}

MacDonald, I.L., Zucchini, W. (1997).
Hidden Markov and other models for discrete-valued time series.
\emph{CRC press}.

Plummer, M. Best, N. Cowles, K. & Vines, K. (2006).
\code{coda}: Convergence diagnosis and output analysis for MCMC.
\emph{R News}, 6, 7-11.
\url{https://cran.r-project.org/package=coda}

Vihola, M. (2012).
Robust adaptive Metropolis algorithm with coerced acceptance rate.
\emph{Statistics and Computing}, 22, 997-1008.
}
