\name{MARSSboot}
\alias{MARSSboot}
\title{ Bootstrap MARSS Parameter Estimates }
\description{
  Creates bootstrap parameter estimates and simulated (or bootstrapped) data (if appropriate).  This is a base function in the \code{\link{MARSS-package}}.
}
\usage{
MARSSboot(MLEobj, nboot = 1000, 
  output = "parameters", sim = "parametric", 
  param.gen = "KalmanEM", control = NULL, silent = FALSE)
}

\arguments{
  \item{MLEobj}{ An object of class \code{\link{marssMLE}}. Must have a \code{$par} element containing MLE parameter estimates. }
  \item{nboot}{ Number of bootstraps to perform. }
  \item{output}{ Output to be returned: "data", "parameters" or "all".} 
  \item{sim}{ Type of bootstrap: "parametric" or "innovations". See Details. }
  \item{param.gen}{ Parameter generation method: "hessian" or "KalmanEM". } 
  \item{control}{ The options in \code{MLEobj$control} are used by default. If supplied here, must contain all of the following:
    \describe{
      \item{\code{max.iter}}{ Maximum number of EM iterations. } 
      \item{\code{tol}}{ Optional tolerance for log-likelihood change.  If log-likelihood decreases less than this amount relative to the previous iteration, the EM algorithm exits. } 
      \item{\code{allow.degen}}{ Whether to try setting Q or R elements to zero if they appear to be going to zero. }  
    } 
  }
  \item{silent}{ Suppresses printing of progress bar. }
}

\details{
Approximate confidence intervals (CIs) on the model parameters can be calculated by numerically estimating the Hessian matrix (the matrix of partial 2nd derivatives of the parameter estimates).  The Hessian CIs are based on the asymptotic normality of ML estimates under a large-sample approximation. CIs that are not based on asymptotic theory can be calculated using parametric and non-parametric bootstrapping.

Stoffer and Wall (1991) present an algorithm for generating CIs via a non-parametric bootstrap for state-space models (\code{sim = "innovations"}).  The basic idea is that the Kalman filter can be used to generate estimates of the residuals of the model fit.  These residuals are then standardized and resampled and used to generate bootstrapped data using the MARSS model and its maximum-likelihood parameter estimates.  One of the limitations of the Stoffer and Wall algorithm is that it cannot be used when there are missing data, unless all data at time t are missing.  An alternative approach is a parametric bootstrap (\code{sim = "parametric"}), in which the ML parameter estimates are used to produce bootstrapped data directly from the state-space model.
}
\value{
  A list with the following components:
  \item{boot.params}{ Matrix (number of params x nboot) of parameter estimates from the bootstrap. }
  \item{boot.data}{ Array (n x t x nboot) of simulated (or bootstrapped) data (if requested and appropriate). }
  \item{model}{ The \code{\link{marssm}} object that was passed in via \code{MLEobj$model}. }
  \item{nboot}{ Number of bootstraps performed. }
  \item{output}{ Type of output returned.} 
  \item{sim}{ Type of bootstrap. }
  \item{param.gen}{ Parameter generation method: "hessian" or "KalmanEM".  } 
}
\references{ 
Holmes, E. E. and E. J. Ward. 2010. Analysis of multivariate time-series using the MARSS package. NOAA Fisheries, Northwest Fisheries Science
Center, 2725 Montlake Blvd E., Seattle, WA 98112; this is the user guide accessed via \code{RShowDoc("UserGuide",package="MARSS")}

Stoffer, D. S., and K. D. Wall. 1991. Bootstrapping state-space models: Gaussian maximum likelihood estimation and the Kalman filter. Journal of the American Statistical Association 86:1024-1033.

Cavanaugh, J. E., and R. H. Shumway. 1997. A bootstrap variant of AIC for state-space model selection. Statistica Sinica 7:473-496.
 }
\author{ 
  Eli Holmes and Eric Ward, NOAA, Seattle, USA.

  eli(dot)holmes(at)noaa(dot)gov, eric(dot)ward(at)noaa(dot)gov
}
\seealso{ 
  \code{\link{marssMLE}}  \code{\link{marssm}}  \code{\link{MARSSaic}}  
}
\examples{
  #nboot is set low in these examples in order to run quickly
  #normally nboot would be >1000 at least
  dat = t(harborSealnomiss)
  dat = dat[2:4,]
  #normally one would not use the delta logLik test for convergence (abstol)
  #but used here with big abstol and minit to speed up the example
  kem = MARSS(dat, model=list(U="equal",Q="diagonal and equal"), 
     control=list(minit=40,abstol=0.1))
  hess.list = MARSSboot(kem, param.gen="hessian", nboot=5)
  # (no missing values)
  boot.list = MARSSboot(kem, output="all", sim="innovations", nboot=5)

  # Bootstrap CIs for data with missing values
  dat = t(harborSealWA)
  dat = dat[2:4,]
  kem = MARSS(dat, model=list(Q="diagonal and equal"), 
     control=list(minit=40,abstol=0.1))
  boot.list = MARSSboot(kem, output="all", sim="parametric", nboot=5)
}



