#' Orthogonal Latin Hypercube Design
#'
#' \code{OLHD.B2001} returns a \code{n} by \code{k} orthogonal Latin hypercube design generated by the construction method of Butler (2001)
#'
#' @param n An odd prime number that is greater than or equal to 3.
#' @param k A positive integer that is smaller than or equal to n-1.
#'
#' @return If all inputs are logical, then the output will be a \code{n} by \code{k} orthogonal LHD.
#'
#' @references Butler, N.A. (2001) Optimal and orthogonal Latin hypercube designs for computer experiments. \emph{Biometrika}, \strong{88}(3), 847-857.
#'
#' @examples
#' #create an orthogonal LHD with n=11 and k=5
#' OLHD.B2001(n=11,k=5)
#'
#' #create an orthogonal LHD with n=7 and k=6
#' OLHD.B2001(n=7,k=6)
#'
#' @export

OLHD.B2001=function(n,k){

  if(k>=n){
    stop("k must be smaller than or equal to n-1")
  }

  if(n<3){
    stop("n must be greater than or equal to 3")
  }

  if(numbers::isPrime(n)==FALSE | n%%2!=1){
    stop("n must be an odd prime number")
  }

  n0=(n-1)/2

  if(k<=n0){
    g=sample(seq(1,n0),k)    #generator g

    W=matrix(0,nrow=n,ncol=k)

    for (i in 1:n) {
      for (j in 1:k) {

        if(n%%4==1){
          W[i,j]=(i*g[j]+(n-1)/4)%%n
        }

        if(n%%4==3){
          W[i,j]=(i*g[j]+(3*n-1)/4)%%n
        }

      }
    }

    X=WT(W)
  }

  if(k>n0){

    #X0 starts
    g0=seq(1,n0)    #generator g0

    W0=matrix(0,nrow=n,ncol=n0)

    for (i in 1:n) {
      for (j in 1:n0) {

        if(n%%4==1){
          W0[i,j]=(i*g0[j]+(n-1)/4)%%n
        }

        if(n%%4==3){
          W0[i,j]=(i*g0[j]+(3*n-1)/4)%%n
        }

      }
    }

    X0=WT(W0)
    #X0 ends

    #X1 starts
    r=k-n0

    g1=sample(seq(1,n0),r)    #generator g1

    W1=matrix(0,nrow=n,ncol=r)

    for (i in 1:n) {
      for (j in 1:r) {

          W1[i,j]=(i*g1[j])%%n

      }
    }

    X1=WT(W1)


    #X1 ends
    X=cbind(X0,X1)
  }

  X
}
