#' Leave-one-out Method
#'
#' \code{LOO} returns a maximin distance LHD matrix generated by Leave-one-out method, along with its maximum minimum L_1 distance
#'
#' @param n A positive integer, which stands for the number of rows (or run size).
#' @param k A positive integer, which stands for the number of columns (or factor size). \code{k} must be smaller than \code{n+1}.
#' @param h A vector whose length is \code{k}, with its elements that are smaller than and coprime to \code{n+1}. The default is set to be a random sample of \code{k} elements between 1 and \code{n}.
#' @param method A distance measure method. The default setting is "manhattan", and it could be one of the following: "euclidean", "maximum", "manhattan", "canberra", "binary" or "minkowski". Any unambiguous substring can be given.
#'
#' @return If all inputs are logical, then the output will be a list, which contains a \code{n} by \code{k} LHD and its maximum minimum L_1 distance. This is the construction method in section 2.2 of the reference, and it is particularly useful when \code{n+1} is prime and \code{k} equals \code{n}.
#'
#' @references Wang, L., Xiao, Q., and Xu, H. (2018)  Optimal maximin $L_{1}$-distance Latin hypercube designs based on good lattice point designs. \emph{The Annals of Statistics}, \strong{46}(6B), 3741-3766.
#'
#' @examples
#' #generate a 10 by 10 maximin distance LHD with default setting
#' try=LOO(n=10,k=10)
#' try
#' phi_p(try[[1]])   #calculate the phi_p of "try".
#'
#' #Another example
#' #generate a 12 by 12 maximin distance LHD with given h vector
#' try2=LOO(n=12,k=12,h=1:12)
#' try2
#' phi_p(try2[[1]])   #calculate the phi_p of "try2".
#'
#' @export

LOO=function(n,k,h=sample(seq(from=1,to=n),k),method="manhattan"){
  ### special case of n \times (k<=n) design when n+1 is prime
  ### search over 0 to +n-1 mod n of the basedesign given the run size: n
  #This is the Leave-one-out method
  #k: number of factors
  #h: chosen vector that is coprime to n+1
  #method: the distance method, with default setting is "manhattan".

  if(k>=(n+1)){
    stop("k must be smaller than (n+1)")
  }

  basedesign=GLP(n=n+1,k=k,h=h)    #starting design: D_b with b=0

  resultdesign=WT(basedesign,baseline=0)[-(n+1),]   #E_b with b=0

  minvalue=min(stats::dist(resultdesign, method = method))  #current minimum distance

  for (c in 1:n)      #for b=1, ... , (n+1)-1
  {
    newdesign = (basedesign + c) %% (n+1)    #new D_b

    newresult = WT(newdesign,baseline=0)     #new E_b

    cutv = newresult[(n+1),1]   #cut value: this is value of the constant row

    newresult=newresult[-(n+1),] #get rid of the last row since it is constant

    for(i in 1:n){    #bring one number down for remaining elements > cutv
      for(j in 1:k){

        if(newresult[i,j] > cutv){
          newresult[i,j] = newresult[i,j] - 1
        }

      }
    }

    newminvalue = min(stats::dist(newresult, method = method))

    if (newminvalue > minvalue)      #updating minimum distance
    {
      resultdesign = newresult
      minvalue = newminvalue
    }

  }

  list(resultdesign, minvalue)
}
