

#' Dominance `(n,k)` rule for magnitude tables
#' 
#' Supports application of multiple values for `n` and `k`. The function works on magnitude tables containing negative cell values by calculating contribution based on absolute values.
#' 
#' Currently the implementation only supports a single numeric variable.
#' @param data the dataset
#' @param x ModelMatrix generated by parent function
#' @param crossTable crossTable generated by parent function
#' @param numVar vector containing numeric values in the data set
#' @param n parameter `n` in dominance rule. 
#' @param k parameter `k` in dominance rule.
#' @param ... unused parameters
#' @return logical vector that is `TRUE` in positions corresponding to cells breaching the dominance rules.
#' @export
#'
#' @author Daniel Lupp 
#' 
DominanceRule <- function(data, x, crossTable, numVar, n, k, ...) {
  if (length(n) != length(k))
    stop("You must provide an equal number of inputs for n and k.")
  if (length(numVar) != 1)
    stop("You must provide exactly one numeric variable.")
  
  abs_num <- as.data.frame(as.matrix(crossprod(x, as.matrix(abs(data[, numVar, drop = FALSE])))))
  abs_cellvals <- abs(data[[numVar]])
  
  primary <- mapply(function (a,b) single_dominance_rule(x, abs_cellvals, abs_num, a,b),
                    n,k)

  apply(primary, 1, function (x) Reduce(`|`, x))
}

single_dominance_rule <- function(x, cellvals, num, n, k) {
  max_cont <- MaxContribution(x, cellvals, n = n)
  max_cont[is.na(max_cont)] <- 0
  as.vector(num > 0 & rowSums(max_cont) > num*k/100)
}
