% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.FluMoDL.R
\name{predict.FluMoDL}
\alias{predict.FluMoDL}
\title{Predict method for FluMoDL objects}
\usage{
\method{predict}{FluMoDL}(object, temp = NULL, proxyH1 = NULL,
  proxyH3 = NULL, proxyB = NULL, proxyRSV = NULL, se.fit = FALSE,
  byWeek = FALSE, ...)
}
\arguments{
\item{object}{A FluMoDL object}

\item{temp}{A vector of daily mean temperatures. See 'Details'.}

\item{proxyH1}{A vector of daily influenza A(H1N1)pdm09 incidence proxies. See 'Details'.}

\item{proxyH3}{A vector of daily influenza A(H3N2) incidence proxies. See 'Details'.}

\item{proxyB}{A vector of daily influenza B incidence proxies. See 'Details'.}

\item{proxyRSV}{An vector of daily RSV incidence proxies (used only if the FluMoDL
object includes an RSV term). See 'Details'.}

\item{se.fit}{Logical switch indicating if standard errors are required.
Requires \code{byWeek=FALSE}.}

\item{byWeek}{If \code{TRUE}, aggregate fitted estimates by week. Has priority
over argument \code{se.fit}. If both \code{se.fit} and \code{byWeek} are
\code{TRUE}, \code{se.fit} is set to \code{FALSE} and a warning is returned.}

\item{...}{Further arguments passed to or from other methods}
}
\value{
A vector of daily predicted deaths (corresponding to the rows in
    \code{object$data}). If \code{byWeek=TRUE}, the predictions are automatically
    aggregated by week (as per \code{object$data$yearweek}) and the vector contains
    the respective week (in YYYYWW format) as names.

    If \code{se.fit=TRUE}, a list
    is returned with elements \code{$fit} and \code{$se.fit} containing the
    (daily) predicted deaths and their associated log standard errors.

    Note that the first 30 elements (or first 5 elements if \code{byWeek=TRUE}) will be
    \code{NA} by default, as FluMoDL uses a maximum lag of 30 days.
}
\description{
Obtains predictions (predicted daily or weekly deaths) and optionally estimates
standard errors of those predictions
}
\details{
Arguments \code{temp}, \code{proxyH1}, \code{proxyH3}, \code{proxyB} and
    (if \code{hasRSV(object)} is \code{TRUE}) \code{proxyRSV} take a numeric vector
    as input, which is recycled to a length of \code{nrow(object$data)}. Alternatively
    they can take \code{NULL}, in which case the respective column of \code{object$data}
    is used as input. Argument \code{temp} can also take the string \code{"MMP"}, which
    is interpreted as the "Minimum Mortality Point", i.e. the temperature at which
    mortality is lowest (found in \code{object$MMP}).

    In this way, the \code{predict()} method can be flexibly used to calculate the
    predicted "baseline" mortality (by setting \code{temp="MMP"} and all incidence proxies
    to zero), the model-predicted mortality for the actual input (by leaving all input
    arguments to their default \code{NULL}), or predicted mortalities for any combination of
    temperature and incidence proxy inputs.
}
\examples{
data(greece) # Use example surveillance data from Greece
m <- with(greece, fitFluMoDL(deaths = daily$deaths,
    temp = daily$temp, dates = daily$date,
    proxyH1 = weekly$ILI * weekly$ppH1,
    proxyH3 = weekly$ILI * weekly$ppH3,
    proxyB = weekly$ILI * weekly$ppB,
    yearweek = weekly$yearweek))
m

# Calculate FluMoDL baseline
baseline <- predict(m, temp="MMP", proxyH1=0, proxyH3=0, proxyB=0, byWeek=TRUE)

# Calculate fitted predictions
fitted <- predict(m, byWeek=TRUE)

\donttest{
# Plot everything
plot(with(m$data, tapply(deaths, yearweek, sum)), type="l",
     xaxt="n", ylab="Weekly deaths", xlab="Time")
points(baseline, type="l", col="blue")
points(fitted, type="l", col="green")
legend("topleft", c("Actual", "Baseline", "Fitted"), lty="solid",
    col=c("black", "blue", "green"), bty="n")
}

}
