% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EllCopEst.R
\name{EllCopEst}
\alias{EllCopEst}
\title{Estimate the density generator of a (meta-)elliptical copula}
\usage{
EllCopEst(
  dataU,
  Sigma_m1,
  h,
  grid = seq(0, 10, by = 0.01),
  niter = 10,
  a = 1,
  Kernel = "epanechnikov",
  verbose = 1,
  startPoint = "identity",
  prenormalization = FALSE
)
}
\arguments{
\item{dataU}{the data matrix on the \eqn{[0,1]} scale.}

\item{Sigma_m1}{the inverse of the correlation matrix of the components of data}

\item{h}{bandwidth of the kernel for Liebscher's procedure}

\item{grid}{the grid at which the density generator is estimated.}

\item{niter}{the number of iterations}

\item{a}{tuning parameter to improve the performance at 0.
See Liebscher (2005), Example p.210}

\item{Kernel}{kernel used for the smoothing.
Possible choices are \code{gaussian}, \code{epanechnikov} and \code{triangular}.}

\item{verbose}{if 1, prints the progress of the iterations.
If 2, prints the normalization constants used at each iteration,
as computed by \code{\link{DensityGenerator.normalize}}.}

\item{startPoint}{is the given starting point of the procedure \itemize{
\item \code{startPoint = "gaussian"} for using the gaussian generator as starting point ;
\item \code{startPoint = "identity"} for a data-driven starting point ;
\item \code{startPoint = "A~Phi^{-1}"} for another data-driven starting point using
the Gaussian quantile function.
}}

\item{prenormalization}{if \code{TRUE}, the procedure will normalize the variables
at each iteration so that the variance is \eqn{1}.}
}
\value{
a list of two elements:
\itemize{
\item \code{g_d_norm}: the estimated elliptical copula generator at each point of the grid;
\item \code{list_path_gdh}: the list of estimated elliptical copula generator at each iteration.
}
}
\description{
This function estimates the density generator of a (meta-)elliptical copula
using the iterative procedure described in (Derumigny and Fermanian, 2022).
This iterative procedure consists in alternating a step of estimating the data
via Liebscher's procedure \code{\link[=EllDistrEst]{EllDistrEst()}} and estimating the quantile function
of the underlying elliptical distribution to transform the data back to the unit cube.
}
\examples{
\donttest{
# Simulation from a Gaussian copula
grid = seq(0,10,by = 0.01)
g_d = DensityGenerator.normalize(grid, grid_g = exp(-grid), d = 3)
n = 10
# To have a nice estimation, we suggest to use rather n=200
# (around 20s of computation time)
U = EllCopSim(n = n, d = 3, grid = grid, g_d = g_d)
result = EllCopEst(dataU = U, grid, Sigma_m1 = diag(3),
                   h = 0.1, a = 0.5)
plot(grid, g_d, type = "l", xlim = c(0,2))
lines(grid, result$g_d_norm, col = "red", xlim = c(0,2))

# Adding missing observations
n_NA = 2
U_NA = U
for (i in 1:n_NA){
  U_NA[sample.int(n,1), sample.int(3,1)] = NA
}
resultNA = EllCopEst(dataU = U_NA, grid, Sigma_m1 = diag(3),
                     h = 0.1, a = 0.5)
lines(grid, resultNA$g_d_norm, col = "blue", xlim = c(0,2))
}

}
\references{
Derumigny, A., & Fermanian, J. D. (2022).
Identifiability and estimation of meta-elliptical copula generators.
Journal of Multivariate Analysis, article 104962.
\doi{10.1016/j.jmva.2022.104962}.

Liebscher, E. (2005).
A semiparametric density estimator based on elliptical distributions.
Journal of Multivariate Analysis, 92(1), 205.
\doi{10.1016/j.jmva.2003.09.007}
}
\seealso{
\code{\link{EllDistrEst}} for the estimation of elliptical distributions,
\code{\link{EllCopSim}} for the simulation of elliptical copula samples,
\code{\link{EllCopLikelihood}} for the computation of the likelihood of a given generator,
\code{\link{DensityGenerator.normalize}} to compute the normalized version of a given generator.
}
