#' Cross-Validated Accuracy for Supervised Learning Model
#'
#' eztune.cv is a function that will return the cross-validated
#' accuracy for a model generated by the function eztune.
#' The function eztune can tune a model using resubstitution or
#' cross-validation. If the resubstitution is used to tune the model, the
#' accuracy obtained from the function is inflated. The function
#' eztune.cv will return a cross-validated accuracy for such a model.
#' @param x Matrix or data frame containing the dependent variables used
#' to create the model.
#' @param y Numeric vector of 0s and 1s for the response used to create
#' the model.
#' @param model Object generated with the function eztune.
#' @param fold Number of folds to use for n-fold cross validation.
#' @keywords adaboost, svm, gbm, tuning, cross-validation
#' @return Function returns a numeric value that represents the
#' cross-validated accuracy of the model.
#'
#' @examples
#' library(mlbench)
#' data(Glass)
#'
#' glass <- Glass[as.numeric(as.character(Glass$Type)) < 3, ]
#' glass <- glass[sample(1:nrow(glass), 80), ]
#' y <- ifelse(glass$Type == 1, 0, 1)
#' x <- glass[, 1:9]
#'
#' glass_svm <- eztune(x, y, type = "binary", method = "svm")
#' eztune.cv(x, y, glass_svm)
#'
#' @export
#'
eztune.cv <- function(x, y, model, fold = 10) {


  param <- switch(class(model$best.model)[1],
                  ada = c(model$iter, model$nu),
                  gbm = c(model$interaction.depth, model$n.trees,
                            model$shrinkage),
                  svm.formula = c(model$epsilon, model$cost))

  acc <- switch(class(model$best.model)[1],
                 ada = ada.b.cv(x, y, param, fold),
                 gbm = gbm.b.cv(x, y, param, fold),
                 svm.formula = svm.b.cv(x, y, param, fold))

  acc
}
