#' ctmaPRaw
#'
#' @description Converts empirical correlation matrices to pseudo raw data (i.e. random data, that perfectly reproduce the correlations)
#'
#' @param empCovMat empirical primary study covariance matrix
#' @param empNMat matrix of (possibly pairwise) N
#' @param empN N (in case of listwise N)
#' @param studyNumber internal number
#' @param empMeanVector vector of means for all variables, usually 0
#' @param empVarVector vector of variances for all variables, usually 1
#' @param activateRPB set TRUE to receive push messages with 'CoTiMA' notifications on your phone
#'
#' @importFrom RPushbullet pbPost
#' @importFrom stats rnorm
#' @importFrom MASS mvrnorm
#' @importFrom psych corr.test
#'
#'
ctmaPRaw <- function(empCovMat=NULL, empNMat=matrix(0,0,0), empN=NULL, studyNumber=NULL,
                          empMeanVector=NULL, empVarVector=NULL, activateRPB=FALSE)
{  # begin function definition (until end of file)

  if (is.null(empCovMat)) {
    if (activateRPB==TRUE) {RPushbullet::pbPost("note", paste0("CoTiMA (",Sys.time(),")" ), paste0(Sys.info()[[4]], "\n","Data processing stopped.\nYour attention is required."))}
    ErrorMsg <- "\nNo empirical covariance matrix provided for pseudo raw data generation! \nGood luck for the next try!"
    stop(ErrorMsg)
  }

  if (!(isSymmetric(empCovMat))) {
    if (activateRPB==TRUE) {RPushbullet::pbPost("note", paste0("CoTiMA (",Sys.time(),")" ), paste0(Sys.info()[[4]], "\n","Data processing stopped.\nYour attention is required."))}
    ErrorMsg <- "\nThe empirical covariance matrix provided is not symmetric! \nGood luck for the next try!"
    stop(ErrorMsg)
  }

  if (!(is.null(empNMat))) {
    if (!(isSymmetric(empNMat))) {
      if (activateRPB==TRUE) {RPushbullet::pbPost("note", paste0("CoTiMA (",Sys.time(),")" ), paste0(Sys.info()[[4]], "\n","Data processing stopped.\nYour attention is required."))}
      ErrorMsg <- "\nThe pairwise N matrix provided is not symmetrix! \nGood luck for the next try!"
      stop(ErrorMsg)
    }
  }

  if ( (is.null(empNMat) & is.null(empN)) ) {
    if (activateRPB==TRUE) {RPushbullet::pbPost("note", paste0("CoTiMA (",Sys.time(),")" ), paste0(Sys.info()[[4]], "\n","Data processing stopped.\nYour attention is required."))}
    ErrorMsg <- "\nEITHER a matrix with pairwise N OR an overall N has to be provided pseudo raw data generation! \nGood luck for the next try!"
    stop(ErrorMsg)
  }

  if ( (!(is.null(empMeanVector))) & (length(empMeanVector) != (dim(empCovMat)[1]) )  ){
    if (activateRPB==TRUE) {RPushbullet::pbPost("note", paste0("CoTiMA (",Sys.time(),")" ), paste0(Sys.info()[[4]], "\n","Data processing stopped.\nYour attention is required."))}
    ErrorMsg <- "\nThe number of means provided does not match the number of variables in the empirical covariance matrix! \nGood luck for the next try!"
    stop(ErrorMsg)
  }

  rowNACounter <- colNACounter <- c()
  for (i in 1:(dim(empCovMat)[1])) {
    rowNACounter[i] <- length(which(is.na(empCovMat[i, ]) == TRUE))
    colNACounter[i] <- length(which(is.na(empCovMat[ ,i]) == TRUE))
  }
  if (any(rowNACounter != colNACounter)) {
    if (activateRPB==TRUE) {RPushbullet::pbPost("note", paste0("CoTiMA (",Sys.time(),")" ), paste0(Sys.info()[[4]], "\n","Data processing stopped.\nYour attention is required."))}
    ErrorMsg <- "\nCurrently missing correlations can only be handled if a variable is entirely missing. \nThe NA-pattern provided implies this is not the case. \nConsider setting all correlations involving a critical variable to NA. \nGood luck for the next try!"
    stop(ErrorMsg)
  }

  # Define objects
  if (dim(empNMat)[1] == 0) currentN <- matrix(empN, dim(empCovMat)[1], dim(empCovMat)[1]) else  currentN <- empNMat
  currentR <- empCovMat; currentR
  if (is.null(studyNumber)) studyNumber <- 1

  ###### Handling of missing correlations (NA) ######
  origN <- currentN; origN
  eigenValueProblems <- "Congratulations! This covariance matrix is positive definite (no negative eigenvalues)."
  newMat <- currentR; newMat
  k <- which(is.na(newMat), arr.ind=TRUE); k
  if (dim(k)[1] > 0) {
    sortedK <-  t(apply(k,1,sort)); sortedK
    uniqueK <- unique(sortedK); uniqueK
    reversedK <- cbind(uniqueK[,2], uniqueK[,1]); reversedK
    k <- rbind(uniqueK, reversedK); k
    randomCors <- stats::rnorm(length(uniqueK)/2, 0, .0); randomCors
    randomCors <- c(randomCors, randomCors); randomCors
    newMat[k] <- randomCors
    diag(newMat) <- 1; newMat
  }

  ### solve problem with non-positive definite matrices (https://www.r-bloggers.com/fixing-non-positive-definite-correlation-matrices-using-r-2/)
  #counter <- 0
  ##eigen(newMat)$values
  #while(any(eigen(newMat)$values < 0)) {
  #  counter <- counter + 1
  #  print(cat("Study", studyNumber, "has negative eigenvalues. Try to fix this problem at iteration =", counter, "!"))
  #  newEig <- eigen(newMat); newEig
  #  newEig2 <- ifelse(newEig$values < 0, 0, newEig$values)
  #  newMat <- newEig$vectors %*% diag(newEig2) %*% t(newEig$vectors)
  #  newMat <- newMat/sqrt(diag(newMat) %*% t(diag(newMat)))
  #  eigenValueProblems <- paste0("The correlation matrix of Study ", i, "had negative Eigenvalues. I tried to fix this, but better check the matrix.")
  #}

  toBeMadeNA <- makeNAdueToNACors <- makeNAdueToZeroN <- NULL
  # store rows and cols with correlations == NA
  if (dim(k)[1] > 0) makeNAdueToNACors <- which(table(c(k[,2])) > dim(newMat)[2]/2); makeNAdueToNACors # which columns in the final data set should be made NA
  makeNAdueToNACors <- as.numeric(names(makeNAdueToNACors)); makeNAdueToNACors ##### CHECK!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  # store rows and cols with N == 0
  makeNAdueToZeroN <- which((apply(currentN, 1, sum) == 0) & (apply(currentN, 2, sum) == 0) == TRUE); makeNAdueToZeroN
  toBeMadeNA <- unique(c(makeNAdueToNACors, makeNAdueToZeroN)); toBeMadeNA

  currentR <- newMat; currentR
  currentN[currentN == 0] <- max(currentN[currentN > 0]); currentN #replace N=0 (missing correlations) by max N

  ###### Pairwise deletion ######
  numberOfMatrices <- 2^(dim(currentN)[1]); numberOfMatrices
  if (numberOfMatrices > 10^6) numberOfMatrices <- 10^6     # prevent memory overflow
  matrixL <- lapply(1:numberOfMatrices, list); matrixL
  matrixL[[1]] <- list(r=currentR, n=currentN); matrixL[[1]]
  dimnames(matrixL[[1]]$r) <- dimnames(matrixL[[1]]$n) <- list(1:(dim(matrixL[[1]]$r)[1]), 1:(dim(matrixL[[1]]$n)[1]))
  whileCounter <- matCounter <- 1
  ncounter <- 0
  while(length(matrixL[[whileCounter]]) > 1) {
    minN <- min(matrixL[[whileCounter]]$n); minN
    matDim <- dim(matrixL[[whileCounter]]$n)[1]; matDim
    if (is.null(matDim)) matDim <- 1; matDim
    if ( (minN == 1 & matDim == 1) | (minN == 0) ) {
      matrixL[[whileCounter]]$data <- (matrix(NA, 1, matDim))
    } else {
      if (matDim > 1) minNcoord <- which(min(matrixL[[whileCounter]]$n) == matrixL[[whileCounter]]$n, arr.ind=TRUE)[1, ] else minNcoord <- c(1,1)
      # correction for minN < dimMat by deleting rows cols with N < dimMat
      while (minN <= matDim) {
        # retain submatrix with largest sum of N
        tmp1 <- matrixL[[whileCounter]]$n[-minNcoord[1], -minNcoord[1]]; tmp1
        tmp2 <- matrixL[[whileCounter]]$n[-minNcoord[2], -minNcoord[2]]; tmp2
        tmp1 <- sum(tmp1[lower.tri(tmp1)]); tmp1
        tmp2 <- sum(tmp2[lower.tri(tmp2)]); tmp2
        if (tmp1 >= tmp2) {
          matrixL[[whileCounter]]$r <- matrixL[[whileCounter]]$r[-minNcoord[1], -minNcoord[1]]; matrixL[[whileCounter]]$r
          matrixL[[whileCounter]]$n <- matrixL[[whileCounter]]$n[-minNcoord[1], -minNcoord[1]]; matrixL[[whileCounter]]$n
        } else {
          matrixL[[whileCounter]]$r <- matrixL[[whileCounter]]$r[-minNcoord[2], -minNcoord[2]]; matrixL[[whileCounter]]$r
          matrixL[[whileCounter]]$n <- matrixL[[whileCounter]]$n[-minNcoord[2], -minNcoord[2]]; matrixL[[whileCounter]]$n
        }
        minN <- min(matrixL[[whileCounter]]$n); minN
        if (is.null(dim(matrixL[[whileCounter]]$n)[1])) matDim <- 1 else matDim <- dim(matrixL[[whileCounter]]$n)[1]
        if (matDim > 1) minNcoord <- which(min(matrixL[[whileCounter]]$n) == matrixL[[whileCounter]]$n, arr.ind=TRUE)[1, ] else minNcoord <- c(1,1)
        if (minN == 1 & matDim == 1) break #leave while loop
      }
      if (!(minN == 1 & matDim == 1)) {
        r <- matrixL[[whileCounter]]$r; r
        n <- matrixL[[whileCounter]]$n; n
        # generate data
        matrixL[[whileCounter]]$data <- MASS::mvrnorm(n=minN, mu=rep(0, matDim), Sigma=r, empirical = TRUE); matrixL[[whileCounter]]$data
        ncounter <- ncounter + minN
      }
      if (matDim > 1) {
        matrixL[[matCounter + 1]] <- matrixL[[matCounter + 2]] <- list()
        # reduce correlation matrices
        matrixL[[matCounter + 1]]$r <- r[-minNcoord[1], -minNcoord[1]]; matrixL[[matCounter + 1]]$r
        matrixL[[matCounter + 2]]$r <- r[-minNcoord[2], -minNcoord[2]]; matrixL[[matCounter + 2]]$r
        # preserve names and make matrix if dim = NUL (only 1 element)
        if (is.null(dim(matrixL[[matCounter + 1]]$r))) {
          matrixL[[matCounter + 1]]$r <- matrix(matrixL[[matCounter + 1]]$r, 1, 1,
                                                dimnames=list(unlist(dimnames(r)[1])[-minNcoord[1]], unlist(dimnames(r)[1])[-minNcoord[1]]))
        }
        if (is.null(dim(matrixL[[matCounter + 2]]$r))) {
          matrixL[[matCounter + 2]]$r <- matrix(matrixL[[matCounter + 2]]$r, 1, 1,
                                                dimnames=list(unlist(dimnames(r)[1])[-minNcoord[2]], unlist(dimnames(r)[1])[-minNcoord[2]]))
        }
        ## reduce matrices with pairwise sample size and reduce sample sizes by n already created
        matrixL[[matCounter + 1]]$n <- n[-minNcoord[1], -minNcoord[1]] - minN; matrixL[[matCounter + 1]]$n
        matrixL[[matCounter + 2]]$n <- n[-minNcoord[2], -minNcoord[2]] - minN; matrixL[[matCounter + 2]]$n
        # preserve names
        if (is.null(dim(matrixL[[matCounter + 1]]$n))) {
          matrixL[[matCounter + 1]]$n <- matrix(matrixL[[matCounter + 1]]$n, 1, 1,
                                                dimnames=list(unlist(dimnames(r)[1])[-minNcoord[1]], unlist(dimnames(r)[1])[-minNcoord[1]]))
        }
        if (is.null(dim(matrixL[[matCounter + 2]]$n))) {
          matrixL[[matCounter + 2]]$n <- matrix(matrixL[[matCounter + 2]]$n, 1, 1,
                                                dimnames=list(unlist(dimnames(r)[1])[-minNcoord[2]], unlist(dimnames(r)[1])[-minNcoord[2]]))
        }
        # variables included in both matrices
        targets <- (colnames(matrixL[[matCounter + 1]]$n)[colnames(matrixL[[matCounter + 1]]$n)
                                                          %in% colnames(matrixL[[matCounter + 2]]$n)]);targets
        # take only half of n in overlapping variables
        matrixL[[matCounter + 1]]$n[targets, targets] <- round(matrixL[[matCounter + 1]]$n[targets, targets]/2 +.01); matrixL[[matCounter + 1]]$n
        matrixL[[matCounter + 2]]$n[targets, targets] <- round(matrixL[[matCounter + 2]]$n[targets, targets]/2 -.01); matrixL[[matCounter + 2]]$n
        matCounter <- matCounter + 2; matCounter
      }
    }
    whileCounter <- whileCounter + 1; whileCounter
  }

  # combine data
  newData <- as.data.frame(matrix(NA, ncounter, ncol=(dim(currentR)[1]) ) ); newData
  colnames(newData) <- 1:(dim(currentR)[1]); colnames(newData)
  rowCounter <- 1
  for (i in 1:(whileCounter-1)){
    if (!(is.null(dim(matrixL[[i]]$data)[1]))) tmpN <- dim(matrixL[[i]]$data)[1] else tmpN <- 1
    currentRows <- rowCounter:(rowCounter+tmpN-1); currentRows
    currentNames <- colnames(matrixL[[i]]$data); currentNames
    newData[currentRows, currentNames] <- matrixL[[i]]$data
    rowCounter <- max(currentRows) + 1; rowCounter
  }

  ### too small N in the diagonal can be corrected
  nDiff <- psych::corr.test(newData, ci=FALSE)$n - origN; nDiff
  rDiff <- round(psych::corr.test(newData, ci=FALSE)$r - currentR, 6); rDiff
  currentData <- list()
  counter <- 0
  for (i in 1:(dim(nDiff)[1])) {
    if (nDiff[i,i] < -1) {
      counter <- counter +1
      currentData[[counter]] <- MASS::mvrnorm(n=-nDiff[i,i], mu=rep(0), Sigma=matrix(currentR[i,i], 1, 1), empirical = TRUE)
      colnames(currentData[[counter]]) <- i; currentData[[counter]]
    }
  }
  # add data
  if (counter > 0) {
    rowCounter <- dim(newData)[1]+1; rowCounter
    for (i in 1:counter) {
      if (!(is.null(dim(currentData[[i]])[1]))) tmpN <- dim(currentData[[i]])[1] else tmpN <- 1
      currentRows <- rowCounter:(rowCounter+tmpN-1); currentRows
      currentNames <- colnames(currentData[[i]]); currentNames
      tmpMat <- as.data.frame(matrix(NA, tmpN, dim(newData)[2])); tmpMat
      colnames(tmpMat) <- colnames(newData)
      newData <- rbind(newData, tmpMat)
      newData[currentRows, currentNames] <- currentData[[i]]
      rowCounter <- max(currentRows) + 1; rowCounter
    }
  }
  if (all(diag(currentR) == 1)) newData <- scale(newData)

  # replace values which cannot exist
  if (!(is.null("toBeMadeNA"))) newData[,toBeMadeNA] <- NA

  # add means if they are provided
  if (!(is.null(empMeanVector))) {
    for (i in 1:(dim(newData)[2])) {
      newData[,i] <- newData[,i] + empMeanVector[i]
    }
  }

  nDiff <- psych::corr.test(newData, ci=FALSE)$n - origN; nDiff
  rDiff <- round(psych::corr.test(newData, ci=FALSE)$r - currentR, 6); rDiff
  overallNDiff <- sum(nDiff[lower.tri(nDiff, diag=TRUE)]); overallNDiff
  relativeNDiff <- overallNDiff/sum(origN[lower.tri(origN, diag=TRUE)]); relativeNDiff
  overallRDiff <- sum(rDiff[lower.tri(rDiff, diag=TRUE)], na.rm=TRUE); overallRDiff
  results <- list(newData, eigenValueProblems, nDiff, overallNDiff, relativeNDiff)
  names(results) <- c("data", "problems", "lostN", "overallLostN", "relativeLostN")

  invisible(results)
}
