% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cfX_LogNormal.R
\name{cfX_LogNormal}
\alias{cfX_LogNormal}
\title{Characteristic function of Lognormal distribution}
\usage{
cfX_LogNormal(t, mu = 0, sigma = 1, tol = 1e-06)
}
\arguments{
\item{t}{numerical values (number, vector...)}

\item{mu}{real, default value mu = 0}

\item{sigma}{> 0, default value sigma = 1}

\item{tol}{tolerance, default value tol = 1e-6}
}
\value{
characteristic function cf(t) of the Lognormal distribution computed for real (vector)
}
\description{
cfX_LogNormal(t,mu,sigma) Computes the characteristic function cf(t) of
the Lognormal distribution with parameters mu (real) and sigma > 0,
computed for real (vector) argument t, i.e.
cf(t) = cfX_LogNormal(t,mu,sigma);
}
\details{
In probability theory, a log-normal (or lognormal) distribution is a
continuous probability distribution of a random variable whose logarithm
is normally distributed. The lognormal distribution is defined for x in
(0,+inf) by its PDF/CDF/CF, as follows
 \eqn{pdf(x) = 1/(x*sigma*sqrt(2*pi))*exp(-(ln(x)-mu)^2/(2*sigma^2))
 cdf(x) = 1/2+1/2*erf((ln(x)-mu)/(sqrt(2)*sigma))
 cf(t)  = sum_0^infinity{(it)^n/n!*exp(n*mu + (n*sigma)^2/2)}.}
As noted, this representation is asymptotically divergent but sufficient
for numerical purposes.

cfX_LogNormal is based on the standard integral representation of the
characteristic function of the lognormal distribution, i.e.
 cf(t) = Integral_0^inf exp(i*t*x) * PDF(x) dx.
By using the half-space Fourier integral transformation we get
 cf(t) = Integral_0^inf (i/t) * exp(-x) * PDF(i*x/t) dx.
If we define the integrand as funCF(t,x) = (i/t) * exp(-x) * PDF(i*x/t),
then by using a stabilizing transformation from [0,inf] to [0,1], we can
evaluate the CF by the following (well behaved) integral:
 cf(t) = Integral_0^1 2x/(1-x)^3 * funCF(t,(x/(1-x))^2) dx.

cfX_LogNormal evaluates this integral by using the R built in
function 'integrate', with precission specified by tolerance tol (default
value is tol = 1e-6).
}
\examples{
# EXAMPLE1 (CF of the Lognormal distribution with mu = 0,sigma = 1)
mu <- 0
sigma <- 1
t <- seq(-20, 20, length.out = 2 ^ 10 + 1)
plotGraf(function(t)
  cfX_LogNormal(t, mu, sigma), t, title = "Characteristic function of the Lognormal distribution")

# EXAMPLE2 (CDF/PDF of the Lognormal distribution with mu = 0,sigma = 1)
mu <- 0
sigma <- 1
x <- seq(0, 15, length.out = 101)
prob <- c(0.9, 0.95, 0.99)
cf <- function(t)
  cfX_LogNormal(t, mu, sigma)
result <-
  cf2DistGP(cf,
            x,
            prob,
            N = 2 ^ 10,
            SixSigmaRule = 8,
            xMin = 0)

# EXAMPLE3 (PDF/CDF of the compound Poisson-Lognormal distribution)
mu <- 0
sigma <- 1
lambda <- 10
x <- seq(0, 70, length.out = 101)
prob <- c(0.9, 0.95, 0.99)
cfX <- function(t)
  cfX_LogNormal(t, mu, sigma)
cf <- function(t)
  cfN_Poisson(t, lambda, cfX)
result <- cf2DistGP(cf, x, prob, isCompound = TRUE, N = 2 ^ 10)
}
\seealso{
For more details see WIKIPEDIA:
\url{https://en.wikipedia.org/wiki/Log-normal_distribution}

Other Continuous Probability distribution: \code{\link{cfS_Arcsine}},
  \code{\link{cfS_Beta}}, \code{\link{cfS_Gaussian}},
  \code{\link{cfS_Rectangular}},
  \code{\link{cfS_StudentT}},
  \code{\link{cfS_Trapezoidal}},
  \code{\link{cfS_Triangular}}, \code{\link{cfX_Beta}},
  \code{\link{cfX_ChiSquared}},
  \code{\link{cfX_Exponential}}, \code{\link{cfX_Gamma}},
  \code{\link{cfX_InverseGamma}}, \code{\link{cfX_Normal}},
  \code{\link{cfX_PearsonV}},
  \code{\link{cfX_Rectangular}},
  \code{\link{cfX_Triangular}}
}

