% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/workflowfunction.R
\name{BiBitWorkflow}
\alias{BiBitWorkflow}
\title{BiBit Workflow}
\usage{
BiBitWorkflow(matrix, minr = 2, minc = 2, similarity_type = "col",
  func = "agnes", link = "average", par.method = 0.625,
  cut_type = "gap", cut_pm = "Tibs2001SEmax", gap_B = 500,
  gap_maxK = 50, noise = 0.1, noise_select = 0, plots = c(3:5),
  BCresult = NULL, simmatresult = NULL, treeresult = NULL,
  plot.type = "device", filename = "BiBitWorkflow", verbose = TRUE)
}
\arguments{
\item{matrix}{The binary input matrix.}

\item{minr}{The minimum number of rows of the Biclusters.}

\item{minc}{The minimum number of columns of the Biclusters.}

\item{similarity_type}{Which dimension to use for the Jaccard Index in Step 2. This is either columns (\code{"col"}, default) or both (\code{"both"}).}

\item{func}{Which clustering function to use in Step 3. Either \code{"agnes"} (= default) or \code{"hclust"}.}

\item{link}{Which clustering link to use in Step 3. The available links (depending on \code{func}) are:
\itemize{
\item{\code{hclust}: }\code{"ward.D"}, \code{"ward.D2"}, \code{"single"}, \code{"complete"}, \code{"average"}, \code{"mcquitty"}, \code{"median"} or \code{"centroid"}  
\item{\code{agnes}: }\code{"average"} (default), \code{"single"}, \code{"complete"}, \code{"ward"}, \code{"weighted"}, \code{"gaverage"} or \code{"flexible"}   
}
(More details in \code{\link[stats]{hclust}} and \code{\link[cluster]{agnes}})}

\item{par.method}{Additional parameters used for flexible link (See \code{\link[cluster]{agnes}}). Default is \code{c(0.625)}}

\item{cut_type}{Which method should be used to decide the number of clusters in the tree in Step 4? 
\itemize{
\item \code{"gap"}: Use the Gap Statistic (default).
\item \code{"number"}: Select a set number of clusters.
\item \code{"height"}: Cut the tree at specific dissimilarity height.
}}

\item{cut_pm}{Cut Parameter (depends on \code{cut_type}) for Step 4
\itemize{
\item Gap Statistic (\code{cut_type="gap"}): How to compute optimal number of clusters? Choose one of the following: \code{"Tibs2001SEmax"} (default), \code{"globalmax"}, \code{"firstmax"}, \code{"firstSEmax"} or \code{"globalSEmax"}.
\item Number (\code{cut_type="number"}): Integer for number of clusters.
\item Height (\code{cut_type="height"}): Numeric dissimilarity value where the tree should be cut (\code{[0,1]}).
}}

\item{gap_B}{Number of bootstrap samples (default=500) for Gap Statistic (\code{\link[cluster]{clusGap}}).}

\item{gap_maxK}{Number of clusters to consider (default=50) for Gap Statistic (\code{\link[cluster]{clusGap}}).}

\item{noise}{The allowed noise level when growing the rows on the merged patterns in Step 6. (default=\code{0.1}, namely allow 10\% noise.)
\itemize{
\item \code{noise=0}: No noise allowed.
\item \code{0<noise<1}: The \code{noise} parameter will be a noise percentage. The number of allowed 0's in a row in the bicluster will depend on the column size of the bicluster.
More specifically \code{zeros_allowed = ceiling(noise * columnsize)}. For example for \code{noise=0.10} and a bicluster column size of \code{5}, the number of allowed 0's would be \code{1}.
\item \code{noise>=1}: The \code{noise} parameter will be the number of allowed 0's in a row in the bicluster independent from the column size of the bicluster. In this noise option, the noise parameter should be an integer.
}}

\item{noise_select}{Should the allowed noise level be automatically selected for each pattern? (Using ad hoc method to find the elbow/kink in the Noise Scree plots)
\itemize{
\item \code{noise_select=0}: Do \emph{NOT} automatically select the noise levels. Use the the noise level given in the \code{noise} parameter (default).
\item \code{noise_select=1}: Using the Noise Scree plot (with 'Added Rows' on the y-axis), find the noise level where the current number of added rows at this noise level is larger than the mean of 'added rows' at the lower noise levels. 
After locating this noise level, lower the noise level by 1. This is your automatically selected elbow/kink and therefore your noise level.
\item \code{noise_select=2}: Applies the same steps as for \code{noise_select=1}, but instead of decreasing the noise level by only 1, keep decreasing the noise level until the number of added rows isn't decreasing anymore either.
}}

\item{plots}{Vector for which plots to draw:
\enumerate{
\item Image plot of the similarity matrix computed in Step 2.
\item Same as \code{plots=1}, but the rows and columns are reordered with the hierarchical tree.
\item Dendrogram of the tree, its clusters colored after the chosen cut has been applied.
\item Noise Scree plots for all the Saved Patterns. Two plots will be plotted, both with Noise on the x-axis. The first one will have the number of Added Number of Rows on that noise level on the y-axis, while the second will have the Total Number of Rows (i.e. cumulative of the first).
If the title of one of the subplots is red, then this means that the Bicluster grown from this pattern, using the chosen noise level, was eventually deleted due to being a duplicate or non-maximal.
\item Image plot of the Jaccard Index similarity matrix between the final biclusters after Step 6.
}}

\item{BCresult}{Import a BiBit Biclust result for Step 1 (e.g. extract from an older BiBitWorkflow object \code{$info$BiclustInitial}). This can be useful if you want to cut the tree differently/make different plots, but don't want to do the BiBit calculation again.}

\item{simmatresult}{Import a (custom) Similarity Matrix (e.g. extract from older BiBitWorkflow object \code{$info$BiclustSimInitial}). Note that Step 1 (BiBit) will still be executed if \code{BCresult} is not provided.}

\item{treeresult}{Import a (custom) tree (\code{hclust} object) based on the BiBit/Similarity (e.g. extract from older BiBitWorkflow object \code{$info$Tree}).}

\item{plot.type}{Output Type
\itemize{
\item \code{"device"}: All plots are outputted to new R graphics devices (default).
\item \code{"file"}: All plots are saved in external files. Plots 1 and 2 are saved in separate \code{.png} files while all other plots are joint together in a single \code{.pdf} file.
\item \code{"other"}: All plots are outputted to the current graphics device, but will overwrite each other. Use this if you want to include one or more plots in a sweave/knitr file or if you want to export a single plot by your own chosen format.
}}

\item{filename}{Base filename (with/without directory) for the plots if \code{plot.type="file"} (default=\code{"BiBitWorkflow"}).}

\item{verbose}{Logical value if progress of workflow should be printed.}
}
\value{
A BiBitWorkflow S3 List Object with 3 slots:
\itemize{
\item \code{Biclust}: Biclust Class Object of Final Biclustering Result (after Step 6).
\item \code{BiclustSim}: Jaccard Index Similarity Matrix of Final Biclustering Result (after Step 6).
\item \code{info}: List Object containing:
\itemize{
\item \code{BiclustInitial}: Biclust Class Object of Initial Biclustering Result (after Step 1).
\item \code{BiclustSimInitial}: Jaccard Index Similarity Matrix of Initial Biclustering Result (after Step 1).
\item \code{Tree}: Hierarchical Tree of \code{BiclustSimInitial} as \code{hclust} object.
\item \code{Number}: Vector containing the initial number of biclusters (\code{InitialNumber}), the number of saved patterns after cutting the tree (\code{PatternNumber}) and the final number of biclusters (\code{FinalNumber}).
\item \code{GapStat}: Vector containing all different optimal cluster numbers based on the Gap Statistic.
\item \code{BC.Merge}: A list (length of merged saved patterns) containing which biclusters were merged together after cutting the tree.
\item \code{MergedColPatterns}: A list (length of merged saved patterns) containing the indices of which columns make up that pattern.
\item \code{MergedNoiseThresholds}: A vector containing the selected noise levels for the merged saved patterns.
\item \code{Coverage}: A list containing: 1. a vector of the total number (and percentage) of unique rows the final biclusters cover. 2. a table showing how many rows are used more than a single time in the final biclusters.
\item \code{Call}: A match.call of the original function call.
}
}
}
\description{
Workflow to discover larger (noisy) patterns in big data using BiBit
}
\details{
Looking for Noisy Biclusters in large data using BiBit (\code{\link{bibit2}}) often results in many (overlapping) biclusters.
In order decrease the number of biclusters and find larger meaningful patterns which make up noisy biclusters, the following workflow can be applied.
Note that this workflow is primarily used for data where there are many more rows (e.g. patients) than columns (e.g. symptoms). For example the workflow would discover larger meaningful symptom patterns which, conditioned on the allowed noise/zeros, subsets of the patients share.
\enumerate{
\item Apply BiBit with \emph{no noise} (Preferably with high enough \code{minr} and \code{minc}).
\item Compute Similarity Matrix (Jaccard Index) of all biclusters. By default this measure is only based on column similarity.
This implies that the rows of the BC's are not of interest in this step. The goal then would be to discover highly overlapping column patterns and, in the next steps, merge them together.
\item Apply Agglomerative Hierarchical Clustering on Similarity Matrix (default = average link)
\item Cut the dendrogram of the clustering result and merge the biclusters based on this. (default = number of clusters is determined by the Tibs2001SEmax Gap Statistic)
\item Extract Column Memberships of the Merged Biclusters. These are saved as the new column \emph{Patterns}.
\item Starting from these patterns, \emph{(noisy) rows} are grown which match the pattern, creating a single final bicluster for each pattern. At the end duplicate/non-maximal BC's are deleted.
}
Using the described workflow (and column similarity in Step 2), the final result will contain biclusters which focus on larger column patterns.
}
\examples{
\dontrun{
## Simulate Data ##
# DATA: 10000x50
# BC1: 200x10
# BC2: 100x10
# BC1 and BC2 overlap 5 columns

# BC3: 200x10
# BC4: 100x10
# BC3 and bC4 overlap 2 columns

# Background 1 percentage: 0.15
# BC Signal Percentage: 0.9
 
set.seed(273)
mat <- matrix(sample(c(0,1),10000*50,replace=TRUE,prob=c(1-0.15,0.15)),
              nrow=10000,ncol=50)
mat[1:200,1:10] <- matrix(sample(c(0,1),200*10,replace=TRUE,prob=c(1-0.9,0.9)),
                          nrow=200,ncol=10)
mat[300:399,6:15] <- matrix(sample(c(0,1),100*10,replace=TRUE,prob=c(1-0.9,0.9)),
                            nrow=100,ncol=10)
mat[400:599,21:30] <- matrix(sample(c(0,1),200*10,replace=TRUE,prob=c(1-0.9,0.9)),
                             nrow=200,ncol=10)
mat[700:799,29:38] <- matrix(sample(c(0,1),100*10,replace=TRUE,prob=c(1-0.9,0.9)),
                             nrow=100,ncol=10)
mat <- mat[sample(1:10000,10000,replace=FALSE),sample(1:50,50,replace=FALSE)]


# Computing gap statistic for initial 1381 BC takes approx. 15 min.
# Gap Statistic chooses 4 clusters. 
out <- BiBitWorkflow(matrix=mat,minr=50,minc=5,noise=0.2) 
summary(out$Biclust)

# Reduce computation by selecting number of clusters manually.
# Note: The "ClusterRowCoverage" function can be used to provided extra info 
#       on the number of cluster choice.
#       How?
#       - More clusters result in smaller column patterns and more matching rows.
#       - Less clusters result in larger column patterns and less matching rows.
# Step 1: Initial Workflow Run
out2 <- BiBitWorkflow(matrix=mat,minr=50,minc=5,noise=0.2,cut_type="number",cut_pm=10)
# Step 2: Use ClusterRowCoverage
temp <- ClusterRowCoverage(result=out2,matrix=mat,noise=0.2,plots=2)
# Step 3: Use BiBitWorkflow again (using previously computed parts) with new cut parameter
out3 <- BiBitWorkflow(matrix=mat,minr=50,minc=5,noise=0.2,cut_type="number",cut_pm=4,
                      BCresult = out2$info$BiclustInitial,
                      simmatresult = out2$info$BiclustSimInitial)
summary(out3$Biclust)
}
}
\author{
Ewoud De Troyer
}

