% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ic.R
\name{calc.ic}
\alias{calc.ic}
\title{Calculate an information criterion.}
\usage{
calc.ic(
  y_hat,
  y,
  ic = c("aicc", "bicc", "aic", "bic", "gcv", "cp"),
  df,
  sigma = NULL
)
}
\arguments{
\item{y_hat}{A vector of fitted values with \code{length(y_hat)=length(y)=n}, or
a matrix, with \code{nrow(coef)=length(y)=n} and \code{ncol(y_hat)=m}, containing m different fits.}

\item{y}{A vector of response variable, with \code{length(y)=n}.}

\item{ic}{A specified IC to calculate. Default is AICc ('aicc'). Other choices include AIC ('aic'),
BIC ('bic'), BICc ('bicc'), GCV ('gcv') and Mallows' Cp ('cp').}

\item{df}{A number if y_hat is a vector, or a vector with \code{length(df)=ncol(y_hat)=m} if y_hat is
a matrix. df represents the degrees of freedom for each fit.}

\item{sigma}{Standard deviation of the error term. It only needs to be specified if the argument \code{ic='cp'}.}
}
\value{
The value(s) of the specified IC for each fit.
}
\description{
Calculate a specified information criterion (IC) for an estimate or a group of estimates.
The choices of IC include AIC, BIC, AICc, BICc, GCV and Mallows' Cp.
}
\details{
This function enables the computation of various common IC for model fits, which can
  further be used to choose the optimal fit. This allows user comparing the effect of different IC.
  In order to calculate an IC, degrees of freedoms (df) needs to be specified. To be more specific,
  here are the formulas used to calculate each IC:

  \deqn{AIC = \log(\frac{RSS}{n}) + 2\frac{df}{n}}{AIC = log(RSS/n) + 2*df/n}
  \deqn{BIC = \log(\frac{RSS}{n}) + \log(n)\frac{df}{n}}{BIC = log(RSS/n) + log(n)*df/n}
  \deqn{AICc = \log(\frac{RSS}{n}) + 2\frac{df+1}{n-df-2}}{AICc = log(RSS/n) + 2*(df+1)/(n-df-2)}
  \deqn{BICc = \log(\frac{RSS}{n}) + \log(n)\frac{df+1}{n-df-2}}{BICc = log(RSS/n) + log(n)*(df+1)/(n-df-2)}
  \deqn{GCV = \frac{RSS}{(n-df)^2}}{GCV = RSS/(n-df)^2}
  \deqn{Mallows' Cp = RSS + 2\times \sigma^2 \times df}{AIC = RSS + 2*\sigma^2*df}
}
\examples{
## Generate a trivial dataset, X has mean 0 and norm 1, y has mean 0
set.seed(11)
n = 20
p = 5
x = matrix(rnorm(n*p), nrow=n, ncol=p)
x = scale(x, center = colMeans(x))
x = scale(x, scale = sqrt(colSums(x^2)))
beta = c(1, 1, 0, 0, 0)
y = x\%*\%beta + scale(rnorm(20, sd=0.01), center = TRUE, scale = FALSE)

## Fit the model
boss_result = boss(x, y)
## Print the values of AICc-hdf for all subsets given by BOSS
print(boss_result$IC_boss$aicc)
## calculate them manually using the calc.ic function
y_hat = cbind(rep(1,n),x)\%*\%boss_result$beta_boss
print(calc.ic(y_hat, y, df=boss_result$hdf_boss))
}
\author{
Sen Tian
}
