% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rsi_analysis.R
\name{rsi}
\alias{rsi}
\alias{rsi_df}
\alias{n_rsi}
\title{Resistance of isolates}
\usage{
rsi(ab1, ab2 = NA, interpretation = "IR", minimum = 30,
  as_percent = FALSE, info = FALSE, warning = TRUE)

rsi_df(tbl, ab, interpretation = "IR", minimum = 30, as_percent = FALSE,
  info = TRUE, warning = TRUE)

n_rsi(ab1, ab2 = NA)
}
\arguments{
\item{ab1, ab2}{vector of antibiotic interpretations, they will be transformed internally with \code{\link{as.rsi}}}

\item{interpretation}{antimicrobial interpretation of which the portion must be calculated. Valid values are \code{"S"}, \code{"SI"}, \code{"I"}, \code{"IR"} or \code{"R"}.}

\item{minimum}{minimal amount of available isolates. Any number lower than \code{minimum} will return \code{NA} with a warning (when \code{warning = TRUE}).}

\item{as_percent}{return output as percent (text), will else (at default) be a double}

\item{info}{calculate the amount of available isolates and print it, like \code{n = 423}}

\item{warning}{show a warning when the available amount of isolates is below \code{minimum}}

\item{tbl}{\code{data.frame} containing columns with antibiotic interpretations.}

\item{ab}{character vector with 1, 2 or 3 antibiotics that occur as column names in \code{tbl}, like \code{ab = c("amox", "amcl")}}
}
\value{
Double or, when \code{as_percent = TRUE}, a character.
}
\description{
This functions can be used to calculate the (co-)resistance of isolates (i.e. percentage S, SI, I, IR or R [of a vector] of isolates). The functions \code{rsi} and \code{n_rsi} can be used in \code{dplyr}s \code{\link[dplyr]{summarise}} and support grouped variables, see \emph{Examples}.
}
\details{
Remember that you should filter your table to let it contain \strong{only first isolates}!
\if{html}{
  \cr \cr
  To calculate the probability (\emph{p}) of susceptibility of one antibiotic, we use this formula:
  \out{<div style="text-align: center">}\figure{mono_therapy.png}\out{</div>}
  To calculate the probability (\emph{p}) of susceptibility of more antibiotics (i.e. combination therapy), we need to check whether one of them has a susceptible result (as numerator) and count all cases where all antibiotics were tested (as denominator). \cr
  For two antibiotics:
  \out{<div style="text-align: center">}\figure{combi_therapy_2.png}\out{</div>}
  \cr
  For three antibiotics:
  \out{<div style="text-align: center">}\figure{combi_therapy_3.png}\out{</div>}
}
}
\examples{
library(dplyr)

septic_patients \%>\%
  group_by(hospital_id) \%>\%
  summarise(cipro_susceptibility = rsi(cipr, interpretation = "S"),
            n = n_rsi(cipr)) # n_rsi works like n_distinct in dplyr

septic_patients \%>\%
  group_by(hospital_id) \%>\%
  summarise(cipro_S = rsi(cipr, interpretation = "S",
                          as_percent = TRUE, warning = FALSE),
            cipro_n = n_rsi(cipr),
            genta_S = rsi(gent, interpretation = "S",
                          as_percent = TRUE, warning = FALSE),
            genta_n = n_rsi(gent),
            combination_S = rsi(cipr, gent, interpretation = "S",
                                as_percent = TRUE, warning = FALSE),
            combination_n = n_rsi(cipr, gent))

# calculate resistance
rsi(septic_patients$amox)
# or susceptibility
rsi(septic_patients$amox, interpretation = "S")

# calculate co-resistance between amoxicillin/clav acid and gentamicin,
# so we can review that combination therapy does a lot more than mono therapy:
septic_patients \%>\% rsi_df(ab = "amcl", interpretation = "S") # = 67.8\%
septic_patients \%>\% rsi_df(ab = "gent", interpretation = "S") # = 69.1\%
septic_patients \%>\% rsi_df(ab = c("amcl", "gent"), interpretation = "S") # = 90.6\%

\dontrun{
# calculate current empiric combination therapy of Helicobacter gastritis:
my_table \%>\%
  filter(first_isolate == TRUE,
         genus == "Helicobacter") \%>\%
  rsi_df(ab = c("amox", "metr")) # amoxicillin with metronidazole
}
}
\keyword{antibiotics}
\keyword{isolate}
\keyword{isolates}
\keyword{rsi}
