% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/linalg.R
\name{linalg_householder_product}
\alias{linalg_householder_product}
\title{Computes the first \code{n} columns of a product of Householder matrices.}
\usage{
linalg_householder_product(A, tau)
}
\arguments{
\item{A}{(Tensor): tensor of shape \verb{(*, m, n)} where \code{*} is zero or more batch dimensions.}

\item{tau}{(Tensor): tensor of shape \verb{(*, k)} where \code{*} is zero or more batch dimensions.}
}
\description{
Letting \teqn{\mathbb{K}} be \teqn{\mathbb{R}} or \teqn{\mathbb{C}},
for a matrix \teqn{V \in \mathbb{K}^{m \times n}} with columns \teqn{v_i \in \mathbb{K}^m}
with \teqn{m \geq n} and a vector \teqn{\tau \in \mathbb{K}^k} with \teqn{k \leq n},
this function computes the first \teqn{n} columns of the matrix
}
\details{
\Sexpr[results=rd, stage=build]{torch:::math_to_rd("
H_1H_2 ... H_k \\\\qquad with \\\\qquad H_i = \\\\mathrm{I}_m - \\\\tau_i v_i v_i^{H}
")}

where \teqn{\mathrm{I}_m} is the \code{m}-dimensional identity matrix and
\teqn{v^{H}} is the conjugate transpose when \teqn{v} is complex, and the transpose when \teqn{v} is real-valued.
See \href{https://www.netlib.org/lapack/lug/node128.html}{Representation of Orthogonal or Unitary Matrices} for
further details.

Supports inputs of float, double, cfloat and cdouble dtypes.
Also supports batches of matrices, and if the inputs are batches of matrices then
the output has the same batch dimensions.
}
\note{
This function only uses the values strictly below the main diagonal of \code{A}.
The other values are ignored.
}
\examples{
if (torch_is_installed()) {
A <- torch_randn(2, 2)
h_tau <- torch_geqrf(A)
Q <- linalg_householder_product(h_tau[[1]], h_tau[[2]])
torch_allclose(Q, linalg_qr(A)[[1]])

}
}
\seealso{
\itemize{
\item \code{\link[=torch_geqrf]{torch_geqrf()}} can be used together with this function to form the \code{Q} from the
\code{\link[=linalg_qr]{linalg_qr()}} decomposition.
\item \code{\link[=torch_ormqr]{torch_ormqr()}} is a related function that computes the matrix multiplication
of a product of Householder matrices with another matrix.
However, that function is not supported by autograd.
}

Other linalg: 
\code{\link{linalg_cholesky_ex}()},
\code{\link{linalg_cholesky}()},
\code{\link{linalg_det}()},
\code{\link{linalg_eigh}()},
\code{\link{linalg_eigvalsh}()},
\code{\link{linalg_eigvals}()},
\code{\link{linalg_eig}()},
\code{\link{linalg_inv_ex}()},
\code{\link{linalg_inv}()},
\code{\link{linalg_lstsq}()},
\code{\link{linalg_matrix_norm}()},
\code{\link{linalg_matrix_power}()},
\code{\link{linalg_matrix_rank}()},
\code{\link{linalg_multi_dot}()},
\code{\link{linalg_norm}()},
\code{\link{linalg_pinv}()},
\code{\link{linalg_qr}()},
\code{\link{linalg_slogdet}()},
\code{\link{linalg_solve}()},
\code{\link{linalg_svdvals}()},
\code{\link{linalg_svd}()},
\code{\link{linalg_tensorinv}()},
\code{\link{linalg_tensorsolve}()},
\code{\link{linalg_vector_norm}()}
}
\concept{linalg}
