% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lrtestspsur.R
\name{lrtestspsur}
\alias{lrtestspsur}
\title{Likelihood Ratio tests for the specification of spatial SUR models.}
\usage{
lrtestspsur(
  Form = NULL,
  data = NULL,
  W = NULL,
  X = NULL,
  Y = NULL,
  time = NULL,
  G = NULL,
  N = NULL,
  Tm = NULL
)
}
\arguments{
\item{Form}{An object created with the package \code{\link[Formula]{Formula}}
that describes the model to be estimated. This model may contain several
responses (explained variables) and a varying number of regressors in each equation.}

\item{data}{An object of class data.frame or a matrix.}

\item{W}{A spatial weighting matrix of order \emph{(NxN)}, assumed to be the
same for all equations and time periods.}

\item{X}{A data matrix of order \emph{(NTmGxp)} with the observations of the regressors
The number of covariates in the SUR model is p = \eqn{sum(p_{g})} where \emph{\eqn{p_{g}}}
is the number of regressors (including the intercept) in the g-th equation, \emph{g = 1,...,G}).
The specification of \emph{X} is only necessary if not available a \code{\link[Formula]{Formula}}
and a data frame. Default = \code{NULL}.}

\item{Y}{A column vector of order \emph{(NTmGx1)}, with the observations of the
explained variables. The ordering of the data must be (first) equation,
(second) time dimension and (third) Cross-sectional/spatial units.
The specification of \emph{Y} is only necessary if not available a \code{\link[Formula]{Formula}}
and a data frame. Default = \code{NULL}.}

\item{time}{Time variable.}

\item{G}{Number of equations.}

\item{N}{Number of cross-section or spatial units}

\item{Tm}{Number of time periods.}
}
\value{
\code{\link{lrtestspsur}}, first, prints the value of the estimated log-likelihood for
   the major spatial specifications. Then, the function shows the values of the LR statistics corresponding to the nested
   and nesting models compared, together with their associated p-value.
}
\description{
The function computes a set of Likelihood Ratio tests, LR, that help
 the user to select the spatial structure of the SUR model. To achieve this goal, \code{\link{lrtestspsur}}
 needs to estimate the SUR models \strong{"sim"}, \strong{"slm"}, \strong{"sem"}, \strong{"sdm"},
 and \strong{"sarar"}, using the function \code{\link{spsurml}}.

 The five models listed above are related by a nesting sequence, so they can be compared using
 the adequate LR tests. The function shows the log-likelihood corresponding to the maximum-likelihood
 estimates and the sequence of LR tests.
}
\details{
A fundamental result in maximum-likelihood estimation shows that if \emph{model A} is nested
in \emph{model B}, by a set of \emph{n} restrictions on the parameters of \emph{model B}, then,
as the sample size increases, the test statistic: \emph{\eqn{-2log[l(H_{0}) / l(H_{A})]}}
is a \eqn{\chi^{2}(n)}, being l(H_{0} the estimated likelihood under the null hypothesis
(\emph{model A}) and  l(H_{A} the estimated likelihood under the alternative hypothesis (\emph{model B}).

 The list of (spatial) models that can be estimated with the function \code{\link{spsurml}} includes
  the following (in addition to the \strong{"slx"} and \strong{"sdem"}):

 \itemize{
    \item \strong{"sim"}: SUR model with no spatial effects
      \deqn{ y_{tg} = X_{tg} \beta_{g} + \epsilon_{tg} }
    \item \strong{"slm"}: SUR model with spatial lags of the explained variables
      \deqn{y_{tg} = \lambda_{g} Wy_{tg} + X_{tg} \beta_{g} + \epsilon_{tg} }
    \item \strong{"sem"}: SUR model with spatial errors
      \deqn{ y_{tg} = X_{tg} \beta_{g} + u_{tg} }
      \deqn{ u_{tg} = \rho_{g} Wu_{tg} + \epsilon_{tg} }
    \item \strong{"sdm"}: SUR model of the Spatial Durbin type
      \deqn{ y_{tg} = \lambda_{g} Wy_{tg} + X_{tt} \beta_{g} + WX_{tg} \theta_{g} + \epsilon_{tg} }
    \item \strong{"sarar"}: SUR model with spatial lags of the explained variables and spatial
      errors
      \deqn{ y_{tg} = \lambda_{g} Wy_{tg} + X_{tg} \beta_{g} + u_{tg} }
      \deqn{ u_{tg} = \rho_{g} W u_{tg} + \epsilon_{tg} }
  }


  This collection of models can be compared, on objective bases, using the LR principle  and the
   following  nesting relations:

  \itemize{
    \item  \strong{"sim"} vs \strong{"sem"}, where the null hypotheses, in the \strong{"sem"} equation, are:

  \deqn{ H_{0}: \rho_{g}=0 forall g vs  H_{A}: \rho_{g} ne 0 exist g}

    \item  \strong{"sim"} vs \strong{"slm"}, where the null hypotheses, in the \strong{"slm"} equation, are:

  \deqn{ H_{0}: \lambda_{g}=0 forall g vs  H_{A}: \lambda_{g} ne 0 exist g}

    \item  \strong{"sim"} vs \strong{"sarar"}, where the null hypotheses, in the \strong{"sarar"} equation, are:

  \deqn{ H_{0}: \rho_{g}=\lambda_{g}=0 forall g vs  H_{A}: \rho_{g} ne 0 or \lambda_{g} ne 0 exist g}

    \item  \strong{"sem"} vs \strong{"sarar"}, where the null hypotheses, in the \strong{"sarar"} equation, are:

  \deqn{ H_{0}: \lambda_{g}=0 forall g vs  H_{A}: \lambda_{g} ne 0 exist g}

    \item  \strong{"slm"} vs \strong{"sarar"}, where the null hypotheses, in the \strong{"sarar"} equation, are:

  \deqn{ H_{0}: \rho_{g}=0 forall g vs  H_{A}: \rho_{g} ne 0 exist g}

    \item  \strong{"sem"} vs \strong{"sdm"}, also known as \emph{LR-COMFAC}, where the null hypotheses, in the \strong{"sdm"}
     equation, are:

   \deqn{ H_{0}: -\lambda_{g}\beta_{g}=\theta_{g} forall g vs  H_{A}: -\lambda_{g}\beta_{g} ne \theta_{g} exist g}

  }

 The degrees of freedom of the corresponding \eqn{\chi^{2}} distribution is \emph{G} in the cases of \strong{"sim"}
 vs \strong{"sem"}, \strong{"sim"} vs \strong{"slm"}, \strong{"sem"} vs \strong{"sarar"}, \strong{"slm"} vs
 \strong{"sarar"}  and \strong{"sem"} vs  \strong{"sdm"} and \emph{2G} in the case of \strong{"sim"} vs \strong{"sarar"}.
  Moreover, function \code{\link{lrtestspsur}} also returns the p-values associated to the corresponding LR.
}
\examples{
#################################################
######## CROSS SECTION DATA (nG=1; nT>1) ########
#################################################

#### Example 1: Spatial Phillips-Curve. Anselin (1988, p. 203)
rm(list = ls()) # Clean memory
data("spc")
Tformula <- WAGE83 | WAGE81 ~ UN83 + NMR83 + SMSA | UN80 + NMR80 + SMSA
## It usually requires 1-2 minutes maximum
## LRs <- lrtestspsur(Form = Tformula, data = spc, W = Wspc)

#################################################
######## CROSS SECTION DATA (nG>1; nT=1) ########
#################################################

#### Example 2: Homicides & Socio-Economics (1960-90)
# Homicides and selected socio-economic characteristics for
# continental U.S. counties.
# Data for four decennial census years: 1960, 1970, 1980 and 1990.
# https://geodacenter.github.io/data-and-lab/ncovr/
\donttest{
## It could require some minutes
rm(list = ls()) # Clean memory
data("NCOVR")
Tformula <- HR70 | HR80  | HR90 ~ PS70 + UE70 | PS80 + UE80 + RD80 |
            PS90 + UE90 + RD90 + PO90
LRs <- lrtestspsur(Form = Tformula, data = NCOVR, W = W)
}

################################################################
######## PANEL DATA: TEMPORAL CORRELATIONS (nG=1; nT>1) ########
################################################################

#### Example 3: Classic panel data
\donttest{
## It could require some minutes
rm(list = ls()) # Clean memory
data(NCOVR)
N <- nrow(NCOVR)
Tm <- 4
index_time <- rep(1:Tm, each = N)
index_indiv <- rep(1:N, Tm)
pHR <- c(NCOVR$HR60, NCOVR$HR70, NCOVR$HR80, NCOVR$HR90)
pPS <- c(NCOVR$PS60, NCOVR$PS70, NCOVR$PS80, NCOVR$PS90)
pUE <- c(NCOVR$UE60, NCOVR$UE70, NCOVR$UE80, NCOVR$UE90)
pNCOVR <- data.frame(indiv = index_indiv, time = index_time, HR = pHR, PS = pPS, UE = pUE)
rm(NCOVR,pHR,pPS,pUE,index_time,index_indiv)
form_pHR <- HR ~ PS + UE
LRs <- lrtestspsur(Form = form_pHR, data = pNCOVR, W = W, time = pNCOVR$time)
}
}
\references{
\itemize{
    \item Mur, J., López, F., and Herrera, M. (2010). Testing for spatial
      effects in seemingly unrelated regressions.
      \emph{Spatial Economic Analysis}, 5(4), 399-440.
     \item López, F.A., Mur, J., and Angulo, A. (2014). Spatial model
       selection strategies in a SUR framework. The case of regional
       productivity in EU. \emph{Annals of Regional Science}, 53(1),
       197-220.
  }
}
\seealso{
\code{\link{spsurml}}, \code{\link{lmtestspsur}}
}
\author{
\tabular{ll}{
  Fernando López  \tab \email{fernando.lopez@upct.es} \cr
  Román Mínguez  \tab \email{roman.minguez@uclm.es} \cr
  Jesús Mur  \tab \email{jmur@unizar.es} \cr
  }
}
