% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/setupCritPlot.R
\name{setupCritPlot}
\alias{setupCritPlot}
\alias{setupCritPlot.rlars}
\alias{setupCritPlot.grplars}
\alias{setupCritPlot.tslarsP}
\alias{setupCritPlot.seqModel}
\alias{setupCritPlot.tslars}
\alias{setupCritPlot.sparseLTS}
\alias{setupCritPlot.perrySeqModel}
\alias{setupCritPlot.perrySparseLTS}
\title{Set up an optimality criterion plot of a sequence of regression models}
\usage{
setupCritPlot(object, ...)

\method{setupCritPlot}{seqModel}(object, which = c("line", "dot"), ...)

\method{setupCritPlot}{tslars}(object, p, ...)

\method{setupCritPlot}{sparseLTS}(
  object,
  which = c("line", "dot"),
  fit = c("reweighted", "raw", "both"),
  ...
)

\method{setupCritPlot}{perrySeqModel}(object, which = c("line", "dot", "box", "density"), ...)

\method{setupCritPlot}{perrySparseLTS}(
  object,
  which = c("line", "dot", "box", "density"),
  fit = c("reweighted", "raw", "both"),
  ...
)
}
\arguments{
\item{object}{the model fit from which to extract information.}

\item{\dots}{additional arguments to be passed down.}

\item{which}{a character string specifying the type of plot.  Possible
values are \code{"line"} (the default) to plot the (average) results for
each model as a connected line, \code{"dot"} to create a dot plot,
\code{"box"} to create a box plot, or \code{"density"} to create a smooth
density plot.  Note that the last two plots are only available in case of
prediction error estimation via repeated resampling.}

\item{p}{an integer giving the lag length for which to extract information
(the default is to use the optimal lag length).}

\item{fit}{a character string specifying for which estimator to extract
information.  Possible values are \code{"reweighted"} (the default) for
the reweighted fits, \code{"raw"} for the raw fits, or \code{"both"} for
both estimators.}
}
\value{
An object inheriting from class \code{"setupCritPlot"} with the
following components:
\describe{
  \item{\code{data}}{a data frame containing the following columns:
  \describe{
    \item{\code{Fit}}{a vector or factor containing the identifiers of the
    models along the sequence.}
    \item{\code{Name}}{a factor specifying the estimator for which the
    optimality criterion was estimated (\code{"reweighted"} or \code{"raw"};
    only returned if both are requested in the \code{"sparseLTS"} or
    \code{"perrySparseLTS"} methods).}
    \item{\code{PE}}{the estimated prediction errors (only returned if
    applicable).}
    \item{\code{BIC}}{the estimated values of the Bayesian information
    criterion (only returned if applicable).}
    \item{\code{Lower}}{the lower end points of the error bars (only
    returned if possible to compute).}
    \item{\code{Upper}}{the upper end points of the error bars (only
    returned if possible to compute).}
  }
  }
  \item{\code{which}}{a character string specifying the type of plot.}
  \item{\code{grouped}}{a logical indicating whether density plots should
  be grouped due to multiple model fits along the sequence (only returned
  in case of density plots for the \code{"perrySeqModel"} and
  \code{"perrySparseLTS"} methods).}
  \item{\code{includeSE}}{a logical indicating whether error bars based on
  standard errors are available (only returned in case of line plots or dot
  plots).}
  \item{\code{mapping}}{default aesthetic mapping for the plots.}
  \item{\code{facets}}{default faceting formula for the plots (only
  returned if both estimators are requested in the \code{"sparseLTS"}
  or \code{"perrySparseLTS"} methods).}
  \item{\code{tuning}}{a data frame containing the grid of tuning parameter
  values for which the optimality criterion was estimated (only returned for
  the \code{"sparseLTS"} and \code{"perrySparseLTS"} methods).}
}
}
\description{
Extract the relevent information for a plot of the values of the optimality
criterion for a sequence of regression models, such as submodels along a
robust or groupwise least angle regression sequence, or sparse least trimmed
squares regression models for a grid of values for the penalty parameter.
}
\examples{
## generate data
# example is not high-dimensional to keep computation time low
library("mvtnorm")
set.seed(1234)  # for reproducibility
n <- 100  # number of observations
p <- 25   # number of variables
beta <- rep.int(c(1, 0), c(5, p-5))  # coefficients
sigma <- 0.5      # controls signal-to-noise ratio
epsilon <- 0.1    # contamination level
Sigma <- 0.5^t(sapply(1:p, function(i, j) abs(i-j), 1:p))
x <- rmvnorm(n, sigma=Sigma)    # predictor matrix
e <- rnorm(n)                   # error terms
i <- 1:ceiling(epsilon*n)       # observations to be contaminated
e[i] <- e[i] + 5                # vertical outliers
y <- c(x \%*\% beta + sigma * e)  # response
x[i,] <- x[i,] + 5              # bad leverage points


## robust LARS
# fit model
fitRlars <- rlars(x, y, sMax = 10)
# extract information for plotting
setup <- setupCritPlot(fitRlars)
critPlot(setup)


## sparse LTS over a grid of values for lambda
# fit model
frac <- seq(0.2, 0.05, by = -0.05)
fitSparseLTS <- sparseLTS(x, y, lambda = frac, mode = "fraction")
# extract information for plotting
setup1 <- setupCritPlot(fitSparseLTS)
critPlot(setup1)
setup2 <- setupCritPlot(fitSparseLTS, fit = "both")
critPlot(setup2)
}
\seealso{
\code{\link{critPlot}}, \code{\link{rlars}},
\code{\link{grplars}}, \code{\link{rgrplars}}, \code{\link{tslarsP}},
\code{\link{rtslarsP}}, \code{\link{tslars}}, \code{\link{rtslars}},
\code{\link{sparseLTS}}
}
\author{
Andreas Alfons
}
\keyword{utilities}
