% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rstandard.R
\name{rstandard.seqModel}
\alias{rstandard.seqModel}
\alias{rstandard.rlars}
\alias{rstandard.grplars}
\alias{rstandard.tslarsP}
\alias{rstandard.tslars}
\alias{rstandard.perrySeqModel}
\alias{rstandard.sparseLTS}
\title{Extract standardized residuals from a sequence of regression models}
\usage{
\method{rstandard}{seqModel}(model, s = NA, drop = !is.null(s), ...)

\method{rstandard}{tslars}(model, p, ...)

\method{rstandard}{perrySeqModel}(model, ...)

\method{rstandard}{sparseLTS}(
  model,
  s = NA,
  fit = c("reweighted", "raw", "both"),
  drop = !is.null(s),
  ...
)
}
\arguments{
\item{model}{the model fit from which to extract standardize residuals.}

\item{s}{for the \code{"seqModel"} method, an integer vector giving the
steps of the submodels for which to extract the standardized residuals (the
default is to use the optimal submodel).  For the \code{"sparseLTS"} method,
an integer vector giving the indices of the models for which to extract
standardized residuals.  If \code{fit} is \code{"both"}, this can be a list
with two components, with the first component giving the indices of the
reweighted fits and the second the indices of the raw fits.  The default is
to use the optimal model for each of the requested estimators.  Note that
the optimal models may not correspond to the same value of the penalty
parameter for the reweighted and the raw estimator.}

\item{drop}{a logical indicating whether to reduce the dimension to a
vector in case of only one step.}

\item{\dots}{for the \code{"tslars"} method, additional arguments to be
passed down to the \code{"seqModel"} method.  For the other methods,
additional arguments are currently ignored.}

\item{p}{an integer giving the lag length for which to extract standardized
residuals (the default is to use the optimal lag length).}

\item{fit}{a character string specifying which standardized residuals to
extract.  Possible values are \code{"reweighted"} (the default) for the
standardized residuals from the reweighted estimator, \code{"raw"} for the
standardized residuals from the raw estimator, or \code{"both"} for the
standardized residuals from both estimators.}
}
\value{
A numeric vector or matrix containing the requested standardized residuals.
}
\description{
Extract standardized residuals from a sequence of regression models, such as
submodels along a robust or groupwise least angle regression sequence, or
sparse least trimmed squares regression models for a grid of values for the
penalty parameter.
}
\examples{
## generate data
# example is not high-dimensional to keep computation time low
library("mvtnorm")
set.seed(1234)  # for reproducibility
n <- 100  # number of observations
p <- 25   # number of variables
beta <- rep.int(c(1, 0), c(5, p-5))  # coefficients
sigma <- 0.5      # controls signal-to-noise ratio
epsilon <- 0.1    # contamination level
Sigma <- 0.5^t(sapply(1:p, function(i, j) abs(i-j), 1:p))
x <- rmvnorm(n, sigma=Sigma)    # predictor matrix
e <- rnorm(n)                   # error terms
i <- 1:ceiling(epsilon*n)       # observations to be contaminated
e[i] <- e[i] + 5                # vertical outliers
y <- c(x \%*\% beta + sigma * e)  # response
x[i,] <- x[i,] + 5              # bad leverage points


## robust LARS
# fit model
fitRlars <- rlars(x, y, sMax = 10)
# extract standardized residuals
rstandard(fitRlars)
head(rstandard(fitRlars, s = 1:5))


## sparse LTS over a grid of values for lambda
# fit model
frac <- seq(0.2, 0.05, by = -0.05)
fitSparseLTS <- sparseLTS(x, y, lambda = frac, mode = "fraction")
# extract standardized residuals
rstandard(fitSparseLTS)
head(rstandard(fitSparseLTS, fit = "both"))
head(rstandard(fitSparseLTS, s = NULL))
head(rstandard(fitSparseLTS, fit = "both", s = NULL))
}
\seealso{
\code{\link[stats]{rstandard}}, \code{\link[=residuals.seqModel]{residuals}}

\code{\link{rlars}}, \code{\link{grplars}}, \code{\link{rgrplars}},
\code{\link{tslarsP}}, \code{\link{rtslarsP}}, \code{\link{tslars}},
\code{\link{rtslars}}, \code{\link{sparseLTS}}
}
\author{
Andreas Alfons
}
\keyword{regression}
