% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/capacity_value.R
\name{capacity_value}
\alias{capacity_value}
\title{Calculate capacity value for variable generation}
\usage{
capacity_value(time.data, outage.table, VG.cols = NULL, marginal = FALSE,
  scale.first = FALSE, ...)
}
\arguments{
\item{time.data}{Time series data (or subset) formatted with \code{\link{format_timedata}}}

\item{outage.table}{Outage table used in the lookup, created with \code{\link{outage_table}}}

\item{VG.cols}{Order in which VG capacity value is calculated (see details for more info)}

\item{marginal}{Should the VG capacity value be calculated as the last resource (\code{TRUE}) or added progressively to the load (\code{FALSE})?}

\item{scale.first}{This triggers a special case to calculate capacity value. See details for more information.}

\item{...}{Additional parameters passed to \code{\link{calculate_elcc}} and \code{\link{sliding_window}}}
}
\description{
Given time data series and an outage table, calculate the capacity value of variable generation (VG) time series.
Capacity value is calculated by iteratively removing one type of VG at a time and recalculating effective
load carrying capability (ELCC).
}
\details{
This function calculates ELCC for the system with different combinations of VG. The metric and desired reliability
metric can be set; see \code{\link{calculate_elcc}} for details and a list of parameters. Additionally, a sliding
window can be used in the calculation (see \code{\link{calculate_elcc}}).

When calculating the capacity value of VG, the results depends on the order in which the different technologies
are added into the system. The variable \code{VG.cols} is used to determine this order manually. If not determined
the order will default to the order in which the VG columns appear in \code{time.data}.

This \code{marginal} parameter determines if the capacity value is calculated by taking each VG profile out
of the total mix (when is set to \code{TRUE}) or by progressively stacking the VG profiles on top the conventional
generators (when is set to \code{FALSE}). The latter option is the default and the order is determined by the
\code{VG.cols} parameter or, if not set, by the order in which the columns appear in the \code{time.data} object.

If \code{scale.first} is set to \code{TRUE} the capacity value a special calculation is triggered. Before perfoming the
calculations, the load is scaled so that the total mix (conventional generators and VG) provide the desired adequacy
level, which is specified with the parameters passed to \code{\link{calculate_elcc}}. Once the load is scaled, the
capacity value calculations are performed using the flat block method.
}
\examples{
\dontrun{
# Create outage table with 200 5-MW units
gens <- data.frame(Capacity = rep(5, 200),
                   EFOR = rep(0.08, 200))
out.table <- outage_table(gens)

# Create random load and wind data and format
tdata <- data.frame(Time = 1:8760,
                    Load = runif(8760, 450, 850),
                    Wind = runif(8760, 0, 100),
                    Wind2 = runif(8760, 0, 200))
td <- format_timedata(tdata)

# Calculate capacity value (both are equivalent)
capacity_value(td, out.table)
capacity_value(td, out.table, c("Wind", "Wind2"))

# Calculate capacity value of Wind2 first and Wind second
capacity_value(td, out.table, c("Wind", "Wind2"))

# Calculate capacity value with a sliding window that uses adjacent hours
capacity_value(td, out.table, win.h.size = c(-1, 1))
}
}
\seealso{
\code{\link{format_timedata}} and \code{\link{outage_table}} to create \code{time.data}
         and \code{outage.table} objects, respectively

\code{\link{sliding_window}} is used internally to extend \code{time.data}

\code{\link{calculate_elcc}} is used internally to evaluate ELCC
}

