#' @name exportMetaData
#' 
#' @title Export Meta Data from a REDCap Database
#' @description Retrieves the meta data for a REDcap database, including 
#' field names, labels, types, formulas, etc.  This file can be used to parse 
#' levels of factors, apply labels, and other data management tasks once the 
#' data are retrieved
#' 
#' @param rcon A REDCap connection object as generated by \code{redcapConnection.}
#' @param fields A character vector of field names for which the metadata is to 
#'   be retrieved.  
#' @param forms A character vector of forms for which the metadata is to be
#'   retrieved. Note that if a form name is given, all of the fields on that form
#'   will be returned, regardless of whether it is included in \code{fields} or 
#'   not.  Be careful to use the form names in the second column of the data 
#'   dictionary, and not the display names shown on the webpage.
#' @param ... Arguments to be passed to other methods.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param drop_utf8 \code{logical(1)}. In some cases, UTF-8 characters can 
#'   pose problems for exporting the data dictionary.  Set this to \code{TRUE}
#'   to replace any UTF-8 characters with empty characters.
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#' 
#' @details A record of this export is placed in the REDCap logging page, 
#' but the file that is exported is not stored in the database.
#' 
#' @section REDCap API Documentation:
#' This function allows you to export the metadata for a project
#' 
#' @section REDCap Version:
#' 5.8.2+ (and earlier, but we don't know how much earlier)
#' 
#' @author Jeffrey Horner
#' 
#' @export

exportMetaData <- function(rcon, ...){
  UseMethod("exportMetaData")
}

#' @rdname exportMetaData
#' @export

exportMetaData.redcapApiConnection <- function(rcon, 
                                               fields = character(0), 
                                               forms = character(0),
                                               ...,
                                               drop_utf8 = FALSE, 
                                               error_handling = getOption("redcap_error_handling"), 
                                               config = list(), 
                                               api_param = list()){
  # Argument validation ---------------------------------------------
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon,
                          classes = "redcapApiConnection",
                          add = coll)
  
  checkmate::assert_character(x = fields, 
                              add = coll)
  
  checkmate::assert_character(x = forms, 
                              add = coll)
  
  checkmate::assert_logical(x = drop_utf8,
                            len = 1,
                            add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling, 
                                        choices = c("null", "error"), 
                                        .var.name = "error_handling", 
                                        add = coll)

  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
  # Build the Body List ---------------------------------------------
  body <- c(list(token = rcon$token,
               content = "metadata",
               format = "csv",
               returnFormat = "csv"), 
            vectorToApiBodyList(fields, 
                                parameter_name = "fields"), 
            vectorToApiBodyList(forms, 
                                parameter_name = "forms"))
  
  body <- body[lengths(body) > 0]
 
  # API Call --------------------------------------------------------
  response <- makeApiCall(rcon, 
                          body = c(body, api_param), 
                          config = config)
  
  if (response$status_code != 200){
    redcap_error(response, 
                 error_handling = error_handling)
  }
  
  # Post processing -------------------------------------------------
  response <- as.character(response)
  if (drop_utf8)
  {
    response <- iconv(response, "utf8", "ASCII", sub = "")
  }
  
  utils::read.csv(text = response, 
                  stringsAsFactors = FALSE,
                  na.strings = "")
}
