% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cal-estimate-multinom.R
\name{cal_estimate_multinomial}
\alias{cal_estimate_multinomial}
\alias{cal_estimate_multinomial.data.frame}
\alias{cal_estimate_multinomial.tune_results}
\alias{cal_estimate_multinomial.grouped_df}
\title{Uses a Multinomial calibration model to calculate new probabilities}
\usage{
cal_estimate_multinomial(
  .data,
  truth = NULL,
  estimate = dplyr::starts_with(".pred_"),
  smooth = TRUE,
  parameters = NULL,
  ...
)

\method{cal_estimate_multinomial}{data.frame}(
  .data,
  truth = NULL,
  estimate = dplyr::starts_with(".pred_"),
  smooth = TRUE,
  parameters = NULL,
  ...,
  .by = NULL
)

\method{cal_estimate_multinomial}{tune_results}(
  .data,
  truth = NULL,
  estimate = dplyr::starts_with(".pred_"),
  smooth = TRUE,
  parameters = NULL,
  ...
)

\method{cal_estimate_multinomial}{grouped_df}(
  .data,
  truth = NULL,
  estimate = NULL,
  smooth = TRUE,
  parameters = NULL,
  ...
)
}
\arguments{
\item{.data}{An ungrouped \code{data.frame} object, or \code{tune_results} object,
that contains predictions and probability columns.}

\item{truth}{The column identifier for the true class results
(that is a factor). This should be an unquoted column name.}

\item{estimate}{A vector of column identifiers, or one of \code{dplyr} selector
functions to choose which variables contains the class probabilities. It
defaults to the prefix used by tidymodels (\code{.pred_}). The order of the
identifiers will be considered the same as the order of the levels of the
\code{truth} variable.}

\item{smooth}{Applies to the logistic models. It switches between logistic
spline when \code{TRUE}, and simple logistic regression when \code{FALSE}.}

\item{parameters}{(Optional)  An optional tibble of tuning parameter values
that can be used to filter the predicted values before processing. Applies
only to \code{tune_results} objects.}

\item{...}{Additional arguments passed to the models or routines used to
calculate the new probabilities.}

\item{.by}{The column identifier for the grouping variable. This should be
a single unquoted column name that selects a qualitative variable for
grouping. Default to \code{NULL}. When \code{.by = NULL} no grouping will take place.}
}
\description{
Uses a Multinomial calibration model to calculate new probabilities
}
\details{
When \code{smooth = FALSE}, \code{\link[nnet:multinom]{nnet::multinom()}} function is used to estimate the
model, otherwise \code{\link[mgcv:gam]{mgcv::gam()}} is used.
}
\examples{
\dontshow{if (!probably:::is_cran_check() & rlang::is_installed(c("modeldata", "parsnip", "randomForest"))) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(modeldata)
library(parsnip)
library(dplyr)

f <-
  list(
    ~ -0.5 + 0.6 * abs(A),
    ~ ifelse(A > 0 & B > 0, 1.0 + 0.2 * A / B, -2),
    ~ -0.6 * A + 0.50 * B - A * B
  )

set.seed(1)
tr_dat <- sim_multinomial(500, eqn_1 = f[[1]], eqn_2 = f[[2]], eqn_3 = f[[3]])
cal_dat <- sim_multinomial(500, eqn_1 = f[[1]], eqn_2 = f[[2]], eqn_3 = f[[3]])
te_dat <- sim_multinomial(500, eqn_1 = f[[1]], eqn_2 = f[[2]], eqn_3 = f[[3]])

set.seed(2)
rf_fit <-
  rand_forest() |>
  set_mode("classification") |>
  set_engine("randomForest") |>
  fit(class ~ ., data = tr_dat)

cal_pred <-
  predict(rf_fit, cal_dat, type = "prob") |>
  bind_cols(cal_dat)
te_pred <-
  predict(rf_fit, te_dat, type = "prob") |>
  bind_cols(te_dat)

cal_plot_windowed(cal_pred, truth = class, window_size = 0.1, step_size = 0.03)

smoothed_mn <- cal_estimate_multinomial(cal_pred, truth = class)

new_test_pred <- cal_apply(te_pred, smoothed_mn)

cal_plot_windowed(new_test_pred, truth = class, window_size = 0.1, step_size = 0.03)
\dontshow{\}) # examplesIf}
}
\seealso{
\url{https://www.tidymodels.org/learn/models/calibration/},
\code{\link[=cal_validate_multinomial]{cal_validate_multinomial()}}
}
