% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ip_pewma.R
\name{IpPewma}
\alias{IpPewma}
\title{Incremental Processing Probabilistic-EWMA (PEWMA).}
\usage{
IpPewma(data, n.train = 5, alpha0 = 0.8, beta = 0, l = 3,
  last.res = NULL)
}
\arguments{
\item{data}{Numerical vector with training and test dataset.}

\item{n.train}{Number of points of the dataset that correspond to the
training set.}

\item{alpha0}{Maximal weighting parameter.}

\item{beta}{Weight placed on the probability of the given observation.}

\item{l}{Control limit multiplier.}

\item{last.res}{Last result returned by the algorithm.}
}
\value{
A list of the following items.

  \item{result}{dataset conformed by the following columns.}
  \itemize{
     \item \code{is.anomaly} 1 if the value is anomalous 0, otherwise.
     \item \code{ucl} Upper control limit.
     \item \code{lcl} Lower control limit.
 }
 \item{last.res}{Last result returned by the algorithm. Is a dataset
 containing the parameters calculated in the last iteration and necessary
 for the next one.}
}
\description{
\code{IpPewma} allows the calculation of anomalies using PEWMA
in an incremental processing mode. See also \code{\link{OipPewma}}, the
optimized and faster function of this function This algorithm is a
probabilistic method of EWMA which dynamically adjusts the parameterization
based on the probability of the given observation. This method produces
dynamic, data-driven anomaly thresholds which are robust to abrupt transient
changes, yet quickly adjust to long-term distributional shifts.
}
\details{
\code{data} must be a numerical vector without NA values.
\code{alpha0} must be a numeric value where 0 < \code{alpha0} < 1. If a
faster adjustment to the initial shift is desirable, simply lowering
\code{alpha0} will suffice. \code{beta} is the weight placed on the
probability of the given observation. it must be a numeric value where
0 <= \code{beta} <= 1. Note that \code{beta} equals 0, PEWMA converges to a
standard EWMA. Finally \code{l} is the parameter that determines the control
limits. By default, 3 is used. \code{last.res} is the last result returned
by some previous execution of this algorithm. The first time the algorithm
is executed its value is NULL. However, to run a new batch
of data without having to include it in the old dataset and restart the
process, the two parameters returned by the last run are only needed.

This algorithm can be used for both classical and incremental processing. It
should be noted that in case of having a finite dataset the
\code{\link{CpPewma}} or \code{\link{OcpPewma}} algorithms are faster.
Incremental processing can be used in two ways. 1) Processing all available
data and saving \code{last.res} for future runs in which there is new data.
2) Using the \href{https://CRAN.R-project.org/package=stream}{stream} library
for when there is too much data and it does not fit into the memory.
An example has been made for this use case.
}
\examples{
## EXAMPLE 1: ----------------------
## It can be used in the same way as with CpPewma passing the whole dataset as
## an argument.

## Generate data
set.seed(100)
n <- 350
x <- sample(1:100, n, replace = TRUE)
x[70:90] <- sample(110:115, 21, replace = TRUE)
x[25] <- 200
x[320] <- 170
df <- data.frame(timestamp = 1:n,value = x)

## Calculate anomalies
result <- IpPewma(
  data = df$value,
  alpha0 = 0.8,
  beta = 0.1,
  n.train = 5,
  l = 3,
  last.res = NULL
)
res <- cbind(df, result$result)

## Plot results
PlotDetections(res, title = "PEWMA ANOMALY DETECTOR")

## EXAMPLE 2: ----------------------
## You can use it in an incremental way. This is an example using the stream
## library. This library allows the simulation of streaming operation.
\donttest{
# install.packages("stream")
library("stream")

## Generate data
set.seed(100)
n <- 500
x <- sample(1:100, n, replace = TRUE)
x[70:90] <- sample(110:115, 21, replace = TRUE)
x[25] <- 200
x[320] <- 170
df <- data.frame(timestamp = 1:n, value = x)
dsd_df <- DSD_Memory(df)

## Initialize parameters for the loop
last.res <- NULL
res <- NULL
nread <- 100
numIter <- n\%/\%nread

## Calculate anomalies
for(i in 1:numIter) {
  # read new data
  newRow <- get_points(dsd_df, n = nread, outofpoints = "ignore")
  # calculate if it's an anomaly
  last.res <- IpPewma(
    data = newRow$value,
    n.train = 5,
    alpha0 = 0.8,
    beta = 0.1,
    l = 3,
    last.res = last.res$last.res
  )
  # prepare the result
  if(!is.null(last.res$result)){
    res <- rbind(res, cbind(newRow, last.res$result))
  }
}

## Plot results
PlotDetections(res, title = "PEWMA ANOMALY DETECTOR")
}


}
\references{
M. Carter, Kevin y W. Streilein. Probabilistic reasoning for
streaming anomaly detection. 2012 IEEE Statistical Signal Processing Workshop
(SSP), pp. 377-380, Aug 2012.
}
