
#' Add gene tracks to a ggplot2 plot
#'
#' Adds gene tracks to an existing ggplot2 plot.
#'
#' @param p ggplot2 plot object. This can be generated by [gg_scatter()] and
#'   then modified.
#' @param loc Object of class 'locus' to use for plot. See [locus()].
#' @param height A unit object specifying height of the lower gene track.
#' @param ... Additional arguments passed to [gg_genetracks()] to control
#'   colours of gene tracks etc.
#' @return A gtable plotting object.
#' @details
#' The combined plot is outputted to the current device. A `gtable` plotting
#' object is returned invisibly. This can be plotted using [grid.draw()].
#' @seealso [gg_scatter()] [gg_genetracks()]
#' @examples
#' if(require(EnsDb.Hsapiens.v75)) {
#' library(EnsDb.Hsapiens.v75)
#' data(SLE_gwas_sub)
#' loc <- locus(SLE_gwas_sub, gene = 'IRF5', flank = c(7e4, 2e5), LD = "r2",
#'              ens_db = "EnsDb.Hsapiens.v75")
#' p <- gg_scatter(loc)
#' gg_addgenes(p, loc)
#' }
#' @importFrom ggplot2 ggplotGrob find_panel layer_scales
#' @importFrom gtable gtable_add_rows gtable_add_grob
#' @importFrom grid grid.newpage grid.draw
#' @export

gg_addgenes <- function(p, loc,
                        height = unit(5, "cm"),
                        ...) {
  # check x axis range
  xl <- layer_scales(p)$x$get_limits() * 1e6
  if (!identical(loc$xrange, xl)) {
    message("Plot x axis limits and locus range differ: check correct locus?")
    message("x axis limits: ", xl[1], " to ", xl[2])
    message("locus range: ", loc$xrange[1], " to ", loc$xrange[2])
  }
  
  g <- ggplotGrob(p)
  panels_extent <- g %>% find_panel()
  pg <- g %>%
    gtable_add_rows(heights = height) %>%
    gtable_add_grob(gg_genetracks(loc, ...),
                    t = -1, b = -1,
                    l = panels_extent$l, r = panels_extent$l +1)
  grid.newpage()
  grid.draw(pg)
  invisible(pg)
}
