% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/regularizeSEMInterface.R
\name{elasticNet}
\alias{elasticNet}
\title{elasticNet}
\usage{
elasticNet(
  lavaanModel,
  regularized,
  lambdas,
  alphas,
  method = "glmnet",
  modifyModel = lessSEM::modifyModel(),
  control = lessSEM::controlGlmnet()
)
}
\arguments{
\item{lavaanModel}{model of class lavaan}

\item{regularized}{vector with names of parameters which are to be regularized.
If you are unsure what these parameters are called, use
getLavaanParameters(model) with your lavaan model object}

\item{lambdas}{numeric vector: values for the tuning parameter lambda}

\item{alphas}{numeric vector with values of the tuning parameter alpha. Must be
between 0 and 1. 0 = ridge, 1 = lasso.}

\item{method}{which optimizer should be used? Currently implemented are ista
and glmnet. With ista, the control argument can be used to switch to related procedures
(currently gist).}

\item{modifyModel}{used to modify the lavaanModel. See ?modifyModel.}

\item{control}{used to control the optimizer. This element is generated with
the lessSEM::controlIsta() and controlGlmnet() functions.}
}
\value{
Model of class regularizedSEM
}
\description{
Implements elastic net regularization for structural equation models.
The penalty function is given by:
\deqn{p( x_j) = \alpha\lambda| x_j| + (1-\alpha)\lambda x_j^2}
Note that the elastic net combines ridge and lasso regularization. If \eqn{\alpha = 0},
the elastic net reduces to ridge regularization. If \eqn{\alpha = 1} it reduces
to lasso regularization. In between, elastic net is a compromise between the shrinkage of
the lasso and the ridge penalty.
}
\details{
Identical to \pkg{regsem}, models are specified using \pkg{lavaan}. Currently,
most standard SEM are supported. \pkg{lessSEM} also provides full information
maximum likelihood for missing data. To use this functionality,
fit your \pkg{lavaan} model with the argument \code{sem(..., missing = 'ml')}.
\pkg{lessSEM} will then automatically switch to full information maximum likelihood
as well.

Elastic net regularization:
\itemize{
\item Zou, H., & Hastie, T. (2005). Regularization and variable selection via the elastic net.
Journal of the Royal Statistical Society: Series B, 67(2), 301–320. https://doi.org/10.1111/j.1467-9868.2005.00503.x
}

Regularized SEM
\itemize{
\item Huang, P.-H., Chen, H., & Weng, L.-J. (2017). A Penalized Likelihood Method for Structural Equation Modeling. Psychometrika, 82(2), 329–354. https://doi.org/10.1007/s11336-017-9566-9
\item Jacobucci, R., Grimm, K. J., & McArdle, J. J. (2016). Regularized Structural Equation Modeling. Structural
Equation Modeling: A Multidisciplinary Journal, 23(4), 555–566. https://doi.org/10.1080/10705511.2016.1154793
}

For more details on GLMNET, see:
\itemize{
\item Friedman, J., Hastie, T., & Tibshirani, R. (2010).
Regularization Paths for Generalized Linear Models via Coordinate Descent.
Journal of Statistical Software, 33(1), 1–20. https://doi.org/10.18637/jss.v033.i01
\item Yuan, G.-X., Chang, K.-W., Hsieh, C.-J., & Lin, C.-J. (2010).
A Comparison of Optimization Methods and Software for Large-scale
L1-regularized Linear Classification. Journal of Machine Learning Research, 11, 3183–3234.
\item Yuan, G.-X., Ho, C.-H., & Lin, C.-J. (2012).
An improved GLMNET for l1-regularized logistic regression.
The Journal of Machine Learning Research, 13, 1999–2030. https://doi.org/10.1145/2020408.2020421
}

For more details on ISTA, see:
\itemize{
\item Beck, A., & Teboulle, M. (2009). A Fast Iterative Shrinkage-Thresholding
Algorithm for Linear Inverse Problems. SIAM Journal on Imaging Sciences, 2(1),
183–202. https://doi.org/10.1137/080716542
\item Gong, P., Zhang, C., Lu, Z., Huang, J., & Ye, J. (2013).
A General Iterative Shrinkage and Thresholding Algorithm for Non-convex
Regularized Optimization Problems. Proceedings of the 30th International
Conference on Machine Learning, 28(2)(2), 37–45.
\item Parikh, N., & Boyd, S. (2013). Proximal Algorithms. Foundations and
Trends in Optimization, 1(3), 123–231.
}
}
\examples{
library(lessSEM)

# Identical to regsem, lessSEM builds on the lavaan
# package for model specification. The first step
# therefore is to implement the model in lavaan.

dataset <- simulateExampleData()

lavaanSyntax <- "
f =~ l1*y1 + l2*y2 + l3*y3 + l4*y4 + l5*y5 +
     l6*y6 + l7*y7 + l8*y8 + l9*y9 + l10*y10 +
     l11*y11 + l12*y12 + l13*y13 + l14*y14 + l15*y15
f ~~ 1*f
"

lavaanModel <- lavaan::sem(lavaanSyntax,
                           data = dataset,
                           meanstructure = TRUE,
                           std.lv = TRUE)

# Regularization:

lsem <- elasticNet(
  # pass the fitted lavaan model
  lavaanModel = lavaanModel,
  # names of the regularized parameters:
  regularized = paste0("l", 6:15),
  lambdas = seq(0,1,length.out = 5),
  alphas = seq(0,1,length.out = 3))

# the coefficients can be accessed with:
coef(lsem)

# elements of lsem can be accessed with the @ operator:
lsem@parameters[1,]

# optional: plotting the paths requires installation of plotly
# plot(lsem)

#### Advanced ###
# Switching the optimizer #
# Use the "method" argument to switch the optimizer. The control argument
# must also be changed to the corresponding function:
lsemIsta <- elasticNet(
  lavaanModel = lavaanModel,
  regularized = paste0("l", 6:15),
  lambdas = seq(0,1,length.out = 5),
  alphas = seq(0,1,length.out = 3),
  method = "ista",
  control = controlIsta())

# Note: The results are basically identical:
lsemIsta@parameters - lsem@parameters
}
