#' httk-pop: Correlated human physiological parameter Monte Carlo
#' 
#' @description
#' This is the core function for httk-pop correlated human physiological 
#' variability simulation as described by Ring et al. (2017) 
#' (\doi{10.1016/j.envint.2017.06.004}). This functions 
#' takes the data table of population biometrics (one individual per row)
#' generated by \code{\link{httkpop_generate}}, and converts it
#' to the corresponding table of HTTK model parameters for a specified HTTK 
#' model.
#' 
#' @details
#' The Monte Carlo methods used here were recently updated and described by
#' Breen et al. (submitted).
#' 
#' @param model One of the HTTK models: "1compartment", "3compartmentss",
#' "3compartment", or "pbtk".
#' @param samples The number of Monte Carlo samples to use (can often think of these
#' as separate individuals)
#' @param httkpop.dt A data table generated by \code{\link{httkpop_generate}}.
#' This defaults to NULL, in which case \code{\link{httkpop_generate}} is 
#' called to generate this table.
#' @param ... Additional arugments passed on to \code{\link{httkpop_generate}}.
#' 
#' @author Caroline Ring and John Wambaugh
#' 
#' @return 
#' A data.table with a row for each individual in the sample and a column for
#' each parater in the model.
#' 
#' @references 
#' \insertRef{ring2017identifying}{httk} 
#'
#' \insertRef{breen2022simulating}{httk} 
#' 
#' \insertRef{rowland1973clearance}{httk} 
#'
#' @examples
#' 
#' set.seed(42)
#' indiv_examp <- httkpop_generate(method="d", nsamp=10)
#' 
#' httk_param <- httkpop_mc(httkpop.dt=indiv_examp, 
#'                         samples=10,
#'                         model="1compartment")
#' 
#' @keywords httk-pop monte-carlo
#'
#' @export httkpop_mc
httkpop_mc <- function(model,
                       samples=1000,
                       httkpop.dt=NULL,
                       ...)
{
  if (is.null(model)) stop("Model must be specified.")
# We need to know model-specific information (from modelinfo_[MODEL].R]) 
# to set up the solver:
  model <- tolower(model)
  if (!(model %in% names(model.list)))            
  {
    stop(paste("Model",model,"not available. Please select from:",
      paste(names(model.list),collapse=", ")))
  }

# Generate the initial physiology data from NHANES biometrics:
  if (is.null(httkpop.dt))
  {
     httkpop.dt <- do.call(
                        httkpop_generate,
                        args=list(nsamp=samples,...))
  } else {
    # Needed for scoping (work with local table only):
    httkpop.dt <- copy(httkpop.dt)  
  }
  
  # Convert HTTK-Pop-generated parameters to HTTK physiological parameters
  physiology.dt <- httkpop.dt
  if (!is.null(model.list[[model]]$calc.standard.httkpop2httk))
  {
    if (model.list[[model]]$calc.standard.httkpop2httk)
    {
      physiology.dt <- httkpop_biotophys_default(indiv_dt = httkpop.dt)
    }
  } else {
    warning(paste0("Warning calc.standard.httkpop2httk not set in modelinfo file for model ",
                   model))
  }
  
  # set precision:
  cols <- colnames(physiology.dt)
  physiology.dt[ , (cols) := lapply(.SD, set_httk_precision), .SDcols = cols]

  return(physiology.dt)
}
