% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dfr_hddm.R
\name{dfr_hddm}
\alias{dfr_hddm}
\title{Adapted Hoeffding Drift Detection Method (HDDM) method}
\usage{
dfr_hddm(
  drift_confidence = 0.001,
  warning_confidence = 0.005,
  two_side_option = TRUE
)
}
\arguments{
\item{drift_confidence}{Confidence to the drift}

\item{warning_confidence}{Confidence to the warning}

\item{two_side_option}{Option to monitor error increments and decrements (two-sided) or only increments (one-sided)}
}
\value{
\code{dfr_hddm} object
}
\description{
is a drift detection method based on the Hoeffding’s inequality. HDDM_A uses the average as estimator.  \url{doi:10.1109/TKDE.2014.2345382}.
}
\examples{
require("daltoolbox")
require('ggplot2')
require('caret')

data("st_real_examples")

bfd <- st_real_examples$bfd1
bfd['batch_index'] <- format(bfd['expected_depart'], '\%V')
bfd <- bfd[bfd['depart'] == 'SBSP',]

# Model features
features <- c(
 'depart_elevation', 'depart_visibility', 'depart_day_period', 'depart_pressure', 
 'depart_relative_humidity', 'depart_dew_point', 'depart_wind_speed_scale'
)

## Target
bfd$delay_depart_bin <- bfd$delay_depart > 0
target = 'delay_depart_bin'
bfd = bfd[complete.cases(bfd[target]),]
slevels <- c(TRUE, FALSE)

# Evaluation
th=0.5

results <- c()
ordered_batches <- sort(unique(bfd$batch_index))
old_start_batch <- ordered_batches[1]

# Classification Algorithm
model <- stealthy(daltoolbox::cla_nb(target, slevels), dfr_hddm(drift_confidence=0.8))

for (batch in ordered_batches[2:length(ordered_batches)]){
 print(batch)
 print(old_start_batch)
 
 new_batch <- bfd[bfd$batch_index == batch,]
 last_batch <- bfd[(bfd$batch_index < batch) & (bfd$batch_index >= old_start_batch),]
 
 old_start_batch <- batch
 
 x_train <- last_batch[, features]
 y_train <- last_batch[, target, drop=FALSE]
 
 x_test <- new_batch[, features]
 y_test <- new_batch[, target]
 
 model <- fit(model, x_train, y_train)
 
 test_predictions <- predict(model, x_test)
 y_pred <- test_predictions[, 2] > th
 
 # Evaluation
 precision <- evaluate(mt_precision(), y_pred, y_test)
 recall <- evaluate(mt_recall(), y_pred, y_test)
 f1 <- evaluate(mt_fscore(), y_pred, y_test)
 
 results <- rbind(results, 
                  c(
                    batch,
                    precision,
                    recall,
                    f1,
                    model$drifted
                  )
 )
 
 print(nrow(model$x_train))
 print(nrow(new_batch))
}
results <- as.data.frame(results)
results['index'] <- as.Date(results$index)
names(results) <- c('index', 'precision', 'recall', 'f1', 'drift')
results[, length(names(results))] <- NULL

ggplot(data=results, aes(x=index, y=as.numeric(f1), group=1)) + 
 geom_line() +
 xlab('') +
 ylab('F1') +
 geom_vline(xintercept = results[results['drift']==TRUE, 'index'],
            linetype="dotted", 
            color = "red", linewidth=0.5) +
 theme_classic()

}
