/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 
#define H5S_FRIEND     

#include "H5private.h"   
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Opkg.h"      
#include "H5Spkg.h"      

static void  *H5O__sdspace_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                  size_t p_size, const uint8_t *p);
static herr_t H5O__sdspace_encode(H5F_t *f, uint8_t *p, const void *_mesg);
static void  *H5O__sdspace_copy(const void *_mesg, void *_dest);
static size_t H5O__sdspace_size(const H5F_t *f, const void *_mesg);
static herr_t H5O__sdspace_reset(void *_mesg);
static herr_t H5O__sdspace_free(void *_mesg);
static herr_t H5O__sdspace_pre_copy_file(H5F_t *file_src, const void *mesg_src, bool *deleted,
                                         const H5O_copy_t *cpy_info, void *_udata);
static herr_t H5O__sdspace_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

#define H5O_SHARED_TYPE        H5O_MSG_SDSPACE
#define H5O_SHARED_DECODE      H5O__sdspace_shared_decode
#define H5O_SHARED_DECODE_REAL H5O__sdspace_decode
#define H5O_SHARED_ENCODE      H5O__sdspace_shared_encode
#define H5O_SHARED_ENCODE_REAL H5O__sdspace_encode
#define H5O_SHARED_SIZE        H5O__sdspace_shared_size
#define H5O_SHARED_SIZE_REAL   H5O__sdspace_size
#define H5O_SHARED_DELETE      H5O__sdspace_shared_delete
#undef H5O_SHARED_DELETE_REAL
#define H5O_SHARED_LINK H5O__sdspace_shared_link
#undef H5O_SHARED_LINK_REAL
#define H5O_SHARED_COPY_FILE H5O__sdspace_shared_copy_file
#undef H5O_SHARED_COPY_FILE_REAL
#define H5O_SHARED_POST_COPY_FILE H5O__sdspace_shared_post_copy_file
#undef H5O_SHARED_POST_COPY_FILE_REAL
#undef H5O_SHARED_POST_COPY_FILE_UPD
#define H5O_SHARED_DEBUG      H5O__sdspace_shared_debug
#define H5O_SHARED_DEBUG_REAL H5O__sdspace_debug
#include "H5Oshared.h" 

const H5O_msg_class_t H5O_MSG_SDSPACE[1] = {{
    H5O_SDSPACE_ID,                            
    "dataspace",                               
    sizeof(H5S_extent_t),                      
    H5O_SHARE_IS_SHARABLE | H5O_SHARE_IN_OHDR, 
    H5O__sdspace_shared_decode,                
    H5O__sdspace_shared_encode,                
    H5O__sdspace_copy,                         
    H5O__sdspace_shared_size,                  
    H5O__sdspace_reset,                        
    H5O__sdspace_free,                         
    H5O__sdspace_shared_delete,                
    H5O__sdspace_shared_link,                  
    NULL,                                      
    NULL,                                      
    H5O__sdspace_pre_copy_file,                
    H5O__sdspace_shared_copy_file,             
    H5O__sdspace_shared_post_copy_file,        
    NULL,                                      
    NULL,                                      
    H5O__sdspace_shared_debug                  
}};

H5FL_EXTERN(H5S_extent_t);

H5FL_ARR_EXTERN(hsize_t);

static void *
H5O__sdspace_decode(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, unsigned H5_ATTR_UNUSED mesg_flags,
                    unsigned H5_ATTR_UNUSED *ioflags, size_t p_size, const uint8_t *p)
{
    const uint8_t *p_end = p + p_size - 1; 
    H5S_extent_t  *sdim  = NULL;           
    unsigned       flags, version;
    unsigned       i;
    void          *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    if (NULL == (sdim = H5FL_CALLOC(H5S_extent_t)))
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTALLOC, NULL, "dataspace structure allocation failed");
    sdim->type = H5S_NO_CLASS;

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    version = *p++;

    if (version < H5O_SDSPACE_VERSION_1 || version > H5O_SDSPACE_VERSION_2)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "wrong version number in dataspace message");
    sdim->version = version;

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    sdim->rank = *p++;

    if (sdim->rank > H5S_MAX_RANK)
        HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "simple dataspace dimensionality is too large");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    flags = *p++;

    
    if (version >= H5O_SDSPACE_VERSION_2) {
        if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        sdim->type = (H5S_class_t)*p++;

        if (sdim->type != H5S_SIMPLE && sdim->rank > 0)
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "invalid rank for scalar or NULL dataspace");
    }
    else {
        
        if (sdim->rank > 0)
            sdim->type = H5S_SIMPLE;
        else
            sdim->type = H5S_SCALAR;

        
        if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        p++;
    }

    
    if (version == H5O_SDSPACE_VERSION_1) {
        if (H5_IS_BUFFER_OVERFLOW(p, 4, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        p += 4;
    }

    
    if (sdim->rank > 0) {
        
        if (H5_IS_BUFFER_OVERFLOW(p, (H5F_sizeof_size(f) * sdim->rank), p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");

        
        if (NULL == (sdim->size = (hsize_t *)H5FL_ARR_MALLOC(hsize_t, (size_t)sdim->rank)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, NULL, "memory allocation failed");
        for (i = 0; i < sdim->rank; i++)
            H5F_DECODE_LENGTH(f, p, sdim->size[i]);

        if (flags & H5S_VALID_MAX) {
            
            if (H5_IS_BUFFER_OVERFLOW(p, (H5F_sizeof_size(f) * sdim->rank), p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");

            
            if (NULL == (sdim->max = (hsize_t *)H5FL_ARR_MALLOC(hsize_t, (size_t)sdim->rank)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_CANTALLOC, NULL, "memory allocation failed");
            for (i = 0; i < sdim->rank; i++) {
                H5F_DECODE_LENGTH(f, p, sdim->max[i]);
                if (sdim->size[i] > sdim->max[i])
                    HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL,
                                "dataspace dim %u size of %llu is greater than maxdim size of %llu", i,
                                (unsigned long long)sdim->size[i], (unsigned long long)sdim->max[i]);
            }
        }

        
    }

    
    if (sdim->type == H5S_NULL)
        sdim->nelem = 0;
    else {
        for (i = 0, sdim->nelem = 1; i < sdim->rank; i++)
            sdim->nelem *= sdim->size[i];
    }

    
    ret_value = (void *)sdim;

done:
    if (!ret_value && sdim) {
        H5S__extent_release(sdim);
        H5FL_FREE(H5S_extent_t, sdim);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__sdspace_encode(H5F_t *f, uint8_t *p, const void *_mesg)
{
    const H5S_extent_t *sdim  = (const H5S_extent_t *)_mesg;
    unsigned            flags = 0;
    unsigned            u; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(sdim);

    
    assert(sdim->version > 0);
    assert(sdim->type != H5S_NULL || sdim->version >= H5O_SDSPACE_VERSION_2);
    *p++ = (uint8_t)sdim->version;

    
    *p++ = (uint8_t)sdim->rank;

    
    if (sdim->max)
        flags |= H5S_VALID_MAX;
    *p++ = (uint8_t)flags;

    
    if (sdim->version > H5O_SDSPACE_VERSION_1)
        *p++ = (uint8_t)sdim->type;
    else {
        *p++ = 0; 
        *p++ = 0; 
        *p++ = 0; 
        *p++ = 0; 
        *p++ = 0; 
    }             

    
    if (H5S_SIMPLE == sdim->type) {
        
        if (sdim->rank > 0) {
            for (u = 0; u < sdim->rank; u++)
                H5F_ENCODE_LENGTH(f, p, sdim->size[u]);
            if (flags & H5S_VALID_MAX)
                for (u = 0; u < sdim->rank; u++)
                    H5F_ENCODE_LENGTH(f, p, sdim->max[u]);
        } 
    }     

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__sdspace_copy(const void *_mesg, void *_dest)
{
    const H5S_extent_t *mesg      = (const H5S_extent_t *)_mesg;
    H5S_extent_t       *dest      = (H5S_extent_t *)_dest;
    void               *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(mesg);
    if (!dest && NULL == (dest = H5FL_CALLOC(H5S_extent_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5S__extent_copy_real(dest, mesg, true) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCOPY, NULL, "can't copy extent");

    
    ret_value = dest;

done:
    if (NULL == ret_value)
        if (dest && NULL == _dest)
            dest = H5FL_FREE(H5S_extent_t, dest);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__sdspace_size(const H5F_t *f, const void *_mesg)
{
    const H5S_extent_t *space     = (const H5S_extent_t *)_mesg;
    size_t              ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    ret_value = (size_t)(1 +                                                  
                         1 +                                                  
                         1 +                                                  
                         1 +                                                  
                         ((space->version > H5O_SDSPACE_VERSION_1) ? 0 : 4)); 

    
    ret_value += space->rank * H5F_SIZEOF_SIZE(f);

    
    ret_value += space->max ? (space->rank * H5F_SIZEOF_SIZE(f)) : 0;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__sdspace_reset(void *_mesg)
{
    H5S_extent_t *mesg = (H5S_extent_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    H5S__extent_release(mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__sdspace_free(void *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(mesg);

    mesg = H5FL_FREE(H5S_extent_t, mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__sdspace_pre_copy_file(H5F_t H5_ATTR_UNUSED *file_src, const void *mesg_src, bool H5_ATTR_UNUSED *deleted,
                           const H5O_copy_t *cpy_info, void *_udata)
{
    const H5S_extent_t *src_space_extent = (const H5S_extent_t *)mesg_src; 
    H5D_copy_file_ud_t *udata            = (H5D_copy_file_ud_t *)_udata;   
    herr_t              ret_value        = SUCCEED;                        

    FUNC_ENTER_PACKAGE

    
    assert(file_src);
    assert(src_space_extent);
    assert(cpy_info);
    assert(cpy_info->file_dst);

    
    if (src_space_extent->version > H5O_sdspace_ver_bounds[H5F_HIGH_BOUND(cpy_info->file_dst)])
        HGOTO_ERROR(H5E_OHDR, H5E_BADRANGE, FAIL, "dataspace message version out of bounds");

    
    if (udata) {
        
        if (NULL == (udata->src_space_extent = H5FL_CALLOC(H5S_extent_t)))
            HGOTO_ERROR(H5E_DATASPACE, H5E_NOSPACE, FAIL, "dataspace extent allocation failed");

        
        if (H5S__extent_copy_real(udata->src_space_extent, src_space_extent, true) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTCOPY, FAIL, "can't copy extent");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__sdspace_debug(H5F_t H5_ATTR_UNUSED *f, const void *mesg, FILE *stream, int indent, int fwidth)
{
    const H5S_extent_t *sdim = (const H5S_extent_t *)mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(sdim);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %lu\n", indent, "", fwidth, "Rank:", (unsigned long)(sdim->rank));

    if (sdim->rank > 0) {
        unsigned u; 

        Rfprintf(stream, "%*s%-*s {", indent, "", fwidth, "Dim Size:");
        for (u = 0; u < sdim->rank; u++)
            Rfprintf(stream, "%s%" PRIuHSIZE, u ? ", " : "", sdim->size[u]);
        Rfprintf(stream, "}\n");

        Rfprintf(stream, "%*s%-*s ", indent, "", fwidth, "Dim Max:");
        if (sdim->max) {
            Rfprintf(stream, "{");
            for (u = 0; u < sdim->rank; u++) {
                if (H5S_UNLIMITED == sdim->max[u])
                    Rfprintf(stream, "%sUNLIM", u ? ", " : "");
                else
                    Rfprintf(stream, "%s%" PRIuHSIZE, u ? ", " : "", sdim->max[u]);
            } 
            Rfprintf(stream, "}\n");
        } 
        else
            Rfprintf(stream, "CONSTANT\n");
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
