/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FDmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FDlog.h"     
#include "H5FDpkg.h"     
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Pprivate.h"  

hid_t H5FD_LOG_id_g = H5I_INVALID_HID;

typedef struct H5FD_log_fapl_t {
    char              *logfile; 
    unsigned long long flags;   
    size_t buf_size; 
} H5FD_log_fapl_t;

static const char *flavors[] = {
    "H5FD_MEM_DEFAULT", "H5FD_MEM_SUPER", "H5FD_MEM_BTREE", "H5FD_MEM_DRAW",
    "H5FD_MEM_GHEAP",   "H5FD_MEM_LHEAP", "H5FD_MEM_OHDR",
};

typedef struct H5FD_log_t {
    H5FD_t  pub; 
    int     fd;  
    haddr_t eoa; 
    haddr_t eof; 
#ifndef H5_HAVE_PREADWRITE
    haddr_t        pos; 
    H5FD_file_op_t op;  
#endif                  
    bool ignore_disabled_file_locks;
    char filename[H5FD_MAX_FILENAME_LEN]; 
#ifndef H5_HAVE_WIN32_API
    
    dev_t device; 
    ino_t inode;  
#else
    
    DWORD nFileIndexLow;
    DWORD nFileIndexHigh;
    DWORD dwVolumeSerialNumber;

    HANDLE hFile; 
#endif 

    
    bool fam_to_single;

    
    unsigned char     *nread;               
    unsigned char     *nwrite;              
    unsigned char     *flavor;              
    unsigned long long total_read_ops;      
    unsigned long long total_write_ops;     
    unsigned long long total_seek_ops;      
    unsigned long long total_truncate_ops;  
    double             total_read_time;     
    double             total_write_time;    
    double             total_seek_time;     
    double             total_truncate_time; 
    size_t             iosize;              
    FILE              *logfp;               
    H5FD_log_fapl_t    fa;                  
} H5FD_log_t;

static void   *H5FD__log_fapl_get(H5FD_t *file);
static void   *H5FD__log_fapl_copy(const void *_old_fa);
static herr_t  H5FD__log_fapl_free(void *_fa);
static H5FD_t *H5FD__log_open(const char *name, unsigned flags, hid_t fapl_id, haddr_t maxaddr);
static herr_t  H5FD__log_close(H5FD_t *_file);
static int     H5FD__log_cmp(const H5FD_t *_f1, const H5FD_t *_f2);
static herr_t  H5FD__log_query(const H5FD_t *_f1, unsigned long *flags);
static haddr_t H5FD__log_alloc(H5FD_t *_file, H5FD_mem_t type, hid_t dxpl_id, hsize_t size);
static herr_t  H5FD__log_free(H5FD_t *_file, H5FD_mem_t type, hid_t dxpl_id, haddr_t addr, hsize_t size);
static haddr_t H5FD__log_get_eoa(const H5FD_t *_file, H5FD_mem_t type);
static herr_t  H5FD__log_set_eoa(H5FD_t *_file, H5FD_mem_t type, haddr_t addr);
static haddr_t H5FD__log_get_eof(const H5FD_t *_file, H5FD_mem_t type);
static herr_t  H5FD__log_get_handle(H5FD_t *_file, hid_t fapl, void **file_handle);
static herr_t  H5FD__log_read(H5FD_t *_file, H5FD_mem_t type, hid_t fapl_id, haddr_t addr, size_t size,
                              void *buf);
static herr_t  H5FD__log_write(H5FD_t *_file, H5FD_mem_t type, hid_t fapl_id, haddr_t addr, size_t size,
                               const void *buf);
static herr_t  H5FD__log_truncate(H5FD_t *_file, hid_t dxpl_id, bool closing);
static herr_t  H5FD__log_lock(H5FD_t *_file, bool rw);
static herr_t  H5FD__log_unlock(H5FD_t *_file);
static herr_t  H5FD__log_delete(const char *filename, hid_t fapl_id);

static const H5FD_class_t H5FD_log_g = {
    H5FD_CLASS_VERSION,      
    H5FD_LOG_VALUE,          
    "log",                   
    H5FD_MAXADDR,            
    H5F_CLOSE_WEAK,          
    NULL,                    
    NULL,                    
    NULL,                    
    NULL,                    
    sizeof(H5FD_log_fapl_t), 
    H5FD__log_fapl_get,      
    H5FD__log_fapl_copy,     
    H5FD__log_fapl_free,     
    0,                       
    NULL,                    
    NULL,                    
    H5FD__log_open,          
    H5FD__log_close,         
    H5FD__log_cmp,           
    H5FD__log_query,         
    NULL,                    
    H5FD__log_alloc,         
    H5FD__log_free,          
    H5FD__log_get_eoa,       
    H5FD__log_set_eoa,       
    H5FD__log_get_eof,       
    H5FD__log_get_handle,    
    H5FD__log_read,          
    H5FD__log_write,         
    NULL,                    
    NULL,                    
    NULL,                    
    NULL,                    
    NULL,                    
    H5FD__log_truncate,      
    H5FD__log_lock,          
    H5FD__log_unlock,        
    H5FD__log_delete,        
    NULL,                    
    H5FD_FLMAP_DICHOTOMY     
};

static const H5FD_log_fapl_t H5FD_log_default_config_g = {NULL, H5FD_LOG_LOC_IO | H5FD_LOG_ALLOC, 4096};

H5FL_DEFINE_STATIC(H5FD_log_t);

herr_t
H5FD__log_register(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (H5I_VFL != H5I_get_type(H5FD_LOG_id_g))
        if ((H5FD_LOG_id_g = H5FD_register(&H5FD_log_g, sizeof(H5FD_class_t), false)) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register log driver");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD__log_unregister(void)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    H5FD_LOG_id_g = H5I_INVALID_HID;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5Pset_fapl_log(hid_t fapl_id, const char *logfile, unsigned long long flags, size_t buf_size)
{
    H5FD_log_fapl_t fa;        
    H5P_genplist_t *plist;     
    herr_t          ret_value; 

    FUNC_ENTER_API(FAIL)

    
    memset(&fa, 0, sizeof(H5FD_log_fapl_t));

    
    if (NULL == (plist = H5P_object_verify(fapl_id, H5P_FILE_ACCESS, false)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");

    
    if (logfile != NULL && NULL == (fa.logfile = H5MM_xstrdup(logfile)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "unable to copy log file name");

    fa.flags    = flags;
    fa.buf_size = buf_size;
    ret_value   = H5P_set_driver(plist, H5FD_LOG, &fa, NULL);

done:
    if (fa.logfile)
        H5MM_free(fa.logfile);

    FUNC_LEAVE_API(ret_value)
} 

static void *
H5FD__log_fapl_get(H5FD_t *_file)
{
    H5FD_log_t *file      = (H5FD_log_t *)_file;
    void       *ret_value = NULL; 

    FUNC_ENTER_PACKAGE_NOERR

    
    ret_value = H5FD__log_fapl_copy(&(file->fa));

    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5FD__log_fapl_copy(const void *_old_fa)
{
    const H5FD_log_fapl_t *old_fa    = (const H5FD_log_fapl_t *)_old_fa;
    H5FD_log_fapl_t       *new_fa    = NULL; 
    void                  *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    assert(old_fa);

    
    if (NULL == (new_fa = (H5FD_log_fapl_t *)H5MM_calloc(sizeof(H5FD_log_fapl_t))))
        HGOTO_ERROR(H5E_FILE, H5E_CANTALLOC, NULL, "unable to allocate log file FAPL");

    
    H5MM_memcpy(new_fa, old_fa, sizeof(H5FD_log_fapl_t));

    
    if (old_fa->logfile != NULL)
        if (NULL == (new_fa->logfile = H5MM_strdup(old_fa->logfile)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "unable to allocate log file name");

    
    ret_value = new_fa;

done:
    if (NULL == ret_value)
        if (new_fa) {
            if (new_fa->logfile)
                new_fa->logfile = (char *)H5MM_xfree(new_fa->logfile);
            H5MM_free(new_fa);
        }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__log_fapl_free(void *_fa)
{
    H5FD_log_fapl_t *fa = (H5FD_log_fapl_t *)_fa;

    FUNC_ENTER_PACKAGE_NOERR

    
    if (fa->logfile)
        fa->logfile = (char *)H5MM_xfree(fa->logfile);
    H5MM_xfree(fa);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static H5FD_t *
H5FD__log_open(const char *name, unsigned flags, hid_t fapl_id, haddr_t maxaddr)
{
    H5FD_log_t            *file = NULL;
    H5P_genplist_t        *plist; 
    const H5FD_log_fapl_t *fa;    
    H5FD_log_fapl_t        default_fa = H5FD_log_default_config_g;
    int                    fd         = -1; 
    int                    o_flags;         
#ifdef H5_HAVE_WIN32_API
    struct _BY_HANDLE_FILE_INFORMATION fileinfo;
#endif
    H5_timer_t open_timer; 
    H5_timer_t stat_timer; 
    h5_stat_t  sb;
    H5FD_t    *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    HDcompile_assert(sizeof(HDoff_t) >= sizeof(size_t));

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "invalid file name");
    if (0 == maxaddr || HADDR_UNDEF == maxaddr)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, NULL, "bogus maxaddr");
    if (H5FD_ADDR_OVERFLOW(maxaddr))
        HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, NULL, "bogus maxaddr");

    
    H5_timer_init(&open_timer);
    H5_timer_init(&stat_timer);

    
    o_flags = (H5F_ACC_RDWR & flags) ? O_RDWR : O_RDONLY;
    if (H5F_ACC_TRUNC & flags)
        o_flags |= O_TRUNC;
    if (H5F_ACC_CREAT & flags)
        o_flags |= O_CREAT;
    if (H5F_ACC_EXCL & flags)
        o_flags |= O_EXCL;

    
    if (NULL == (plist = H5P_object_verify(fapl_id, H5P_FILE_ACCESS, true)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file access property list");
    if (NULL == (fa = (const H5FD_log_fapl_t *)H5P_peek_driver_info(plist))) {
        
        fa = &default_fa;
    }

    
    if (fa->flags & H5FD_LOG_TIME_OPEN)
        H5_timer_start(&open_timer);

    
    if ((fd = HDopen(name, o_flags, H5_POSIX_CREATE_MODE_RW)) < 0) {
        int myerrno = errno;

        HGOTO_ERROR(
            H5E_FILE, H5E_CANTOPENFILE, NULL,
            "unable to open file: name = '%s', errno = %d, error message = '%s', flags = %x, o_flags = %x",
            name, myerrno, strerror(myerrno), flags, (unsigned)o_flags);
    }

    
    if (fa->flags & H5FD_LOG_TIME_OPEN)
        H5_timer_stop(&open_timer);

    
    if (fa->flags & H5FD_LOG_TIME_STAT)
        H5_timer_start(&stat_timer);

    
    memset(&sb, 0, sizeof(h5_stat_t));
    if (HDfstat(fd, &sb) < 0)
        HSYS_GOTO_ERROR(H5E_FILE, H5E_BADFILE, NULL, "unable to fstat file");

    
    if (fa->flags & H5FD_LOG_TIME_STAT)
        H5_timer_stop(&stat_timer);

    
    if (NULL == (file = H5FL_CALLOC(H5FD_log_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "unable to allocate file struct");

    file->fd = fd;
    H5_CHECKED_ASSIGN(file->eof, haddr_t, sb.st_size, h5_stat_size_t);
#ifndef H5_HAVE_PREADWRITE
    file->pos = HADDR_UNDEF;
    file->op  = OP_UNKNOWN;
#endif 
#ifdef H5_HAVE_WIN32_API
    file->hFile = (HANDLE)_get_osfhandle(fd);
    if (INVALID_HANDLE_VALUE == file->hFile)
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, NULL, "unable to get Windows file handle");

    if (!GetFileInformationByHandle((HANDLE)file->hFile, &fileinfo))
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, NULL, "unable to get Windows file information");

    file->nFileIndexHigh       = fileinfo.nFileIndexHigh;
    file->nFileIndexLow        = fileinfo.nFileIndexLow;
    file->dwVolumeSerialNumber = fileinfo.dwVolumeSerialNumber;
#else  
    file->device = sb.st_dev;
    file->inode  = sb.st_ino;
#endif 

    
    strncpy(file->filename, name, sizeof(file->filename) - 1);
    file->filename[sizeof(file->filename) - 1] = '\0';

    
    file->fa.flags = fa->flags;
    if (fa->logfile)
        file->fa.logfile = H5MM_strdup(fa->logfile);
    else
        file->fa.logfile = NULL;
    file->fa.buf_size = fa->buf_size;

    
    if (file->fa.flags != 0) {
        
        file->iosize = fa->buf_size;
        if (file->fa.flags & H5FD_LOG_FILE_READ) {
            file->nread = (unsigned char *)H5MM_calloc(file->iosize);
            assert(file->nread);
        }
        if (file->fa.flags & H5FD_LOG_FILE_WRITE) {
            file->nwrite = (unsigned char *)H5MM_calloc(file->iosize);
            assert(file->nwrite);
        }
        if (file->fa.flags & H5FD_LOG_FLAVOR) {
            file->flavor = (unsigned char *)H5MM_calloc(file->iosize);
            assert(file->flavor);
        }

        
        if (fa->logfile)
            file->logfp = fopen(fa->logfile, "w");
        else
            file->logfp = Rstderr;

        
        if (file->fa.flags & H5FD_LOG_TIME_OPEN) {
            H5_timevals_t open_times; 

            H5_timer_get_times(open_timer, &open_times);
            Rfprintf(file->logfp, "Open took: (%f s)\n", open_times.elapsed);
        }
        if (file->fa.flags & H5FD_LOG_TIME_STAT) {
            H5_timevals_t stat_times; 

            H5_timer_get_times(stat_timer, &stat_times);
            Rfprintf(file->logfp, "Stat took: (%f s)\n", stat_times.elapsed);
        }
    }

    
    if (H5FD_ignore_disabled_file_locks_p != FAIL)
        
        file->ignore_disabled_file_locks = H5FD_ignore_disabled_file_locks_p;
    else {
        
        if (H5P_get(plist, H5F_ACS_IGNORE_DISABLED_FILE_LOCKS_NAME, &file->ignore_disabled_file_locks) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTGET, NULL, "can't get ignore disabled file locks property");
    }

    
    if (H5P_FILE_ACCESS_DEFAULT != fapl_id) {
        
        if (H5P_exist_plist(plist, H5F_ACS_FAMILY_TO_SINGLE_NAME) > 0)
            if (H5P_get(plist, H5F_ACS_FAMILY_TO_SINGLE_NAME, &file->fam_to_single) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTGET, NULL, "can't get property of changing family to single");
    }

    
    ret_value = (H5FD_t *)file;

done:
    if (NULL == ret_value) {
        if (fd >= 0)
            HDclose(fd);
        if (file)
            file = H5FL_FREE(H5FD_log_t, file);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__log_close(H5FD_t *_file)
{
    H5FD_log_t *file = (H5FD_log_t *)_file;
    H5_timer_t  close_timer;         
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(file);

    
    H5_timer_init(&close_timer);

    
    if (file->fa.flags & H5FD_LOG_TIME_CLOSE)
        H5_timer_start(&close_timer);

    
    if (HDclose(file->fd) < 0)
        HSYS_GOTO_ERROR(H5E_IO, H5E_CANTCLOSEFILE, FAIL, "unable to close file");

    
    if (file->fa.flags & H5FD_LOG_TIME_CLOSE)
        H5_timer_stop(&close_timer);

    
    if (file->fa.flags != 0) {
        haddr_t       addr;
        haddr_t       last_addr;
        unsigned char last_val;

        if (file->fa.flags & H5FD_LOG_TIME_CLOSE) {
            H5_timevals_t close_times; 

            H5_timer_get_times(close_timer, &close_times);
            Rfprintf(file->logfp, "Close took: (%f s)\n", close_times.elapsed);
        }

        
        if (file->fa.flags & H5FD_LOG_NUM_READ)
            Rfprintf(file->logfp, "Total number of read operations: %llu\n", file->total_read_ops);
        if (file->fa.flags & H5FD_LOG_NUM_WRITE)
            Rfprintf(file->logfp, "Total number of write operations: %llu\n", file->total_write_ops);
        if (file->fa.flags & H5FD_LOG_NUM_SEEK)
            Rfprintf(file->logfp, "Total number of seek operations: %llu\n", file->total_seek_ops);
        if (file->fa.flags & H5FD_LOG_NUM_TRUNCATE)
            Rfprintf(file->logfp, "Total number of truncate operations: %llu\n", file->total_truncate_ops);

        
        if (file->fa.flags & H5FD_LOG_TIME_READ)
            Rfprintf(file->logfp, "Total time in read operations: %f s\n", file->total_read_time);
        if (file->fa.flags & H5FD_LOG_TIME_WRITE)
            Rfprintf(file->logfp, "Total time in write operations: %f s\n", file->total_write_time);
        if (file->fa.flags & H5FD_LOG_TIME_SEEK)
            Rfprintf(file->logfp, "Total time in seek operations: %f s\n", file->total_seek_time);
        if (file->fa.flags & H5FD_LOG_TIME_TRUNCATE)
            Rfprintf(file->logfp, "Total time in truncate operations: %f s\n", file->total_truncate_time);

        
        if (file->fa.flags & H5FD_LOG_FILE_WRITE) {
            Rfprintf(file->logfp, "Dumping write I/O information:\n");
            last_val  = file->nwrite[0];
            last_addr = 0;
            addr      = 1;
            while (addr < file->eoa) {
                if (file->nwrite[addr] != last_val) {
                    Rfprintf(file->logfp,
                            "\tAddr %10" PRIuHADDR "-%10" PRIuHADDR " (%10lu bytes) written to %3d times\n",
                            last_addr, (addr - 1), (unsigned long)(addr - last_addr), (int)last_val);
                    last_val  = file->nwrite[addr];
                    last_addr = addr;
                }
                addr++;
            }
            Rfprintf(file->logfp,
                    "\tAddr %10" PRIuHADDR "-%10" PRIuHADDR " (%10lu bytes) written to %3d times\n",
                    last_addr, (addr - 1), (unsigned long)(addr - last_addr), (int)last_val);
        }

        
        if (file->fa.flags & H5FD_LOG_FILE_READ) {
            Rfprintf(file->logfp, "Dumping read I/O information:\n");
            last_val  = file->nread[0];
            last_addr = 0;
            addr      = 1;
            while (addr < file->eoa) {
                if (file->nread[addr] != last_val) {
                    Rfprintf(file->logfp,
                            "\tAddr %10" PRIuHADDR "-%10" PRIuHADDR " (%10lu bytes) read from %3d times\n",
                            last_addr, (addr - 1), (unsigned long)(addr - last_addr), (int)last_val);
                    last_val  = file->nread[addr];
                    last_addr = addr;
                }
                addr++;
            }
            Rfprintf(file->logfp,
                    "\tAddr %10" PRIuHADDR "-%10" PRIuHADDR " (%10lu bytes) read from %3d times\n", last_addr,
                    (addr - 1), (unsigned long)(addr - last_addr), (int)last_val);
        }

        
        if (file->fa.flags & H5FD_LOG_FLAVOR) {
            Rfprintf(file->logfp, "Dumping I/O flavor information:\n");
            last_val  = file->flavor[0];
            last_addr = 0;
            addr      = 1;
            while (addr < file->eoa) {
                if (file->flavor[addr] != last_val) {
                    Rfprintf(file->logfp,
                            "\tAddr %10" PRIuHADDR "-%10" PRIuHADDR " (%10lu bytes) flavor is %s\n",
                            last_addr, (addr - 1), (unsigned long)(addr - last_addr), flavors[last_val]);
                    last_val  = file->flavor[addr];
                    last_addr = addr;
                }
                addr++;
            }
            Rfprintf(file->logfp, "\tAddr %10" PRIuHADDR "-%10" PRIuHADDR " (%10lu bytes) flavor is %s\n",
                    last_addr, (addr - 1), (unsigned long)(addr - last_addr), flavors[last_val]);
        }

        
        if (file->fa.flags & H5FD_LOG_FILE_WRITE)
            file->nwrite = (unsigned char *)H5MM_xfree(file->nwrite);
        if (file->fa.flags & H5FD_LOG_FILE_READ)
            file->nread = (unsigned char *)H5MM_xfree(file->nread);
        if (file->fa.flags & H5FD_LOG_FLAVOR)
            file->flavor = (unsigned char *)H5MM_xfree(file->flavor);
        if (file->logfp != Rstderr)
            fclose(file->logfp);
    } 

    if (file->fa.logfile)
        file->fa.logfile = (char *)H5MM_xfree(file->fa.logfile);

    
    file = H5FL_FREE(H5FD_log_t, file);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5FD__log_cmp(const H5FD_t *_f1, const H5FD_t *_f2)
{
    const H5FD_log_t *f1        = (const H5FD_log_t *)_f1;
    const H5FD_log_t *f2        = (const H5FD_log_t *)_f2;
    int               ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

#ifdef H5_HAVE_WIN32_API
    if (f1->dwVolumeSerialNumber < f2->dwVolumeSerialNumber)
        HGOTO_DONE(-1);
    if (f1->dwVolumeSerialNumber > f2->dwVolumeSerialNumber)
        HGOTO_DONE(1);

    if (f1->nFileIndexHigh < f2->nFileIndexHigh)
        HGOTO_DONE(-1);
    if (f1->nFileIndexHigh > f2->nFileIndexHigh)
        HGOTO_DONE(1);

    if (f1->nFileIndexLow < f2->nFileIndexLow)
        HGOTO_DONE(-1);
    if (f1->nFileIndexLow > f2->nFileIndexLow)
        HGOTO_DONE(1);
#else
    if (f1->device < f2->device)
        HGOTO_DONE(-1);
    if (f1->device > f2->device)
        HGOTO_DONE(1);

    if (f1->inode < f2->inode)
        HGOTO_DONE(-1);
    if (f1->inode > f2->inode)
        HGOTO_DONE(1);

#endif

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__log_query(const H5FD_t *_file, unsigned long *flags )
{
    const H5FD_log_t *file = (const H5FD_log_t *)_file;

    FUNC_ENTER_PACKAGE_NOERR

    
    if (flags) {
        *flags = 0;
        *flags |= H5FD_FEAT_AGGREGATE_METADATA;  
        *flags |= H5FD_FEAT_ACCUMULATE_METADATA; 
        *flags |= H5FD_FEAT_DATA_SIEVE; 
        *flags |= H5FD_FEAT_AGGREGATE_SMALLDATA; 
        *flags |= H5FD_FEAT_POSIX_COMPAT_HANDLE; 
        *flags |=
            H5FD_FEAT_SUPPORTS_SWMR_IO; 
        *flags |= H5FD_FEAT_DEFAULT_VFD_COMPATIBLE; 

        
        if (file && file->fam_to_single)
            *flags |= H5FD_FEAT_IGNORE_DRVRINFO; 
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static haddr_t
H5FD__log_alloc(H5FD_t *_file, H5FD_mem_t type, hid_t H5_ATTR_UNUSED dxpl_id, hsize_t size)
{
    H5FD_log_t *file = (H5FD_log_t *)_file;
    haddr_t     addr;
    haddr_t     ret_value = HADDR_UNDEF; 

    FUNC_ENTER_PACKAGE_NOERR

    
    addr = file->eoa;

    
    file->eoa = addr + size;

    
    if (file->fa.flags != 0) {
        if (file->fa.flags & H5FD_LOG_FLAVOR) {
            assert(addr < file->iosize);
            H5_CHECK_OVERFLOW(size, hsize_t, size_t);
            memset(&file->flavor[addr], (int)type, (size_t)size);
        }

        if (file->fa.flags & H5FD_LOG_ALLOC)
            Rfprintf(file->logfp,
                    "%10" PRIuHADDR "-%10" PRIuHADDR " (%10" PRIuHSIZE " bytes) (%s) Allocated\n", addr,
                    (haddr_t)((addr + size) - 1), size, flavors[type]);
    }

    
    ret_value = addr;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__log_free(H5FD_t *_file, H5FD_mem_t type, hid_t H5_ATTR_UNUSED dxpl_id, haddr_t addr, hsize_t size)
{
    H5FD_log_t *file = (H5FD_log_t *)_file;

    FUNC_ENTER_PACKAGE_NOERR

    if (file->fa.flags != 0) {
        
        if (file->fa.flags & H5FD_LOG_FLAVOR) {
            assert(addr < file->iosize);
            H5_CHECK_OVERFLOW(size, hsize_t, size_t);
            memset(&file->flavor[addr], H5FD_MEM_DEFAULT, (size_t)size);
        }

        
        if (file->fa.flags & H5FD_LOG_FREE)
            Rfprintf(file->logfp, "%10" PRIuHADDR "-%10" PRIuHADDR " (%10" PRIuHSIZE " bytes) (%s) Freed\n",
                    addr, (haddr_t)((addr + size) - 1), size, flavors[type]);
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static haddr_t
H5FD__log_get_eoa(const H5FD_t *_file, H5FD_mem_t H5_ATTR_UNUSED type)
{
    const H5FD_log_t *file = (const H5FD_log_t *)_file;

    FUNC_ENTER_PACKAGE_NOERR

    FUNC_LEAVE_NOAPI(file->eoa)
} 

static herr_t
H5FD__log_set_eoa(H5FD_t *_file, H5FD_mem_t type, haddr_t addr)
{
    H5FD_log_t *file = (H5FD_log_t *)_file;

    FUNC_ENTER_PACKAGE_NOERR

    if (file->fa.flags != 0) {
        
        if (H5_addr_gt(addr, file->eoa) && H5_addr_gt(addr, 0)) {
            hsize_t size = addr - file->eoa;

            
            if (file->fa.flags & H5FD_LOG_FLAVOR) {
                assert(addr < file->iosize);
                H5_CHECK_OVERFLOW(size, hsize_t, size_t);
                memset(&file->flavor[file->eoa], (int)type, (size_t)size);
            }

            
            if (file->fa.flags & H5FD_LOG_ALLOC)
                Rfprintf(file->logfp,
                        "%10" PRIuHADDR "-%10" PRIuHADDR " (%10" PRIuHSIZE " bytes) (%s) Allocated\n",
                        file->eoa, addr, size, flavors[type]);
        }

        
        if (H5_addr_lt(addr, file->eoa) && H5_addr_gt(addr, 0)) {
            hsize_t size = file->eoa - addr;

            
            if (file->fa.flags & H5FD_LOG_FLAVOR) {
                assert((addr + size) < file->iosize);
                H5_CHECK_OVERFLOW(size, hsize_t, size_t);
                memset(&file->flavor[addr], H5FD_MEM_DEFAULT, (size_t)size);
            }

            
            if (file->fa.flags & H5FD_LOG_FREE)
                Rfprintf(file->logfp,
                        "%10" PRIuHADDR "-%10" PRIuHADDR " (%10" PRIuHSIZE " bytes) (%s) Freed\n", file->eoa,
                        addr, size, flavors[type]);
        }
    }

    file->eoa = addr;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static haddr_t
H5FD__log_get_eof(const H5FD_t *_file, H5FD_mem_t H5_ATTR_UNUSED type)
{
    const H5FD_log_t *file = (const H5FD_log_t *)_file;

    FUNC_ENTER_PACKAGE_NOERR

    FUNC_LEAVE_NOAPI(file->eof)
} 

static herr_t
H5FD__log_get_handle(H5FD_t *_file, hid_t H5_ATTR_UNUSED fapl, void **file_handle)
{
    H5FD_log_t *file      = (H5FD_log_t *)_file;
    herr_t      ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    if (!file_handle)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file handle not valid");

    *file_handle = &(file->fd);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__log_read(H5FD_t *_file, H5FD_mem_t type, hid_t H5_ATTR_UNUSED dxpl_id, haddr_t addr, size_t size,
               void *buf )
{
    H5FD_log_t   *file      = (H5FD_log_t *)_file;
    size_t        orig_size = size; 
    haddr_t       orig_addr = addr;
    H5_timer_t    read_timer; 
    H5_timevals_t read_times; 
    HDoff_t       offset    = (HDoff_t)addr;
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(file && file->pub.cls);
    assert(buf);

    
    H5_timer_init(&read_timer);

    
    if (!H5_addr_defined(addr))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "addr undefined, addr = %llu", (unsigned long long)addr);
    if (H5FD_REGION_OVERFLOW(addr, size))
        HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, FAIL, "addr overflow, addr = %llu", (unsigned long long)addr);

    
    if (file->fa.flags != 0) {
        size_t  tmp_size = size;
        haddr_t tmp_addr = addr;

        
        if (file->fa.flags & H5FD_LOG_FILE_READ) {
            assert((addr + size) < file->iosize);
            while (tmp_size-- > 0)
                file->nread[tmp_addr++]++;
        }
    }

#ifndef H5_HAVE_PREADWRITE
    
    if (addr != file->pos || OP_READ != file->op) {
        H5_timer_t    seek_timer; 
        H5_timevals_t seek_times; 

        
        H5_timer_init(&seek_timer);

        
        if (file->fa.flags & H5FD_LOG_TIME_SEEK)
            H5_timer_start(&seek_timer);

        if (HDlseek(file->fd, (HDoff_t)addr, SEEK_SET) < 0)
            HSYS_GOTO_ERROR(H5E_IO, H5E_SEEKERROR, FAIL, "unable to seek to proper position");

        
        if (file->fa.flags & H5FD_LOG_TIME_SEEK)
            H5_timer_stop(&seek_timer);

        
        if (file->fa.flags & H5FD_LOG_NUM_SEEK)
            file->total_seek_ops++;

        
        if (file->fa.flags & H5FD_LOG_TIME_SEEK) {
            H5_timer_get_times(seek_timer, &seek_times);
            file->total_seek_time += seek_times.elapsed;
        }

        
        if (file->fa.flags & H5FD_LOG_LOC_SEEK) {
            Rfprintf(file->logfp, "Seek: From %10" PRIuHADDR " To %10" PRIuHADDR, file->pos, addr);

            
            if (file->fa.flags & H5FD_LOG_TIME_SEEK)
                Rfprintf(file->logfp, " (%fs @ %f)\n", seek_times.elapsed, seek_timer.initial.elapsed);
            else
                Rfprintf(file->logfp, "\n");
        }
    }
#endif 

    
    if (file->fa.flags & H5FD_LOG_TIME_READ)
        H5_timer_start(&read_timer);

    
    while (size > 0) {
        h5_posix_io_t     bytes_in   = 0;  
        h5_posix_io_ret_t bytes_read = -1; 

        
        if (size > H5_POSIX_MAX_IO_BYTES)
            bytes_in = H5_POSIX_MAX_IO_BYTES;
        else
            bytes_in = (h5_posix_io_t)size;

        do {
#ifdef H5_HAVE_PREADWRITE
            bytes_read = HDpread(file->fd, buf, bytes_in, offset);
            if (bytes_read > 0)
                offset += bytes_read;
#else
            bytes_read = HDread(file->fd, buf, bytes_in);
#endif 
        } while (-1 == bytes_read && EINTR == errno);

        if (-1 == bytes_read) { 
            int    myerrno = errno;
            time_t mytime  = time(NULL);

            offset = HDlseek(file->fd, 0, SEEK_CUR);

            if (file->fa.flags & H5FD_LOG_LOC_READ)
                Rfprintf(file->logfp, "Error! Reading: %10" PRIuHADDR "-%10" PRIuHADDR " (%10zu bytes)\n",
                        orig_addr, (orig_addr + orig_size) - 1, orig_size);

            HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL,
                        "file read failed: time = %s, filename = '%s', file descriptor = %d, errno = %d, "
                        "error message = '%s', buf = %p, total read size = %llu, bytes this sub-read = %llu, "
                        "bytes actually read = %llu, offset = %llu",
                        ctime(&mytime), file->filename, file->fd, myerrno, strerror(myerrno), buf,
                        (unsigned long long)size, (unsigned long long)bytes_in,
                        (unsigned long long)bytes_read, (unsigned long long)offset);
        }

        if (0 == bytes_read) {
            
            memset(buf, 0, size);
            break;
        }

        assert(bytes_read >= 0);
        assert((size_t)bytes_read <= size);

        size -= (size_t)bytes_read;
        addr += (haddr_t)bytes_read;
        buf = (char *)buf + bytes_read;
    }

    
    if (file->fa.flags & H5FD_LOG_TIME_READ)
        H5_timer_stop(&read_timer);

    
    if (file->fa.flags & H5FD_LOG_NUM_READ)
        file->total_read_ops++;

    
    if (file->fa.flags & H5FD_LOG_TIME_READ) {
        H5_timer_get_times(read_timer, &read_times);
        file->total_read_time += read_times.elapsed;
    }

    
    if (file->fa.flags & H5FD_LOG_LOC_READ) {
        Rfprintf(file->logfp, "%10" PRIuHADDR "-%10" PRIuHADDR " (%10zu bytes) (%s) Read", orig_addr,
                (orig_addr + orig_size) - 1, orig_size, flavors[type]);

        
        if (file->flavor) {
            assert(type == H5FD_MEM_DEFAULT || type == (H5FD_mem_t)file->flavor[orig_addr] ||
                   (H5FD_mem_t)file->flavor[orig_addr] == H5FD_MEM_DEFAULT);
            assert(type == H5FD_MEM_DEFAULT ||
                   type == (H5FD_mem_t)file->flavor[(orig_addr + orig_size) - 1] ||
                   (H5FD_mem_t)file->flavor[(orig_addr + orig_size) - 1] == H5FD_MEM_DEFAULT);
        }

        
        if (file->fa.flags & H5FD_LOG_TIME_READ)
            Rfprintf(file->logfp, " (%fs @ %f)\n", read_times.elapsed, read_timer.initial.elapsed);
        else
            Rfprintf(file->logfp, "\n");
    }

#ifndef H5_HAVE_PREADWRITE
    
    file->pos = addr;
    file->op  = OP_READ;
#endif 

done:
#ifndef H5_HAVE_PREADWRITE
    if (ret_value < 0) {
        
        file->pos = HADDR_UNDEF;
        file->op  = OP_UNKNOWN;
    }
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__log_write(H5FD_t *_file, H5FD_mem_t type, hid_t H5_ATTR_UNUSED dxpl_id, haddr_t addr, size_t size,
                const void *buf)
{
    H5FD_log_t   *file      = (H5FD_log_t *)_file;
    size_t        orig_size = size; 
    haddr_t       orig_addr = addr;
    H5_timer_t    write_timer; 
    H5_timevals_t write_times; 
    HDoff_t       offset    = (HDoff_t)addr;
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(file && file->pub.cls);
    assert(size > 0);
    assert(buf);

    
    H5_timer_init(&write_timer);

    
    if (file->flavor) {
        assert(type == H5FD_MEM_DEFAULT || type == (H5FD_mem_t)file->flavor[addr] ||
               (H5FD_mem_t)file->flavor[addr] == H5FD_MEM_DEFAULT);
        assert(type == H5FD_MEM_DEFAULT || type == (H5FD_mem_t)file->flavor[(addr + size) - 1] ||
               (H5FD_mem_t)file->flavor[(addr + size) - 1] == H5FD_MEM_DEFAULT);
    }

    
    if (!H5_addr_defined(addr))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "addr undefined, addr = %llu", (unsigned long long)addr);
    if (H5FD_REGION_OVERFLOW(addr, size))
        HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, FAIL, "addr overflow, addr = %llu, size = %llu",
                    (unsigned long long)addr, (unsigned long long)size);

    
    if (file->fa.flags & H5FD_LOG_FILE_WRITE) {
        size_t  tmp_size = size;
        haddr_t tmp_addr = addr;

        
        assert((addr + size) < file->iosize);
        while (tmp_size-- > 0)
            file->nwrite[tmp_addr++]++;
    }

#ifndef H5_HAVE_PREADWRITE
    
    if (addr != file->pos || OP_WRITE != file->op) {
        H5_timer_t    seek_timer; 
        H5_timevals_t seek_times; 

        
        H5_timer_init(&seek_timer);

        
        if (file->fa.flags & H5FD_LOG_TIME_SEEK)
            H5_timer_start(&seek_timer);

        if (HDlseek(file->fd, (HDoff_t)addr, SEEK_SET) < 0)
            HSYS_GOTO_ERROR(H5E_IO, H5E_SEEKERROR, FAIL, "unable to seek to proper position");

        
        if (file->fa.flags & H5FD_LOG_TIME_SEEK)
            H5_timer_stop(&seek_timer);

        
        if (file->fa.flags & H5FD_LOG_NUM_SEEK)
            file->total_seek_ops++;

        
        if (file->fa.flags & H5FD_LOG_TIME_SEEK) {
            H5_timer_get_times(seek_timer, &seek_times);
            file->total_seek_time += seek_times.elapsed;
        }

        
        if (file->fa.flags & H5FD_LOG_LOC_SEEK) {
            Rfprintf(file->logfp, "Seek: From %10" PRIuHADDR " To %10" PRIuHADDR, file->pos, addr);

            
            if (file->fa.flags & H5FD_LOG_TIME_SEEK)
                Rfprintf(file->logfp, " (%fs @ %f)\n", seek_times.elapsed, seek_timer.initial.elapsed);
            else
                Rfprintf(file->logfp, "\n");
        }
    }
#endif 

    
    if (file->fa.flags & H5FD_LOG_TIME_WRITE)
        H5_timer_start(&write_timer);

    
    while (size > 0) {
        h5_posix_io_t     bytes_in    = 0;  
        h5_posix_io_ret_t bytes_wrote = -1; 

        
        if (size > H5_POSIX_MAX_IO_BYTES)
            bytes_in = H5_POSIX_MAX_IO_BYTES;
        else
            bytes_in = (h5_posix_io_t)size;

        do {
#ifdef H5_HAVE_PREADWRITE
            bytes_wrote = HDpwrite(file->fd, buf, bytes_in, offset);
            if (bytes_wrote > 0)
                offset += bytes_wrote;
#else
            bytes_wrote = HDwrite(file->fd, buf, bytes_in);
#endif 
        } while (-1 == bytes_wrote && EINTR == errno);

        if (-1 == bytes_wrote) { 
            int    myerrno = errno;
            time_t mytime  = time(NULL);

            offset = HDlseek(file->fd, 0, SEEK_CUR);

            if (file->fa.flags & H5FD_LOG_LOC_WRITE)
                Rfprintf(file->logfp, "Error! Writing: %10" PRIuHADDR "-%10" PRIuHADDR " (%10zu bytes)\n",
                        orig_addr, (orig_addr + orig_size) - 1, orig_size);

            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL,
                        "file write failed: time = %s, filename = '%s', file descriptor = %d, errno = %d, "
                        "error message = '%s', buf = %p, total write size = %llu, bytes this sub-write = "
                        "%llu, bytes actually written = %llu, offset = %llu",
                        ctime(&mytime), file->filename, file->fd, myerrno, strerror(myerrno), buf,
                        (unsigned long long)size, (unsigned long long)bytes_in,
                        (unsigned long long)bytes_wrote, (unsigned long long)offset);
        } 

        assert(bytes_wrote > 0);
        assert((size_t)bytes_wrote <= size);

        size -= (size_t)bytes_wrote;
        addr += (haddr_t)bytes_wrote;
        buf = (const char *)buf + bytes_wrote;
    } 

    
    if (file->fa.flags & H5FD_LOG_TIME_WRITE)
        H5_timer_stop(&write_timer);

    
    if (file->fa.flags & H5FD_LOG_NUM_WRITE)
        file->total_write_ops++;

    
    if (file->fa.flags & H5FD_LOG_TIME_WRITE) {
        H5_timer_get_times(write_timer, &write_times);
        file->total_write_time += write_times.elapsed;
    }

    
    if (file->fa.flags & H5FD_LOG_LOC_WRITE) {
        Rfprintf(file->logfp, "%10" PRIuHADDR "-%10" PRIuHADDR " (%10zu bytes) (%s) Written", orig_addr,
                (orig_addr + orig_size) - 1, orig_size, flavors[type]);

        
        if (file->fa.flags & H5FD_LOG_FLAVOR) {
            if ((H5FD_mem_t)file->flavor[orig_addr] == H5FD_MEM_DEFAULT) {
                memset(&file->flavor[orig_addr], (int)type, orig_size);
                Rfprintf(file->logfp, " (fresh)");
            }
        }

        
        if (file->fa.flags & H5FD_LOG_TIME_WRITE)
            Rfprintf(file->logfp, " (%fs @ %f)\n", write_times.elapsed, write_timer.initial.elapsed);
        else
            Rfprintf(file->logfp, "\n");
    }

    
#ifndef H5_HAVE_PREADWRITE
    file->pos = addr;
    file->op  = OP_WRITE;
#endif 
    if (addr > file->eof)
        file->eof = addr;

done:
#ifndef H5_HAVE_PREADWRITE
    if (ret_value < 0) {
        
        file->pos = HADDR_UNDEF;
        file->op  = OP_UNKNOWN;
    }
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__log_truncate(H5FD_t *_file, hid_t H5_ATTR_UNUSED dxpl_id, bool H5_ATTR_UNUSED closing)
{
    H5FD_log_t *file      = (H5FD_log_t *)_file;
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(file);

    
    if (!H5_addr_eq(file->eoa, file->eof)) {
        H5_timer_t    trunc_timer; 
        H5_timevals_t trunc_times; 

        
        H5_timer_init(&trunc_timer);

        
        if (file->fa.flags & H5FD_LOG_TIME_TRUNCATE)
            H5_timer_start(&trunc_timer);

#ifdef H5_HAVE_WIN32_API
        {
            LARGE_INTEGER li;       
            DWORD         dwPtrLow; 

            
            li.QuadPart = (LONGLONG)file->eoa;

            
            dwPtrLow = SetFilePointer(file->hFile, li.LowPart, &li.HighPart, FILE_BEGIN);
            if (INVALID_SET_FILE_POINTER == dwPtrLow) {
                DWORD dwError; 

                dwError = GetLastError();
                if (dwError != NO_ERROR)
                    HGOTO_ERROR(H5E_FILE, H5E_FILEOPEN, FAIL, "unable to set file pointer");
            }

            if (0 == SetEndOfFile(file->hFile))
                HGOTO_ERROR(H5E_IO, H5E_SEEKERROR, FAIL, "unable to extend file properly");
        }
#else  
        
        if (-1 == HDftruncate(file->fd, (HDoff_t)file->eoa))
            HSYS_GOTO_ERROR(H5E_IO, H5E_SEEKERROR, FAIL, "unable to extend file properly");
#endif 

        
        if (file->fa.flags & H5FD_LOG_TIME_TRUNCATE)
            H5_timer_stop(&trunc_timer);

        
        if (file->fa.flags & H5FD_LOG_NUM_TRUNCATE)
            file->total_truncate_ops++;

        
        if (file->fa.flags & H5FD_LOG_TIME_TRUNCATE) {
            H5_timer_get_times(trunc_timer, &trunc_times);
            file->total_truncate_time += trunc_times.elapsed;
        }

        
        if (file->fa.flags & H5FD_LOG_TRUNCATE) {
            Rfprintf(file->logfp, "Truncate: To %10" PRIuHADDR, file->eoa);

            
            if (file->fa.flags & H5FD_LOG_TIME_TRUNCATE)
                Rfprintf(file->logfp, " (%fs @ %f)\n", trunc_times.elapsed, trunc_timer.initial.elapsed);
            else
                Rfprintf(file->logfp, "\n");
        }

        
        file->eof = file->eoa;

#ifndef H5_HAVE_PREADWRITE
        
        file->pos = HADDR_UNDEF;
        file->op  = OP_UNKNOWN;
#endif 
    }  

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__log_lock(H5FD_t *_file, bool rw)
{
    H5FD_log_t *file = (H5FD_log_t *)_file; 
    int         lock_flags;                 
    herr_t      ret_value = SUCCEED;        

    FUNC_ENTER_PACKAGE

    
    assert(file);

    
    lock_flags = rw ? LOCK_EX : LOCK_SH;

    
    if (HDflock(file->fd, lock_flags | LOCK_NB) < 0) {
        if (file->ignore_disabled_file_locks && ENOSYS == errno) {
            
            errno = 0;
        }
        else
            HSYS_GOTO_ERROR(H5E_VFL, H5E_CANTLOCKFILE, FAIL, "unable to lock file");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__log_unlock(H5FD_t *_file)
{
    H5FD_log_t *file      = (H5FD_log_t *)_file; 
    herr_t      ret_value = SUCCEED;             

    FUNC_ENTER_PACKAGE

    assert(file);

    if (HDflock(file->fd, LOCK_UN) < 0) {
        if (file->ignore_disabled_file_locks && ENOSYS == errno) {
            
            errno = 0;
        }
        else
            HSYS_GOTO_ERROR(H5E_VFL, H5E_CANTUNLOCKFILE, FAIL, "unable to unlock file");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__log_delete(const char *filename, hid_t H5_ATTR_UNUSED fapl_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(filename);

    if (HDremove(filename) < 0)
        HSYS_GOTO_ERROR(H5E_VFL, H5E_CANTDELETEFILE, FAIL, "unable to delete file");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
