/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Iprivate.h" 
#include "H5Tpkg.h"     

H5T_sign_t
H5Tget_sign(hid_t type_id)
{
    H5T_t     *dt = NULL;
    H5T_sign_t ret_value;

    FUNC_ENTER_API(H5T_SGN_ERROR)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5T_SGN_ERROR, "not an integer datatype");

    ret_value = H5T_get_sign(dt);

done:
    FUNC_LEAVE_API(ret_value)
}

H5T_sign_t
H5T_get_sign(H5T_t const *dt)
{
    H5T_sign_t ret_value = H5T_SGN_ERROR; 

    FUNC_ENTER_NOAPI(H5T_SGN_ERROR)

    assert(dt);

    
    while (dt->shared->parent)
        dt = dt->shared->parent;

    
    if (H5T_INTEGER != dt->shared->type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, H5T_SGN_ERROR, "operation not defined for datatype class");

    
    ret_value = dt->shared->u.atomic.u.i.sign;

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

herr_t
H5Tset_sign(hid_t type_id, H5T_sign_t sign)
{
    H5T_t *dt        = NULL;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an integer datatype");
    if (H5T_STATE_TRANSIENT != dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTINIT, FAIL, "datatype is read-only");
    if (sign < H5T_SGN_NONE || sign >= H5T_NSGN)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "illegal sign type");
    if (H5T_ENUM == dt->shared->type && dt->shared->u.enumer.nmembs > 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "operation not allowed after members are defined");
    while (dt->shared->parent)
        dt = dt->shared->parent; 
    if (H5T_INTEGER != dt->shared->type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "operation not defined for datatype class");

    
    dt->shared->u.atomic.u.i.sign = sign;

done:
    FUNC_LEAVE_API(ret_value)
}
