/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Pmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Ppkg.h"      
#include "H5VMprivate.h" 

#define H5P_ENCODE_VERS 0

typedef struct {
    bool    encode;       
    size_t *enc_size_ptr; 
    void  **pp;           
} H5P_enc_iter_ud_t;

herr_t
H5P__encode_size_t(const void *value, void **_pp, size_t *size)
{
    uint64_t  enc_value = (uint64_t) * (const size_t *)value; 
    uint8_t **pp        = (uint8_t **)_pp;
    unsigned  enc_size  = H5VM_limit_enc_size(enc_value); 

    FUNC_ENTER_PACKAGE_NOERR

    
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));
    assert(enc_size < 256);
    assert(size);

    if (NULL != *pp) {
        
        *(*pp)++ = (uint8_t)enc_size;

        
        UINT64ENCODE_VAR(*pp, enc_value, enc_size);
    } 

    
    *size += (1 + enc_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5P__encode_hsize_t(const void *value, void **_pp, size_t *size)
{
    uint64_t  enc_value = (uint64_t) * (const hsize_t *)value; 
    unsigned  enc_size  = H5VM_limit_enc_size(enc_value);      
    uint8_t **pp        = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    HDcompile_assert(sizeof(hsize_t) <= sizeof(uint64_t));
    assert(enc_size < 256);
    assert(size);

    if (NULL != *pp) {
        *(*pp)++ = (uint8_t)enc_size;

        
        UINT64ENCODE_VAR(*pp, enc_value, enc_size);
    } 

    
    *size += (1 + enc_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5P__encode_unsigned(const void *value, void **_pp, size_t *size)
{
    uint8_t **pp = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(value);
    assert(size);

    if (NULL != *pp) {
        
        *(*pp)++ = (uint8_t)sizeof(unsigned);

        
        H5_ENCODE_UNSIGNED(*pp, *(const unsigned *)value);
    } 

    
    *size += (1 + sizeof(unsigned));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5P__encode_uint8_t(const void *value, void **_pp, size_t *size)
{
    uint8_t **pp = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(value);
    assert(size);

    if (NULL != *pp) {
        
        *(*pp)++ = *(const uint8_t *)value;
    } 

    
    *size += 1;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5P__encode_bool(const void *value, void **_pp, size_t *size)
{
    uint8_t **pp = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(value);
    assert(size);

    if (NULL != *pp)
        
        *(*pp)++ = (uint8_t) * (const bool *)value;

    
    *size += 1;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5P__encode_double(const void *value, void **_pp, size_t *size)
{
    uint8_t **pp = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(value);
    assert(size);

    if (NULL != *pp) {
        
        *(*pp)++ = (uint8_t)sizeof(double);

        
        H5_ENCODE_DOUBLE(*pp, *(const double *)value);
    } 

    
    *size += (1 + sizeof(double));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5P__encode_uint64_t(const void *value, void **_pp, size_t *size)
{
    uint8_t **pp = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(value);
    assert(size);

    if (NULL != *pp) {
        
        *(*pp)++ = (uint8_t)sizeof(uint64_t);

        
        UINT64ENCODE(*pp, *(const unsigned *)value);
    } 

    
    *size += (1 + sizeof(uint64_t));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static int
H5P__encode_cb(H5P_genprop_t *prop, void *_udata)
{
    H5P_enc_iter_ud_t *udata     = (H5P_enc_iter_ud_t *)_udata; 
    int                ret_value = H5_ITER_CONT;                

    FUNC_ENTER_PACKAGE

    
    assert(prop);
    assert(udata);

    
    if (prop->encode) {
        size_t prop_name_len;  
        size_t prop_value_len; 

        
        prop_name_len = strlen(prop->name) + 1;
        if (udata->encode) {
            strcpy((char *)*(udata->pp), prop->name);
            *(uint8_t **)(udata->pp) += prop_name_len;
        } 
        *(udata->enc_size_ptr) += prop_name_len;

        
        H5_BEFORE_USER_CB(H5_ITER_ERROR)
            {
                
                prop_value_len = 0;
                ret_value      = (prop->encode)(prop->value, udata->pp, &prop_value_len);
            }
        H5_AFTER_USER_CB(H5_ITER_ERROR)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTENCODE, H5_ITER_ERROR, "property encoding routine failed");
        *(udata->enc_size_ptr) += prop_value_len;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__encode(const H5P_genplist_t *plist, bool enc_all_prop, void *buf, size_t *nalloc)
{
    H5P_enc_iter_ud_t udata;                 
    uint8_t          *p = (uint8_t *)buf;    
    int               idx;                   
    size_t            encode_size = 0;       
    bool              encode      = true;    
    herr_t            ret_value   = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (NULL == nalloc)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "bad allocation size pointer");

    
    if (NULL == p)
        encode = false;

    
    if (encode) {
        
        *p++ = (uint8_t)H5P_ENCODE_VERS;

        
        *p++ = (uint8_t)plist->pclass->type;
    } 
    encode_size += 2;

    
    udata.encode       = encode;
    udata.enc_size_ptr = &encode_size;
    udata.pp           = (void **)&p;

    
    idx = 0;
    if (H5P__iterate_plist(plist, enc_all_prop, &idx, H5P__encode_cb, &udata) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_BADITER, FAIL, "can't iterate over properties");

    
    if (encode)
        *p++ = 0;
    encode_size++;

    
    *nalloc = encode_size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__decode_size_t(const void **_pp, void *_value)
{
    size_t         *value = (size_t *)_value; 
    const uint8_t **pp    = (const uint8_t **)_pp;
    uint64_t        enc_value; 
    unsigned        enc_size;  

    FUNC_ENTER_PACKAGE_NOERR

    
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));
    assert(pp);
    assert(*pp);
    assert(value);

    
    enc_size = *(*pp)++;
    assert(enc_size < 256);

    
    UINT64DECODE_VAR(*pp, enc_value, enc_size);
    H5_CHECKED_ASSIGN(*value, size_t, enc_value, uint64_t);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5P__decode_hsize_t(const void **_pp, void *_value)
{
    hsize_t        *value = (hsize_t *)_value; 
    const uint8_t **pp    = (const uint8_t **)_pp;
    uint64_t        enc_value; 
    unsigned        enc_size;  

    FUNC_ENTER_PACKAGE_NOERR

    
    HDcompile_assert(sizeof(hsize_t) <= sizeof(uint64_t));
    assert(pp);
    assert(*pp);
    assert(value);

    
    enc_size = *(*pp)++;
    assert(enc_size < 256);

    
    UINT64DECODE_VAR(*pp, enc_value, enc_size);
    H5_CHECKED_ASSIGN(*value, hsize_t, enc_value, uint64_t);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5P__decode_unsigned(const void **_pp, void *_value)
{
    unsigned       *value = (unsigned *)_value; 
    const uint8_t **pp    = (const uint8_t **)_pp;
    unsigned        enc_size;            
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(pp);
    assert(*pp);
    assert(value);

    
    enc_size = *(*pp)++;
    if (enc_size != sizeof(unsigned))
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "unsigned value can't be decoded");

    H5_DECODE_UNSIGNED(*pp, *value);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__decode_uint8_t(const void **_pp, void *_value)
{
    uint8_t        *value     = (uint8_t *)_value; 
    const uint8_t **pp        = (const uint8_t **)_pp;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(pp);
    assert(*pp);
    assert(value);

    
    *value = *(*pp)++;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__decode_bool(const void **_pp, void *_value)
{
    bool           *value     = (bool *)_value; 
    const uint8_t **pp        = (const uint8_t **)_pp;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(pp);
    assert(*pp);
    assert(value);

    
    *value = (bool)*(*pp)++;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__decode_double(const void **_pp, void *_value)
{
    double         *value = (double *)_value; 
    const uint8_t **pp    = (const uint8_t **)_pp;
    unsigned        enc_size;            
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(pp);
    assert(*pp);
    assert(value);

    
    enc_size = *(*pp)++;
    if (enc_size != sizeof(double))
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "double value can't be decoded");

    H5_DECODE_DOUBLE(*pp, *value);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5P__decode_uint64_t(const void **_pp, void *_value)
{
    uint64_t       *value = (uint64_t *)_value; 
    const uint8_t **pp    = (const uint8_t **)_pp;
    unsigned        enc_size;            
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(pp);
    assert(*pp);
    assert(value);

    
    enc_size = *(*pp)++;
    if (enc_size != sizeof(uint64_t))
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "uint64_t value can't be decoded");

    UINT64DECODE(*pp, *value);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5P__decode(const void *buf)
{
    H5P_genplist_t  *plist;                            
    void            *value_buf = NULL;                 
    const uint8_t   *p         = (const uint8_t *)buf; 
    H5P_plist_type_t type;                             
    hid_t            plist_id       = -1;              
    size_t           value_buf_size = 0;               
    uint8_t          vers;                             
    hid_t            ret_value = H5I_INVALID_HID;      

    FUNC_ENTER_PACKAGE

    
    if (NULL == p)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "decode buffer is NULL");

    
    vers = (uint8_t)*p++;
    if ((uint8_t)H5P_ENCODE_VERS != vers)
        HGOTO_ERROR(H5E_PLIST, H5E_VERSION, FAIL, "bad version # of encoded information, expected %u, got %u",
                    (unsigned)H5P_ENCODE_VERS, (unsigned)vers);

    
    type = (H5P_plist_type_t)*p++;
    if (type <= H5P_TYPE_USER || type >= H5P_TYPE_MAX_TYPE)
        HGOTO_ERROR(H5E_PLIST, H5E_BADRANGE, FAIL, "bad type of encoded information: %u", (unsigned)type);

    
    if ((plist_id = H5P__new_plist_of_type(type)) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_VERSION, FAIL, "can't create property list of type: %u\n", (unsigned)type);

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(plist_id)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, FAIL, "not a property class");

    
    while (p) {
        H5P_genprop_t *prop; 
        const char    *name; 

        
        if (0 == *p)
            break;

        
        name = (const char *)p;
        p += strlen(name) + 1;

        
        if (NULL == (prop = H5P__find_prop_plist(plist, name)))
            HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "property doesn't exist: '%s'", name);

        
        if (prop->size > value_buf_size) {
            if (NULL == (value_buf = H5MM_realloc(value_buf, prop->size)))
                HGOTO_ERROR(H5E_PLIST, H5E_CANTALLOC, FAIL, "decoding buffer allocation failed");
            value_buf_size = prop->size;
        } 

        
        if (prop->decode) {
            
            H5_BEFORE_USER_CB(FAIL)
                {
                    ret_value = (prop->decode)((const void **)&p, value_buf);
                }
            H5_AFTER_USER_CB(FAIL)
            if (ret_value < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTDECODE, FAIL,
                            "property decoding routine failed, property: '%s'", name);
        } 
        else
            HGOTO_ERROR(H5E_PLIST, H5E_NOTFOUND, FAIL, "no decode callback for property: '%s'", name);

        
        if (H5P_poke(plist, name, value_buf) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value for property: '%s'", name);
    } 

    
    ret_value = plist_id;

done:
    
    if (value_buf)
        value_buf = H5MM_xfree(value_buf);

    
    if (ret_value < 0) {
        if (plist_id > 0 && H5I_dec_ref(plist_id) < 0)
            HDONE_ERROR(H5E_PLIST, H5E_CANTCLOSEOBJ, FAIL,
                        "unable to close partially initialized property list");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 
