% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/emmeans.R
\name{emmobj}
\alias{emmobj}
\title{Construct an \code{emmGrid} object from scratch}
\usage{
emmobj(bhat, V, levels, linfct = diag(length(bhat)), df = NA, dffun,
  dfargs = list(), post.beta = matrix(NA), nesting = NULL, se.bhat,
  se.diff, ...)
}
\arguments{
\item{bhat}{Numeric. Vector of regression coefficients}

\item{V}{Square matrix. Covariance matrix of \code{bhat}}

\item{levels}{Named list or vector. Levels of factor(s) that define the
estimates defined by \code{linfct}. If not a list, we assume one factor
named \code{"level"}}

\item{linfct}{Matrix. Linear functions of \code{bhat} for each combination 
of \code{levels}.}

\item{df}{Numeric value or function with arguments \code{(x, dfargs)}. If a
number, that is used for the degrees of freedom. If a function, it should
return the degrees of freedom for \code{sum(x*bhat)}, with any additional
parameters in \code{dfargs}.}

\item{dffun}{Overrides \code{df} if specified. This is a convenience
to match the slot names of the returned object.}

\item{dfargs}{List containing arguments for \code{df}.
This is ignored if df is numeric.}

\item{post.beta}{Matrix whose columns comprise a sample from the posterior
distribution of the regression coefficients (so that typically, the column
averages will be \code{bhat}). A 1 x 1 matrix of \code{NA} indicates that
such a sample is unavailable.}

\item{nesting}{Nesting specification as in \code{\link{ref_grid}}. This is
ignored if \code{model.info} is supplied.}

\item{se.bhat, se.diff}{Alternative way of specifying \code{V}. If \code{se.bhat}
is specified, \code{V} is constructed using \code{se.bhat}, the standard errors of \code{bhat},
and \code{se.diffs}, the standard errors of its pairwise differences. \code{se.diff}
should be a vector of length \code{k*(k-1)/2} where \code{k} is the length
of \code{se.bhat}, and its elements should be in the order \code{12,13,...,1k,23,...2k,...}.
If \code{se.diff} is missing, \code{V} is computed as if the \code{bhat} are independent.}

\item{...}{Arguments passed to \code{\link{update.emmGrid}}}
}
\value{
An \code{emmGrid} object
}
\description{
This allows the user to incorporate results obtained by some analysis
into an \code{emmGrid} object, enabling the use of \code{emmGrid} methods
to perform related follow-up analyses.
}
\details{
The arguments must be conformable. This includes that the length of
 \code{bhat}, the number of columns of \code{linfct}, and the number of
 columns of \code{post.beta} must all be equal. And that the product of
 lengths in \code{levels} must be equal to the number of rows of
 \code{linfct}. The \code{grid} slot of the returned object is generated 
 by \code{\link{expand.grid}} using \code{levels} as its arguments. So the
 rows of \code{linfct} should be in corresponding order.
 
The functions \code{qdrg} and \code{\link{emmobj}} are close cousins, in that
they both produce \code{emmGrid} objects. When starting with summary
statistics for an existing grid, \code{emmobj} is more useful, while
\code{qdrg} is more useful when starting from an unsupported fitted model.
}
\examples{
# Given summary statistics for 4 cells in a 2 x 2 layout, obtain 
# marginal means and comparisons thereof. Assume heteroscedasticity
# and use the Satterthwaite method
levels <- list(trt = c("A", "B"), dose = c("high", "low"))
ybar <- c(57.6, 43.2, 88.9, 69.8)
s <-    c(12.1, 19.5, 22.8, 43.2)
n <-    c(44,   11,   37,   24)
se2 = s^2 / n
Satt.df <- function(x, dfargs)
    sum(x * dfargs$v)^2 / sum((x * dfargs$v)^2 / (dfargs$n - 1))
    
expt.emm <- emmobj(bhat = ybar, V = diag(se2),
    levels = levels, linfct = diag(c(1, 1, 1, 1)),
    df = Satt.df, dfargs = list(v = se2, n = n), estName = "mean")

( trt.emm <- emmeans(expt.emm, "trt") )
( dose.emm <- emmeans(expt.emm, "dose") )

rbind(pairs(trt.emm), pairs(dose.emm), adjust = "mvt")

### Create an emmobj from means and SEs
### (This illustration reproduces the MOats example for Variety = "Victory")
means = c(71.50000,  89.66667, 110.83333, 118.50000)
semeans = c(5.540591, 6.602048, 8.695358, 7.303221)
sediffs = c(7.310571,  9.894724,  7.463615, 10.248306,  4.935698,  8.694507)
foo = emmobj(bhat = means, se.bhat = semeans, se.diff = sediffs, 
             levels = list(nitro = seq(0, .6, by = .2)), df = 10)
plot(foo, comparisons = TRUE)

}
\seealso{
\code{\link{qdrg}}, an alternative that is useful when starting 
  with a fitted model not supported in \pkg{emmeans}.
}
