% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dsm.var.prop.R
\name{dsm.var.prop}
\alias{dsm.var.prop}
\title{Prediction variance propagation for DSMs}
\usage{
dsm.var.prop(dsm.obj, pred.data, off.set, seglen.varname = "Effort",
  type.pred = "response")
}
\arguments{
\item{dsm.obj}{a model object returned from running \code{\link{dsm}}.}

\item{pred.data}{either: a single prediction grid or list of prediction grids. Each grid should be a \code{data.frame} with the same columns as the original data.}

\item{off.set}{a a vector or list of vectors with as many elements as there are in \code{pred.data}. Each vector is as long as the number of rows in the corresponding element of \code{pred.data}. These give the area associated with each prediction cell. If a single number is supplied it will be replicated for the length of \code{pred.data}.}

\item{seglen.varname}{name for the column which holds the segment length (default value \code{"Effort"}).}

\item{type.pred}{should the predictions be on the "response" or "link" scale? (default \code{"response"}).}
}
\value{
a list with elements
        \tabular{ll}{\code{model} \tab the fitted model object\cr
                     \code{pred.var} \tab variance of each region given
                     in \code{pred.data}.\cr
                     \code{bootstrap} \tab logical, always \code{FALSE}\cr
                     \code{pred.data} \tab as above\cr
                     \code{off.set} \tab as above\cr
                     \code{model}\tab the fitted model with the extra term\cr
                     \code{dsm.object} \tab the original model, as above\cr
                     \code{model.check} \tab simple check of subtracting the coefficients of the two models to see if there is a large difference\cr
                     \code{deriv} \tab numerically calculated Hessian of the offset\cr
                     }
}
\description{
To ensure that uncertainty from the detection function is correctly propagated to the final variance estimate of abundance, this function uses a method first detailed in Williams et al (2011).
}
\details{
The idea is to refit the spatial model but including an extra random effect. This random effect has zero mean and hence to effect on point estimates. Its variance is the Hessian of the detection function. Variance estimates then incorporate detection function uncertainty. Further mathematical details are given in the paper in the references below.

Many prediction grids can be supplied by supplying a list of \code{data.frame}s to the function.

Note that this routine simply calls \code{\link{dsm_varprop}}. If you don't require multiple prediction grids, the other routine will probably be faster.
}
\section{Diagnostics}{

The summary output from the function includes a simply diagnostic that shows the average probability of detection from the "original" fitted model (the model supplied to this function; column \code{Fitted.model}) and the probability of detection from the refitted model (used for variance propagation; column \code{Refitted.model}) along with the standard error of the probability of detection from the fitted model (\code{Fitted.model.se}), at the unique values of any covariates used in the detection function. If there are large differences between the probabilities of detection then there are potentially problems with the fitted model, the variance propagation or both. This can be because the fitted model does not account for enough of the variability in the data and in refitting the variance model accounts for this in the random effect.
}

\section{Limitations}{

Note that this routine is only useful if a detection function has been used in the DSM. It cannot be used when the \code{Nhat}, \code{abundance.est} responses are used. Importantly this requires that if the detection function has covariates, then these do not vary within a segment (so, for example covariates like sex cannot be used).

Negative binomial models fitted using the \code{\link{nb}} family will give strange results (overly big variance estimates due to scale parameter issues) so \code{nb} models are automatically refitted with \code{\link{negbin}} (with a warning). It is probably worth refitting these models with \code{negbin} manually (perhaps giving a smallish range of possible values for the negative binomial parameter) to check that convergence was reached.
}

\examples{
\dontrun{
 library(Distance)
 library(dsm)

 # load the Gulf of Mexico dolphin data (see ?mexdolphins)
 data(mexdolphins)

 # fit a detection function
 df <- ds(distdata, max(distdata$distance),
          key = "hn", adjustment = NULL)

 # fit a simple smooth of x and y
 mod1 <- dsm(count~s(x, y), df, segdata, obsdata, family=tw())

 # Calculate the variance
 # this will give a summary over the whole area in mexdolphins$preddata
 mod1.var <- dsm.var.prop(mod1, preddata, off.set=preddata$area)
 summary(mod1.var)
}
}
\references{
Williams, R., Hedley, S.L., Branch, T.A., Bravington, M.V., Zerbini, A.N. and Findlay, K.P. (2011). Chilean Blue Whales as a Case Study to Illustrate Methods to Estimate Abundance and Evaluate Conservation Status of Rare Species. Conservation Biology 25(3), 526-535.
}
\author{
Mark V. Bravington, Sharon L. Hedley. Bugs added by David L. Miller.
}
