#' @importFrom stats aggregate
make.data <- function(response, ddfobject, segdata, obsdata, group,
                      convert.units, availability, strip.width, segment.area,
                      family, transect){

  # probably want to do something smart here...
  seglength.name<-'Effort'
  segnum.name<-'Sample.Label'
  distance.name<-'distance'
  cluster.name<-'size'

  # Check that observations are between left and right truncation
  # warning only -- observations are excluded below
  # No truncation check for strip transects
  if(!is.null(ddfobject) & !is.null(obsdata[[distance.name]])){
    if(any(obsdata[,distance.name]>ddfobject$meta.data$width)){
      warning("Some observations are outside of detection function truncation!")
    }
  }

  # Estimating group abundance/density
  if(group){
    obsdata[,cluster.name][obsdata[,cluster.name]>0] <- 1
  }

  # if we fitted a detection function
  if(!is.null(ddfobject)){
    # grab the probabilities of detection
    fitted.p <- fitted(ddfobject)

    # remove observations which were not in the detection function
    obsdata <- obsdata[obsdata$object %in% names(fitted.p),]
    # what if there are no matches? Perhaps this is due to the object
    # numbers being wrong? (HINT: yes.)
    if(nrow(obsdata) == 0){
      stop("No observations in detection function matched those in observation table. Check the \"object\" column.")
    }
    # reorder the fitted ps, making sure that
    # they match the ordering in obsdata
    fitted.p <- fitted.p[match(obsdata$object, names(fitted.p))]
  }else{
    # strip transects or presence/absence data
    fitted.p <- 1
    if(is.null(strip.width) & (response != "presence")){
      stop("You must specify strip.width for strip transects!")
    }
  }


  ## Aggregate response values of the sightings over segments
  if(response %in% c("D","density","Dhat","density.est")){
    responsedata <- aggregate(obsdata[,cluster.name]/(fitted.p*availability),
                              list(obsdata[,segnum.name]), sum)
    off.set <- "none"
  }else if(response %in% c("N","count","n")){
    responsedata <- aggregate(obsdata[,cluster.name]/availability,
                              list(obsdata[,segnum.name]), sum)
    off.set <- "eff.area"
  }else if(response %in% c("Nhat","abundance.est")){
    responsedata <- aggregate(obsdata[,cluster.name]/(fitted.p*availability),
                              list(obsdata[,segnum.name]), sum)
    off.set<-"area"
  }else if(response == "presence"){
    responsedata <- aggregate(obsdata[,cluster.name],
                                list(obsdata[,segnum.name]), sum)
    responsedata$x[responsedata$x>0] <- 1
    responsedata$x[responsedata$x<1] <- 0
    off.set <- "none"
  }

  ## warn if any observations were not allocated
  responsecheck <- aggregate(obsdata[,cluster.name],
                             list(obsdata[,segnum.name]), sum)
  if(sum(obsdata[,cluster.name]) != sum(responsecheck[,2])){
    message(paste0("Some observations were not allocated to segments!\n",
                   "Check that Sample.Labels match"))
  }

  # name the response data columns
  names(responsedata) <- c(segnum.name, response)

  # Next merge the response variable with the segment records and any
  # response variable that is NA should be assigned 0 because these
  # occur due to 0 sightings
  dat <- merge(segdata, responsedata, by=segnum.name, all.x=TRUE)
  dat[,response][is.na(dat[, response])] <- 0

  # for the offsets with effective area, need to make sure that
  # the ps match the segments
  # fitted.p is already in the same order as responsedata/obsdata
  if(off.set == "eff.area"){
    # if there are no covariates, and all the fitted ps are the same
    # then just duplicate that value enough times for the segments
    if(length(unique(fitted.p)) == 1){
      fitted.p <- rep(unique(fitted.p), nrow(dat))
    }else{
#      stop("Covariate detection functions are not currently supported with effective area as the offset")
message("Count model with detection function covariates at the segment level: this is EXPERIMENTAL!")

      if(ddfobject$method != "ds"){
        stop("Only dsmodels are supported!")
      }

      # extract formula
      df_formula <- as.formula(ddfobject$ds$aux$ddfobj$scale$formula)
      if(!is.null(ddfobject$ds$aux$ddfobj$shape$formula) &&
         ddfobject$ds$aux$ddfobj$shape$formula != "~1"){
        stop("Shape parameter formulae are not supported!")
      }

      # extract detection function variables
      df_vars <- all.vars(df_formula)

      # check these vars are in the segment table
      if(!all(df_vars %in% colnames(dat))){
        stop(paste0("Detection function covariates are not in the segment data",
                    "\n  Missing: ", df_vars[!(df_vars %in% colnames(dat))]))
      }

      # make a data.frame to predict for
      nd <- dat[,df_vars, drop=FALSE]
      nd$distance <- 0

      fitted.p <- predict(ddfobject, newdata=nd)$fitted

    }
  }


  if(!is.null(segment.area)){

    # pull the column if segment.area is character
    if(is.character(segment.area)){
      segment.area <- dat[, segment.area]
    }

    dat$off.set <- switch(off.set,
                          eff.area=segment.area*fitted.p,
                          area=segment.area,
                          none=1)

    # if we have density then use that as the response
    if(response %in% c("D","density","Dhat","density.est")){
      dat[,response] <- dat[,response]/(segment.area*convert.units)
    }

    # set the segment area in the data
    dat$segment.area <- segment.area*convert.units

  }else{
    # pull this from the detection function
    if (!is.null(ddfobject)){
      # calculate the "width" of the transect first, make sure we get it right
      # if we are doing left truncation
      width <- ddfobject$meta.data$width
      if(!is.null(ddfobject$meta.data$left)){
        width <- width - ddfobject$meta.data$left
      }
    }else{
    # or use strip.width if we have strip transects
      width <- strip.width
      # note we have to reset off.set
      if(response == "presence"){
        off.set <- "none"
      }else{
        off.set <- "area"
      }
    }

    # check that none of the Effort values are zero
    if(any(dat[,seglength.name]==0)){
      stop(paste0("Effort values for segments: ",
                  paste(which(dat[,seglength.name]==0), collapse=", "),
                  " are 0."))
    }

    # calculate the offset
    #   area we just calculate the area
    #   effective area multiply by p
    #   when density is response, offset should be 1 (and is ignored anyway)

    # calculate the area
    # if we have point transects
    if(transect=="point"){
      # here "Effort" is number of visits
      dat$segment.area <- pi*width^2*dat[,seglength.name]
    }else{
    # line transects
      dat$segment.area <- 2*dat[,seglength.name]*width
    }

    dat$off.set <- switch(off.set,
                          eff.area=dat$segment.area*fitted.p,
                          area=dat$segment.area,
                          none=1)

    # calculate the density (count/area)
    if(response %in% c("D","density","Dhat","density.est")){
      dat[,response] <- dat[,response]/(dat$segment.area*convert.units)
    }

    # correct segment area units
    dat$segment.area <- dat$segment.area*convert.units
  }

  # multiply up by conversion factor
  dat$off.set <- dat$off.set*convert.units

  # Set offset as log (or whatever link is) of area or effective area
  dat$off.set <- family$linkfun(dat$off.set)

  return(dat)
}
