% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rocperf.R
\name{rocperf}
\alias{rocperf}
\title{ROC curve performances}
\usage{
rocperf(
  x,
  y,
  sensitivities = NULL,
  specificities = NULL,
  conf.level = 0.95,
  fun = NULL,
  seed = 1,
  boot.n = 2000,
  median = FALSE,
  attr = FALSE,
  parallel = FALSE,
  simplify = TRUE,
  ...
)
}
\arguments{
\item{x}{Numeric vector containing the predicted value for each observation.}

\item{y}{Factor, numeric, logical or character vector encoding the response.}

\item{sensitivities}{Vector of sensitivity thresholds. Default NULL.}

\item{specificities}{Vector of specificity thresholds. Default NULL.}

\item{conf.level}{Width of the confidence interval. Default: 0.95 (i.e., 95\% CI).}

\item{fun}{Function to compute additional statistics. Default NULL.}

\item{seed}{Random seed for bootstrapping. Default 1.}

\item{boot.n}{Number of bootstrap samples. Default 2e3.}

\item{median}{If TRUE, return median bootstrap sensitivities and specificities, otherwise return observed values, otherwise the observe value is provided. Default FALSE.}

\item{attr}{Return bootstrap results and ROC curve as attributes. Default FALSE.}

\item{parallel}{Parallelise bootstrap. Default FALSE.}

\item{simplify}{If TRUE, return only median for results of the function \code{fun} having one value across bootstraps. Default TRUE.}

\item{...}{Additional arguments passed to \code{fun} if not NULL.}
}
\value{
A data frame with the following columns:
\itemize{
\item{\code{n.control}: Number of control patients}
\item{\code{n.case}: Number of case patients}
\item{\code{MannWhitney.pvalue}: Mann Whitney U test p-value}
\item{\code{AUC.pvalue}: p-value for the null hypothesis that AUC=0.5}
\item{\code{AUC}: Area under the ROC curve (point estimate)}
\item{\code{AUC.lCI}: Lower limit of 95\% confidence interval for AUC}
\item{\code{AUC.uCI}: Upper limit of 95\% confidence interval for AUC}
\item{\code{AUC.lQuart}: Lower limit of 50\% confidence interval for AUC}
\item{\code{AUC.uQuart}: Upper limit of 50\% confidence interval for AUC}
\item{\code{Se@SpX}: Sensitivity at X\% specificity}
\item{\code{Se@SpX.lCI}: Lower limit of 95\% confidence interval for sensitivity at X\% specificity}
\item{\code{Se@SpX.uCI}: Upper limit of 95\% confidence interval for sensitivity at X\% specificity}
\item{\code{Se@SpX.lQuart}: Lower limit of 95\% confidence interval for sensitivity at X\% specificity}
\item{\code{Se@SpX.uQuart}: Upper limit of 95\% confidence interval for sensitivity at X\% specificity}
\item{\code{Sp@SeX}: Specificity at X\% sensitivity}
\item{\code{Sp@SeX.lCI}: Lower limit of 95\% confidence interval for specificity at X\% sensitivity}
\item{\code{Sp@SeX.uCI}: Upper limit of 95\% confidence interval for specificity at X\% sensitivity}
\item{\code{Sp@SeX.lQuart}: Lower limit of 50\% confidence interval for specificity at X\% sensitivity}
\item{\code{Sp@SeX.uQuart}: Upper limit of 50\% confidence interval for specificity at X\% sensitivity}
\item{Additional columns for statistics computed by the function \code{fun} if provided}
}

\code{data.frame} with one row with computed statistics in columns.
}
\description{
Range of statistics associated with a ROC curve with confidence interval 
where applicable. This function is faster than the alternatives provided by
the package pROC.
}
\details{
This function computes the area under the ROC curve (AUC) and its confidence
interval, the Mann-Whitney U test p-value, and the p-value for the null
hypothesis that the AUC is equal to 0.5 (DeLong et al. 1988).

The function also computes the sensitivity at specified specificities and
the specificity at specified sensitivities, with confidence intervals and
interquartile ranges if bootstrapping is performed.

The function uses the \code{pROC} package to compute the ROC curve and
confidence intervals, and it can handle parallel processing for bootstrapping.

The function returns a data frame with the computed statistics, including: 
\itemize{
\item Number of control and case patients 
\item Mann Whitney U test p-value 
\item AUC and its confidence intervals 
\item Sensitivity at specified specificities and their confidence intervals 
\item Specificity at specified sensitivities and their confidence intervals 
}

The function \code{fun} must take the following arguments:
\itemize{
\item \code{controls}: vector of control values
\item \code{cases}: vector of case values
\item \code{thresholds}: vector of thresholds used for the ROC curve
\item \code{sensitivities}: vector of sensitivities
\item \code{specificities}: vector of specificities
\item \code{...}: additional arguments
}
and return a named vector of values.
}
\examples{
\dontshow{if (interactive() ) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
fu <- function(controls,cases,thr,se,sp,...) {
  r <- pROC::roc(c(rep(0,length(controls)),
                   rep(1,length(cases))),
                 c(controls,cases),
                 quiet=TRUE)
  c(AUC_fun=r$auc) 
}
set.seed(1)
n <- 123
y <- runif(n)<.5
x <- rnorm(n)+y*1
ans <- rocperf(x,y,fun=fu)
ans <- rocperf(x,y,fun=fu,
               senitivities=c(.5,.75,.9),
               specificities=c(.5,.75,.9))
               
\dontshow{\}) # examplesIf}
}
