#' Extract multiple imputed datasets from an object of class JointAI
#'
#' This function returns a dataset containing multiple imputed datasets stacked
#' onto each other (i.e., long format; optionally including the original, incomplete data).\cr
#' These data can be automatically exported to SPSS (i.e., a .txt file containing the data and a
#' .sps file containing syntax to generate a .sav file). For the export function
#' the \href{https://CRAN.R-project.org/package=foreign}{\strong{foreign}} package needs to be installed.
#' @inheritParams sharedParams
#' @param m number of imputed datasets
#' @param include should the original, incomplete data be included? Default is \code{TRUE}.
#' @param minspace minimum number of iterations between iterations chosen as imputed values.
#' @param seed optional seed
#' @param export_to_SPSS logical; should the completed data be exported to SPSS?
#' @param resdir optional directory for results (if unspecified and
#'               \code{export_to_SPSS = TRUE} the current working directory is used)
#' @param filename optional file name (without ending; if unspecified and
#'                 \code{export_to_SPSS = TRUE} a name is generated automatically)
#'
#' @return A dataframe in which the original data (if \code{include = TRUE}) and
#'         the imputed datasets are stacked onto each other.\cr
#'        The variable \code{Imputation_} indexes the imputation, while
#'        \code{.rownr} links the rows to the rows of the original data.
#'        In cross-sectional datasets the
#'        variable \code{.id} is added as subject identifier.
#'
#' @section Note:
#' In order to be able to extract (multiple) imputed datasets the imputed values
#' must have been monitored, i.e., \code{imps = TRUE} had to be specified in the
#' argument \code{monitor_params} in \code{\link[JointAI:model_imp]{*_imp}}.
#'
#' @seealso \code{\link{plot_imp_distr}}
#'
#' @examples
#' # fit a model and monitor the imputed values with monitor_params = c(imps = TRUE)
#' mod <- lm_imp(y ~ C1 + C2 + M2, data = wideDF, monitor_params = c(imps = TRUE), n.iter = 100)
#'
#' # Example 1: without export to SPSS
#' MIs <- get_MIdat(mod, m = 3, seed = 123)
#'
#'
#' \dontrun{
#' # Example 2: with export for SPSS (here: to the temporary directory "temp_dir")
#' temp_dir <- tempdir()
#' MIs <- get_MIdat(mod, m = 3, seed = 123, resdir = temp_dir,
#'                  filename = "example_imputation",
#'                  export_to_SPSS = TRUE)
#'
#' }
#'
#' @export
#'
get_MIdat <- function(object, m = 10, include = TRUE,
                      start = NULL, minspace = 50, seed = NULL,
                      export_to_SPSS = FALSE,
                      resdir = NULL, filename = NULL){

  if (is.null(object$models) | sum(is.na(object$data[, names(object$models)])) == 0)
    stop("This JointAI object did not impute any values.")

  if (!"foreign" %in% rownames(installed.packages()))
    stop("This function requires the 'foreign' package to be installed.")

  if (!is.null(seed))
    set.seed(seed)

  DF <- object$data
  DF$.rownr <- 1:nrow(DF)

  if (!object$analysis_type %in% c("lme", "glme", 'clmm')) {
    DF$.id <- 1:nrow(DF)
  }

  groups <- match(object$Mlist$groups, unique(object$Mlist$groups))

  meth <- object$models[colSums(is.na(DF[, names(object$models), drop = FALSE])) > 0]

  if (is.null(start)) {
    start <- start(object$MCMC)
  } else {
    start <- max(start, start(object$MCMC))
  }

  MCMC <- do.call(rbind, window(object$MCMC, start = start))
  if (nrow(MCMC) < m)
    stop("The number of imputations must be chosen to be less than or",
         gettextf("equal to the number of MCMC samples (= %s).",
                  nrow(MCMC)))


  # randomly draw which iterations should be used as imputation
  if (nrow(MCMC)/minspace < m)
    stop(gettextf('The total number of iterations (%s) is too small to select %s iterations with spacing of >= %s.',
                  nrow(MCMC), m, minspace))

  cand_iters <- seq(from = sample.int(minspace, size = 1), to = nrow(MCMC), by = minspace)
  imp_iters <- sort(sample(cand_iters, size = m))


  # reduce MCMC to relevant rows
  MCMC <- MCMC[imp_iters, , drop = FALSE]

  DF_list <- list()
  for (i in 1:(m + 1)) {
    DF_list[[i]] <- cbind("Imputation_" = i - 1, DF)
  }

  for (i in seq_along(meth)) {
    impval <- NULL
    # imputation by linear regression --------------------------------------------------------
    if (meth[i] %in% c("norm", "lognorm", "gamma", "beta")) {
      if (names(meth[i]) %in% colnames(object$data_list$Xtrafo)) {
        pat <- paste0("Xtrafo\\[[[:digit:]]*,",
                      match(names(meth)[i], colnames(object$data_list$Xtrafo)),
                      "\\]")
      } else {
        pat <- paste0("Xc\\[[[:digit:]]*,",
                      match(names(meth)[i], colnames(object$data_list$Xc)),
                      "\\]")
      }

      impval <- MCMC[, grep(pat, colnames(MCMC), value = TRUE), drop = FALSE]
      if (!is.null(object$scale_pars)) {
        impval <- impval * object$scale_pars["scale", names(meth)[i]]  +
          object$scale_pars["center", names(meth)[i]]
      }

      if (length(impval) > 0) {
        rownrs <- gsub(",[[:digit:]]*\\]", "",
                       gsub("[[:alpha:]]*\\[", "", colnames(impval)))

        for (j in (1:m) + 1) {
          DF_list[[j]][is.na(DF_list[[j]][, names(meth)[i]]), names(meth)[i]] <-
            impval[j - 1, na.omit(match(groups, as.numeric(rownrs)))]
        }
      }
    }
    # imputation by binary regression --------------------------------------------------------
    if (meth[i] %in% c("logit")) {
      pat <- paste0("Xc\\[[[:digit:]]*,",
                    match(attr(object$Mlist$refs[[names(meth)[i]]], "dummies"),
                          colnames(object$data_list$Xc)),
                    "\\]")
      impval <- MCMC[, grep(pat, colnames(MCMC), value = TRUE), drop = FALSE]

      if (length(impval) > 0) {
        rownrs <- gsub(",[[:digit:]]*\\]", "",
                       gsub("[[:alpha:]]*\\[", "", colnames(impval)))

        for (j in (1:m) + 1) {
          vec <- as.numeric(DF_list[[j]][, names(meth)[i]]) - 1
          vec[is.na(vec)] <- impval[j - 1, na.omit(match(groups, as.numeric(rownrs)))]
          vec <- as.factor(vec)
          levels(vec) <- levels(DF_list[[j]][, names(meth)[i]])
          DF_list[[j]][, names(meth)[i]] <- vec
        }
      }
    }

    # imputation of categorical variables ----------------------------------------------------
    if (meth[i] %in% c("cumlogit", "multilogit")) {
      pat <- paste0("Xcat\\[[[:digit:]]*,",
                    match(names(meth)[i], colnames(object$data_list$Xcat)),
                    "\\]")
      impval <- MCMC[, grep(pat, colnames(MCMC), value = TRUE), drop = FALSE]

      if (length(impval) > 0) {
        rownrs <- gsub(",[[:digit:]]*\\]", "",
                       gsub("[[:alpha:]]*\\[", "", colnames(impval)))

        for (j in (1:m) + 1) {
          vec <- as.numeric(DF_list[[j]][, names(meth)[i]])
          vec[is.na(vec)] <- impval[j - 1, na.omit(match(groups, as.numeric(rownrs)))]
          if (meth[i] == "cumlogit") {
            vec <- as.ordered(vec)
          }else{
            vec <- as.factor(vec)
          }
          levels(vec) <- levels(DF_list[[j]][, names(meth)[i]])
          DF_list[[j]][, names(meth)[i]] <- vec
        }
      }
    }


    # imputation with lmm ------------------------------------------------------
    if (meth[i] %in% c('lmm', 'glmm_lognorm', 'glmm_gamma', 'glmm_poisson')) {
      if (names(meth[i]) %in% colnames(object$data_list$Xl)) {
        pat <- paste0("Xl\\[[[:digit:]]*,",
                      match(names(meth)[i], colnames(object$data_list$Xl)),
                      "\\]")
      } else if (names(meth[i]) %in% colnames(object$data_list$Z)) {
        pat <- paste0("Z\\[[[:digit:]]*,",
                      match(names(meth)[i], colnames(object$data_list$Z)),
                      "\\]")
      } else if (names(meth[i]) %in% colnames(object$data_list$Xltrafo)) {
        pat <- paste0("Xltrafo\\[[[:digit:]]*,",
                      match(names(meth)[i], colnames(object$data_list$Xltrafo)),
                      "\\]")
      }

      impval <- MCMC[, grep(pat, colnames(MCMC), value = TRUE), drop = FALSE]
      if (!is.null(object$scale_pars)) {
        impval <- impval * object$scale_pars["scale", names(meth)[i]]  +
          object$scale_pars["center", names(meth)[i]]
      }

      if (length(impval) > 0) {
        rownrs <- gsub(",[[:digit:]]*\\]", "",
                       gsub("[[:alpha:]]*\\[", "", colnames(impval)))

        for (j in (1:m) + 1) {
          DF_list[[j]][is.na(DF_list[[j]][, names(meth)[i]]), names(meth)[i]] <-
            impval[j - 1, order(as.numeric(rownrs))]
        }
      }
    }

    # imputation with glmm_logit -----------------------------------------------
    if (meth[i] %in% c('glmm_logit')) {
      if (attr(object$Mlist$refs[[names(meth)[i]]], "dummies") %in% colnames(object$data_list$Xl)) {
        pat <- paste0("Xl\\[[[:digit:]]*,",
                      match(attr(object$Mlist$refs[[names(meth)[i]]], "dummies"),
                            colnames(object$data_list$Xl)),
                      "\\]")
      } else if (attr(object$Mlist$refs[[names(meth)[i]]], "dummies") %in% colnames(object$data_list$Z)) {
        pat <- paste0("Z\\[[[:digit:]]*,",
                      match(attr(object$Mlist$refs[[names(meth)[i]]], "dummies"),
                            colnames(object$data_list$Z)),
                      "\\]")
      }

      impval <- MCMC[, grep(pat, colnames(MCMC), value = TRUE), drop = FALSE]

      if (length(impval) > 0) {
        rownrs <- gsub(",[[:digit:]]*\\]", "",
                       gsub("[[:alpha:]]*\\[", "", colnames(impval)))

        for (j in (1:m) + 1) {
          DF_list[[j]][is.na(DF_list[[j]][, names(meth)[i]]), names(meth)[i]] <-
            impval[j - 1, order(as.numeric(rownrs))]
        }
      }
    }

    # imputation with clmm ------------------------------------------------------
    if (meth[i] %in% 'clmm') {
      if (names(meth[i]) %in% colnames(object$data_list$Xlcat)) {
        pat <- paste0("Xlcat\\[[[:digit:]]*,",
                      match(names(meth)[i], colnames(object$data_list$Xlcat)),
                      "\\]")
      } else if (names(meth[i]) %in% colnames(object$data_list$Z)) {
        pat <- paste0("Z\\[[[:digit:]]*,",
                      match(names(meth)[i], colnames(object$data_list$Z)),
                      "\\]")
      }
      impval <- MCMC[, grep(pat, colnames(MCMC), value = TRUE), drop = FALSE]

      if (length(impval) > 0) {
        rownrs <- gsub(",[[:digit:]]*\\]", "",
                       gsub("[[:alpha:]]*\\[", "", colnames(impval)))

        for (j in (1:m) + 1) {
          DF_list[[j]][is.na(DF_list[[j]][, names(meth)[i]]), names(meth)[i]] <-
            impval[j - 1, order(as.numeric(rownrs))]
        }
      }
    }
  }

  if (!include)
    DF_list <- DF_list[-1]

# build dataset --------------------------------------------------------------------------
  impDF <- do.call(rbind, DF_list)

  if (is.null(resdir))
    resdir <- getwd()

  if (is.null(filename))
    filename <- paste0("JointAI-imputation_", Sys.Date())

  if (export_to_SPSS == TRUE) {
    foreign::write.foreign(impDF,
               file.path(resdir, paste0(filename, ".txt")),
               file.path(resdir, paste0(filename, ".sps")),
               package = 'SPSS'
    )
  }

  return(impDF)
}
