



use core::fmt;
use core::mem::MaybeUninit;
pub type Size = usize;
pub type Filesize = u64;
pub type Timestamp = u64;
#[repr(transparent)]
#[derive(Copy, Clone, Hash, Eq, PartialEq, Ord, PartialOrd)]
pub struct Clockid(u32);


pub const CLOCKID_REALTIME: Clockid = Clockid(0);




pub const CLOCKID_MONOTONIC: Clockid = Clockid(1);

pub const CLOCKID_PROCESS_CPUTIME_ID: Clockid = Clockid(2);

pub const CLOCKID_THREAD_CPUTIME_ID: Clockid = Clockid(3);
impl Clockid {
    pub const fn raw(&self) -> u32 {
        self.0
    }

    pub fn name(&self) -> &'static str {
        match self.0 {
            0 => "REALTIME",
            1 => "MONOTONIC",
            2 => "PROCESS_CPUTIME_ID",
            3 => "THREAD_CPUTIME_ID",
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
    pub fn message(&self) -> &'static str {
        match self.0 {
            0 => {
                "The clock measuring real time. Time value zero corresponds with
1970-01-01T00:00:00Z."
            }
            1 => {
                "The store-wide monotonic clock, which is defined as a clock measuring
real time, whose value cannot be adjusted and which cannot have negative
clock jumps. The epoch of this clock is undefined. The absolute time
value of this clock therefore has no meaning."
            }
            2 => "The CPU-time clock associated with the current process.",
            3 => "The CPU-time clock associated with the current thread.",
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
}
impl fmt::Debug for Clockid {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.debug_struct("Clockid")
            .field("code", &self.0)
            .field("name", &self.name())
            .field("message", &self.message())
            .finish()
    }
}

#[repr(transparent)]
#[derive(Copy, Clone, Hash, Eq, PartialEq, Ord, PartialOrd)]
pub struct Errno(u16);

pub const ERRNO_SUCCESS: Errno = Errno(0);

pub const ERRNO_2BIG: Errno = Errno(1);

pub const ERRNO_ACCES: Errno = Errno(2);

pub const ERRNO_ADDRINUSE: Errno = Errno(3);

pub const ERRNO_ADDRNOTAVAIL: Errno = Errno(4);

pub const ERRNO_AFNOSUPPORT: Errno = Errno(5);

pub const ERRNO_AGAIN: Errno = Errno(6);

pub const ERRNO_ALREADY: Errno = Errno(7);

pub const ERRNO_BADF: Errno = Errno(8);

pub const ERRNO_BADMSG: Errno = Errno(9);

pub const ERRNO_BUSY: Errno = Errno(10);

pub const ERRNO_CANCELED: Errno = Errno(11);

pub const ERRNO_CHILD: Errno = Errno(12);

pub const ERRNO_CONNABORTED: Errno = Errno(13);

pub const ERRNO_CONNREFUSED: Errno = Errno(14);

pub const ERRNO_CONNRESET: Errno = Errno(15);

pub const ERRNO_DEADLK: Errno = Errno(16);

pub const ERRNO_DESTADDRREQ: Errno = Errno(17);

pub const ERRNO_DOM: Errno = Errno(18);

pub const ERRNO_DQUOT: Errno = Errno(19);

pub const ERRNO_EXIST: Errno = Errno(20);

pub const ERRNO_FAULT: Errno = Errno(21);

pub const ERRNO_FBIG: Errno = Errno(22);

pub const ERRNO_HOSTUNREACH: Errno = Errno(23);

pub const ERRNO_IDRM: Errno = Errno(24);

pub const ERRNO_ILSEQ: Errno = Errno(25);

pub const ERRNO_INPROGRESS: Errno = Errno(26);

pub const ERRNO_INTR: Errno = Errno(27);

pub const ERRNO_INVAL: Errno = Errno(28);

pub const ERRNO_IO: Errno = Errno(29);

pub const ERRNO_ISCONN: Errno = Errno(30);

pub const ERRNO_ISDIR: Errno = Errno(31);

pub const ERRNO_LOOP: Errno = Errno(32);

pub const ERRNO_MFILE: Errno = Errno(33);

pub const ERRNO_MLINK: Errno = Errno(34);

pub const ERRNO_MSGSIZE: Errno = Errno(35);

pub const ERRNO_MULTIHOP: Errno = Errno(36);

pub const ERRNO_NAMETOOLONG: Errno = Errno(37);

pub const ERRNO_NETDOWN: Errno = Errno(38);

pub const ERRNO_NETRESET: Errno = Errno(39);

pub const ERRNO_NETUNREACH: Errno = Errno(40);

pub const ERRNO_NFILE: Errno = Errno(41);

pub const ERRNO_NOBUFS: Errno = Errno(42);

pub const ERRNO_NODEV: Errno = Errno(43);

pub const ERRNO_NOENT: Errno = Errno(44);

pub const ERRNO_NOEXEC: Errno = Errno(45);

pub const ERRNO_NOLCK: Errno = Errno(46);

pub const ERRNO_NOLINK: Errno = Errno(47);

pub const ERRNO_NOMEM: Errno = Errno(48);

pub const ERRNO_NOMSG: Errno = Errno(49);

pub const ERRNO_NOPROTOOPT: Errno = Errno(50);

pub const ERRNO_NOSPC: Errno = Errno(51);

pub const ERRNO_NOSYS: Errno = Errno(52);

pub const ERRNO_NOTCONN: Errno = Errno(53);

pub const ERRNO_NOTDIR: Errno = Errno(54);

pub const ERRNO_NOTEMPTY: Errno = Errno(55);

pub const ERRNO_NOTRECOVERABLE: Errno = Errno(56);

pub const ERRNO_NOTSOCK: Errno = Errno(57);

pub const ERRNO_NOTSUP: Errno = Errno(58);

pub const ERRNO_NOTTY: Errno = Errno(59);

pub const ERRNO_NXIO: Errno = Errno(60);

pub const ERRNO_OVERFLOW: Errno = Errno(61);

pub const ERRNO_OWNERDEAD: Errno = Errno(62);

pub const ERRNO_PERM: Errno = Errno(63);

pub const ERRNO_PIPE: Errno = Errno(64);

pub const ERRNO_PROTO: Errno = Errno(65);

pub const ERRNO_PROTONOSUPPORT: Errno = Errno(66);

pub const ERRNO_PROTOTYPE: Errno = Errno(67);

pub const ERRNO_RANGE: Errno = Errno(68);

pub const ERRNO_ROFS: Errno = Errno(69);

pub const ERRNO_SPIPE: Errno = Errno(70);

pub const ERRNO_SRCH: Errno = Errno(71);

pub const ERRNO_STALE: Errno = Errno(72);

pub const ERRNO_TIMEDOUT: Errno = Errno(73);

pub const ERRNO_TXTBSY: Errno = Errno(74);

pub const ERRNO_XDEV: Errno = Errno(75);

pub const ERRNO_NOTCAPABLE: Errno = Errno(76);
impl Errno {
    pub const fn raw(&self) -> u16 {
        self.0
    }

    pub fn name(&self) -> &'static str {
        match self.0 {
            0 => "SUCCESS",
            1 => "2BIG",
            2 => "ACCES",
            3 => "ADDRINUSE",
            4 => "ADDRNOTAVAIL",
            5 => "AFNOSUPPORT",
            6 => "AGAIN",
            7 => "ALREADY",
            8 => "BADF",
            9 => "BADMSG",
            10 => "BUSY",
            11 => "CANCELED",
            12 => "CHILD",
            13 => "CONNABORTED",
            14 => "CONNREFUSED",
            15 => "CONNRESET",
            16 => "DEADLK",
            17 => "DESTADDRREQ",
            18 => "DOM",
            19 => "DQUOT",
            20 => "EXIST",
            21 => "FAULT",
            22 => "FBIG",
            23 => "HOSTUNREACH",
            24 => "IDRM",
            25 => "ILSEQ",
            26 => "INPROGRESS",
            27 => "INTR",
            28 => "INVAL",
            29 => "IO",
            30 => "ISCONN",
            31 => "ISDIR",
            32 => "LOOP",
            33 => "MFILE",
            34 => "MLINK",
            35 => "MSGSIZE",
            36 => "MULTIHOP",
            37 => "NAMETOOLONG",
            38 => "NETDOWN",
            39 => "NETRESET",
            40 => "NETUNREACH",
            41 => "NFILE",
            42 => "NOBUFS",
            43 => "NODEV",
            44 => "NOENT",
            45 => "NOEXEC",
            46 => "NOLCK",
            47 => "NOLINK",
            48 => "NOMEM",
            49 => "NOMSG",
            50 => "NOPROTOOPT",
            51 => "NOSPC",
            52 => "NOSYS",
            53 => "NOTCONN",
            54 => "NOTDIR",
            55 => "NOTEMPTY",
            56 => "NOTRECOVERABLE",
            57 => "NOTSOCK",
            58 => "NOTSUP",
            59 => "NOTTY",
            60 => "NXIO",
            61 => "OVERFLOW",
            62 => "OWNERDEAD",
            63 => "PERM",
            64 => "PIPE",
            65 => "PROTO",
            66 => "PROTONOSUPPORT",
            67 => "PROTOTYPE",
            68 => "RANGE",
            69 => "ROFS",
            70 => "SPIPE",
            71 => "SRCH",
            72 => "STALE",
            73 => "TIMEDOUT",
            74 => "TXTBSY",
            75 => "XDEV",
            76 => "NOTCAPABLE",
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
    pub fn message(&self) -> &'static str {
        match self.0 {
            0 => "No error occurred. System call completed successfully.",
            1 => "Argument list too long.",
            2 => "Permission denied.",
            3 => "Address in use.",
            4 => "Address not available.",
            5 => "Address family not supported.",
            6 => "Resource unavailable, or operation would block.",
            7 => "Connection already in progress.",
            8 => "Bad file descriptor.",
            9 => "Bad message.",
            10 => "Device or resource busy.",
            11 => "Operation canceled.",
            12 => "No child processes.",
            13 => "Connection aborted.",
            14 => "Connection refused.",
            15 => "Connection reset.",
            16 => "Resource deadlock would occur.",
            17 => "Destination address required.",
            18 => "Mathematics argument out of domain of function.",
            19 => "Reserved.",
            20 => "File exists.",
            21 => "Bad address.",
            22 => "File too large.",
            23 => "Host is unreachable.",
            24 => "Identifier removed.",
            25 => "Illegal byte sequence.",
            26 => "Operation in progress.",
            27 => "Interrupted function.",
            28 => "Invalid argument.",
            29 => "I/O error.",
            30 => "Socket is connected.",
            31 => "Is a directory.",
            32 => "Too many levels of symbolic links.",
            33 => "File descriptor value too large.",
            34 => "Too many links.",
            35 => "Message too large.",
            36 => "Reserved.",
            37 => "Filename too long.",
            38 => "Network is down.",
            39 => "Connection aborted by network.",
            40 => "Network unreachable.",
            41 => "Too many files open in system.",
            42 => "No buffer space available.",
            43 => "No such device.",
            44 => "No such file or directory.",
            45 => "Executable file format error.",
            46 => "No locks available.",
            47 => "Reserved.",
            48 => "Not enough space.",
            49 => "No message of the desired type.",
            50 => "Protocol not available.",
            51 => "No space left on device.",
            52 => "Function not supported.",
            53 => "The socket is not connected.",
            54 => "Not a directory or a symbolic link to a directory.",
            55 => "Directory not empty.",
            56 => "State not recoverable.",
            57 => "Not a socket.",
            58 => "Not supported, or operation not supported on socket.",
            59 => "Inappropriate I/O control operation.",
            60 => "No such device or address.",
            61 => "Value too large to be stored in data type.",
            62 => "Previous owner died.",
            63 => "Operation not permitted.",
            64 => "Broken pipe.",
            65 => "Protocol error.",
            66 => "Protocol not supported.",
            67 => "Protocol wrong type for socket.",
            68 => "Result too large.",
            69 => "Read-only file system.",
            70 => "Invalid seek.",
            71 => "No such process.",
            72 => "Reserved.",
            73 => "Connection timed out.",
            74 => "Text file busy.",
            75 => "Cross-device link.",
            76 => "Extension: Capabilities insufficient.",
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
}
impl fmt::Debug for Errno {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.debug_struct("Errno")
            .field("code", &self.0)
            .field("name", &self.name())
            .field("message", &self.message())
            .finish()
    }
}
impl fmt::Display for Errno {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{} (error {})", self.name(), self.0)
    }
}

#[cfg(feature = "std")]
extern crate std;
#[cfg(feature = "std")]
impl std::error::Error for Errno {}

pub type Rights = u64;



pub const RIGHTS_FD_DATASYNC: Rights = 1 << 0;


pub const RIGHTS_FD_READ: Rights = 1 << 1;

pub const RIGHTS_FD_SEEK: Rights = 1 << 2;

pub const RIGHTS_FD_FDSTAT_SET_FLAGS: Rights = 1 << 3;



pub const RIGHTS_FD_SYNC: Rights = 1 << 4;



pub const RIGHTS_FD_TELL: Rights = 1 << 5;


pub const RIGHTS_FD_WRITE: Rights = 1 << 6;

pub const RIGHTS_FD_ADVISE: Rights = 1 << 7;

pub const RIGHTS_FD_ALLOCATE: Rights = 1 << 8;

pub const RIGHTS_PATH_CREATE_DIRECTORY: Rights = 1 << 9;

pub const RIGHTS_PATH_CREATE_FILE: Rights = 1 << 10;


pub const RIGHTS_PATH_LINK_SOURCE: Rights = 1 << 11;


pub const RIGHTS_PATH_LINK_TARGET: Rights = 1 << 12;

pub const RIGHTS_PATH_OPEN: Rights = 1 << 13;

pub const RIGHTS_FD_READDIR: Rights = 1 << 14;

pub const RIGHTS_PATH_READLINK: Rights = 1 << 15;

pub const RIGHTS_PATH_RENAME_SOURCE: Rights = 1 << 16;

pub const RIGHTS_PATH_RENAME_TARGET: Rights = 1 << 17;

pub const RIGHTS_PATH_FILESTAT_GET: Rights = 1 << 18;


pub const RIGHTS_PATH_FILESTAT_SET_SIZE: Rights = 1 << 19;

pub const RIGHTS_PATH_FILESTAT_SET_TIMES: Rights = 1 << 20;

pub const RIGHTS_FD_FILESTAT_GET: Rights = 1 << 21;

pub const RIGHTS_FD_FILESTAT_SET_SIZE: Rights = 1 << 22;

pub const RIGHTS_FD_FILESTAT_SET_TIMES: Rights = 1 << 23;

pub const RIGHTS_PATH_SYMLINK: Rights = 1 << 24;

pub const RIGHTS_PATH_REMOVE_DIRECTORY: Rights = 1 << 25;

pub const RIGHTS_PATH_UNLINK_FILE: Rights = 1 << 26;


pub const RIGHTS_POLL_FD_READWRITE: Rights = 1 << 27;

pub const RIGHTS_SOCK_SHUTDOWN: Rights = 1 << 28;

pub const RIGHTS_SOCK_ACCEPT: Rights = 1 << 29;

pub type Fd = u32;
#[repr(C)]
#[derive(Copy, Clone, Debug)]
pub struct Iovec {
    
    pub buf: *mut u8,
    
    pub buf_len: Size,
}
#[repr(C)]
#[derive(Copy, Clone, Debug)]
pub struct Ciovec {
    
    pub buf: *const u8,
    
    pub buf_len: Size,
}
pub type IovecArray<'a> = &'a [Iovec];
pub type CiovecArray<'a> = &'a [Ciovec];
pub type Filedelta = i64;
#[repr(transparent)]
#[derive(Copy, Clone, Hash, Eq, PartialEq, Ord, PartialOrd)]
pub struct Whence(u8);

pub const WHENCE_SET: Whence = Whence(0);

pub const WHENCE_CUR: Whence = Whence(1);

pub const WHENCE_END: Whence = Whence(2);
impl Whence {
    pub const fn raw(&self) -> u8 {
        self.0
    }

    pub fn name(&self) -> &'static str {
        match self.0 {
            0 => "SET",
            1 => "CUR",
            2 => "END",
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
    pub fn message(&self) -> &'static str {
        match self.0 {
            0 => "Seek relative to start-of-file.",
            1 => "Seek relative to current position.",
            2 => "Seek relative to end-of-file.",
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
}
impl fmt::Debug for Whence {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.debug_struct("Whence")
            .field("code", &self.0)
            .field("name", &self.name())
            .field("message", &self.message())
            .finish()
    }
}

pub type Dircookie = u64;
pub type Dirnamlen = u32;
pub type Inode = u64;
#[repr(transparent)]
#[derive(Copy, Clone, Hash, Eq, PartialEq, Ord, PartialOrd)]
pub struct Filetype(u8);

pub const FILETYPE_UNKNOWN: Filetype = Filetype(0);

pub const FILETYPE_BLOCK_DEVICE: Filetype = Filetype(1);

pub const FILETYPE_CHARACTER_DEVICE: Filetype = Filetype(2);

pub const FILETYPE_DIRECTORY: Filetype = Filetype(3);

pub const FILETYPE_REGULAR_FILE: Filetype = Filetype(4);

pub const FILETYPE_SOCKET_DGRAM: Filetype = Filetype(5);

pub const FILETYPE_SOCKET_STREAM: Filetype = Filetype(6);

pub const FILETYPE_SYMBOLIC_LINK: Filetype = Filetype(7);
impl Filetype {
    pub const fn raw(&self) -> u8 {
        self.0
    }

    pub fn name(&self) -> &'static str {
        match self.0 {
            0 => "UNKNOWN",
            1 => "BLOCK_DEVICE",
            2 => "CHARACTER_DEVICE",
            3 => "DIRECTORY",
            4 => "REGULAR_FILE",
            5 => "SOCKET_DGRAM",
            6 => "SOCKET_STREAM",
            7 => "SYMBOLIC_LINK",
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
    pub fn message(&self) -> &'static str {
        match self.0 {0 => "The type of the file descriptor or file is unknown or is different from any of the other types specified.",1 => "The file descriptor or file refers to a block device inode.",2 => "The file descriptor or file refers to a character device inode.",3 => "The file descriptor or file refers to a directory inode.",4 => "The file descriptor or file refers to a regular file inode.",5 => "The file descriptor or file refers to a datagram socket.",6 => "The file descriptor or file refers to a byte-stream socket.",7 => "The file refers to a symbolic link inode.",_ => unsafe { core::hint::unreachable_unchecked() },}
    }
}
impl fmt::Debug for Filetype {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.debug_struct("Filetype")
            .field("code", &self.0)
            .field("name", &self.name())
            .field("message", &self.message())
            .finish()
    }
}

#[repr(C)]
#[derive(Copy, Clone, Debug)]
pub struct Dirent {
    
    pub d_next: Dircookie,
    
    pub d_ino: Inode,
    
    pub d_namlen: Dirnamlen,
    
    pub d_type: Filetype,
}
#[repr(transparent)]
#[derive(Copy, Clone, Hash, Eq, PartialEq, Ord, PartialOrd)]
pub struct Advice(u8);

pub const ADVICE_NORMAL: Advice = Advice(0);

pub const ADVICE_SEQUENTIAL: Advice = Advice(1);

pub const ADVICE_RANDOM: Advice = Advice(2);

pub const ADVICE_WILLNEED: Advice = Advice(3);

pub const ADVICE_DONTNEED: Advice = Advice(4);

pub const ADVICE_NOREUSE: Advice = Advice(5);
impl Advice {
    pub const fn raw(&self) -> u8 {
        self.0
    }

    pub fn name(&self) -> &'static str {
        match self.0 {
            0 => "NORMAL",
            1 => "SEQUENTIAL",
            2 => "RANDOM",
            3 => "WILLNEED",
            4 => "DONTNEED",
            5 => "NOREUSE",
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
    pub fn message(&self) -> &'static str {
        match self.0 {0 => "The application has no advice to give on its behavior with respect to the specified data.",1 => "The application expects to access the specified data sequentially from lower offsets to higher offsets.",2 => "The application expects to access the specified data in a random order.",3 => "The application expects to access the specified data in the near future.",4 => "The application expects that it will not access the specified data in the near future.",5 => "The application expects to access the specified data once and then not reuse it thereafter.",_ => unsafe { core::hint::unreachable_unchecked() },}
    }
}
impl fmt::Debug for Advice {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.debug_struct("Advice")
            .field("code", &self.0)
            .field("name", &self.name())
            .field("message", &self.message())
            .finish()
    }
}

pub type Fdflags = u16;

pub const FDFLAGS_APPEND: Fdflags = 1 << 0;

pub const FDFLAGS_DSYNC: Fdflags = 1 << 1;

pub const FDFLAGS_NONBLOCK: Fdflags = 1 << 2;

pub const FDFLAGS_RSYNC: Fdflags = 1 << 3;



pub const FDFLAGS_SYNC: Fdflags = 1 << 4;

#[repr(C)]
#[derive(Copy, Clone, Debug)]
pub struct Fdstat {
    
    pub fs_filetype: Filetype,
    
    pub fs_flags: Fdflags,
    
    pub fs_rights_base: Rights,
    
    
    pub fs_rights_inheriting: Rights,
}
pub type Device = u64;
pub type Fstflags = u16;

pub const FSTFLAGS_ATIM: Fstflags = 1 << 0;

pub const FSTFLAGS_ATIM_NOW: Fstflags = 1 << 1;

pub const FSTFLAGS_MTIM: Fstflags = 1 << 2;

pub const FSTFLAGS_MTIM_NOW: Fstflags = 1 << 3;

pub type Lookupflags = u32;

pub const LOOKUPFLAGS_SYMLINK_FOLLOW: Lookupflags = 1 << 0;

pub type Oflags = u16;

pub const OFLAGS_CREAT: Oflags = 1 << 0;

pub const OFLAGS_DIRECTORY: Oflags = 1 << 1;

pub const OFLAGS_EXCL: Oflags = 1 << 2;

pub const OFLAGS_TRUNC: Oflags = 1 << 3;

pub type Linkcount = u64;
#[repr(C)]
#[derive(Copy, Clone, Debug)]
pub struct Filestat {
    
    pub dev: Device,
    
    pub ino: Inode,
    
    pub filetype: Filetype,
    
    pub nlink: Linkcount,
    
    pub size: Filesize,
    
    pub atim: Timestamp,
    
    pub mtim: Timestamp,
    
    pub ctim: Timestamp,
}
pub type Userdata = u64;
#[repr(transparent)]
#[derive(Copy, Clone, Hash, Eq, PartialEq, Ord, PartialOrd)]
pub struct Eventtype(u8);


pub const EVENTTYPE_CLOCK: Eventtype = Eventtype(0);


pub const EVENTTYPE_FD_READ: Eventtype = Eventtype(1);


pub const EVENTTYPE_FD_WRITE: Eventtype = Eventtype(2);
impl Eventtype {
    pub const fn raw(&self) -> u8 {
        self.0
    }

    pub fn name(&self) -> &'static str {
        match self.0 {
            0 => "CLOCK",
            1 => "FD_READ",
            2 => "FD_WRITE",
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
    pub fn message(&self) -> &'static str {
        match self.0 {
            0 => {
                "The time value of clock `subscription_clock::id` has
reached timestamp `subscription_clock::timeout`."
            }
            1 => {
                "File descriptor `subscription_fd_readwrite::file_descriptor` has data
available for reading. This event always triggers for regular files."
            }
            2 => {
                "File descriptor `subscription_fd_readwrite::file_descriptor` has capacity
available for writing. This event always triggers for regular files."
            }
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
}
impl fmt::Debug for Eventtype {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.debug_struct("Eventtype")
            .field("code", &self.0)
            .field("name", &self.name())
            .field("message", &self.message())
            .finish()
    }
}

pub type Eventrwflags = u16;

pub const EVENTRWFLAGS_FD_READWRITE_HANGUP: Eventrwflags = 1 << 0;

#[repr(C)]
#[derive(Copy, Clone, Debug)]
pub struct EventFdReadwrite {
    
    pub nbytes: Filesize,
    
    pub flags: Eventrwflags,
}
#[repr(C)]
#[derive(Copy, Clone, Debug)]
pub struct Event {
    
    pub userdata: Userdata,
    
    pub error: Errno,
    
    pub type_: Eventtype,
    
    
    pub fd_readwrite: EventFdReadwrite,
}
pub type Subclockflags = u16;





pub const SUBCLOCKFLAGS_SUBSCRIPTION_CLOCK_ABSTIME: Subclockflags = 1 << 0;

#[repr(C)]
#[derive(Copy, Clone, Debug)]
pub struct SubscriptionClock {
    
    pub id: Clockid,
    
    pub timeout: Timestamp,
    
    
    pub precision: Timestamp,
    
    pub flags: Subclockflags,
}
#[repr(C)]
#[derive(Copy, Clone, Debug)]
pub struct SubscriptionFdReadwrite {
    
    pub file_descriptor: Fd,
}
#[repr(C)]
#[derive(Copy, Clone)]
pub union SubscriptionUU {
    pub clock: SubscriptionClock,
    pub fd_read: SubscriptionFdReadwrite,
    pub fd_write: SubscriptionFdReadwrite,
}
#[repr(C)]
#[derive(Copy, Clone)]
pub struct SubscriptionU {
    pub tag: u8,
    pub u: SubscriptionUU,
}

#[repr(C)]
#[derive(Copy, Clone)]
pub struct Subscription {
    
    
    pub userdata: Userdata,
    
    pub u: SubscriptionU,
}
pub type Exitcode = u32;
#[repr(transparent)]
#[derive(Copy, Clone, Hash, Eq, PartialEq, Ord, PartialOrd)]
pub struct Signal(u8);


pub const SIGNAL_NONE: Signal = Signal(0);


pub const SIGNAL_HUP: Signal = Signal(1);


pub const SIGNAL_INT: Signal = Signal(2);


pub const SIGNAL_QUIT: Signal = Signal(3);


pub const SIGNAL_ILL: Signal = Signal(4);


pub const SIGNAL_TRAP: Signal = Signal(5);


pub const SIGNAL_ABRT: Signal = Signal(6);


pub const SIGNAL_BUS: Signal = Signal(7);


pub const SIGNAL_FPE: Signal = Signal(8);


pub const SIGNAL_KILL: Signal = Signal(9);


pub const SIGNAL_USR1: Signal = Signal(10);


pub const SIGNAL_SEGV: Signal = Signal(11);


pub const SIGNAL_USR2: Signal = Signal(12);


pub const SIGNAL_PIPE: Signal = Signal(13);


pub const SIGNAL_ALRM: Signal = Signal(14);


pub const SIGNAL_TERM: Signal = Signal(15);


pub const SIGNAL_CHLD: Signal = Signal(16);


pub const SIGNAL_CONT: Signal = Signal(17);


pub const SIGNAL_STOP: Signal = Signal(18);


pub const SIGNAL_TSTP: Signal = Signal(19);


pub const SIGNAL_TTIN: Signal = Signal(20);


pub const SIGNAL_TTOU: Signal = Signal(21);


pub const SIGNAL_URG: Signal = Signal(22);


pub const SIGNAL_XCPU: Signal = Signal(23);


pub const SIGNAL_XFSZ: Signal = Signal(24);


pub const SIGNAL_VTALRM: Signal = Signal(25);


pub const SIGNAL_PROF: Signal = Signal(26);


pub const SIGNAL_WINCH: Signal = Signal(27);


pub const SIGNAL_POLL: Signal = Signal(28);


pub const SIGNAL_PWR: Signal = Signal(29);


pub const SIGNAL_SYS: Signal = Signal(30);
impl Signal {
    pub const fn raw(&self) -> u8 {
        self.0
    }

    pub fn name(&self) -> &'static str {
        match self.0 {
            0 => "NONE",
            1 => "HUP",
            2 => "INT",
            3 => "QUIT",
            4 => "ILL",
            5 => "TRAP",
            6 => "ABRT",
            7 => "BUS",
            8 => "FPE",
            9 => "KILL",
            10 => "USR1",
            11 => "SEGV",
            12 => "USR2",
            13 => "PIPE",
            14 => "ALRM",
            15 => "TERM",
            16 => "CHLD",
            17 => "CONT",
            18 => "STOP",
            19 => "TSTP",
            20 => "TTIN",
            21 => "TTOU",
            22 => "URG",
            23 => "XCPU",
            24 => "XFSZ",
            25 => "VTALRM",
            26 => "PROF",
            27 => "WINCH",
            28 => "POLL",
            29 => "PWR",
            30 => "SYS",
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
    pub fn message(&self) -> &'static str {
        match self.0 {
            0 => {
                "No signal. Note that POSIX has special semantics for `kill(pid, 0)`,
so this value is reserved."
            }
            1 => {
                "Hangup.
Action: Terminates the process."
            }
            2 => {
                "Terminate interrupt signal.
Action: Terminates the process."
            }
            3 => {
                "Terminal quit signal.
Action: Terminates the process."
            }
            4 => {
                "Illegal instruction.
Action: Terminates the process."
            }
            5 => {
                "Trace/breakpoint trap.
Action: Terminates the process."
            }
            6 => {
                "Process abort signal.
Action: Terminates the process."
            }
            7 => {
                "Access to an undefined portion of a memory object.
Action: Terminates the process."
            }
            8 => {
                "Erroneous arithmetic operation.
Action: Terminates the process."
            }
            9 => {
                "Kill.
Action: Terminates the process."
            }
            10 => {
                "User-defined signal 1.
Action: Terminates the process."
            }
            11 => {
                "Invalid memory reference.
Action: Terminates the process."
            }
            12 => {
                "User-defined signal 2.
Action: Terminates the process."
            }
            13 => {
                "Write on a pipe with no one to read it.
Action: Ignored."
            }
            14 => {
                "Alarm clock.
Action: Terminates the process."
            }
            15 => {
                "Termination signal.
Action: Terminates the process."
            }
            16 => {
                "Child process terminated, stopped, or continued.
Action: Ignored."
            }
            17 => {
                "Continue executing, if stopped.
Action: Continues executing, if stopped."
            }
            18 => {
                "Stop executing.
Action: Stops executing."
            }
            19 => {
                "Terminal stop signal.
Action: Stops executing."
            }
            20 => {
                "Background process attempting read.
Action: Stops executing."
            }
            21 => {
                "Background process attempting write.
Action: Stops executing."
            }
            22 => {
                "High bandwidth data is available at a socket.
Action: Ignored."
            }
            23 => {
                "CPU time limit exceeded.
Action: Terminates the process."
            }
            24 => {
                "File size limit exceeded.
Action: Terminates the process."
            }
            25 => {
                "Virtual timer expired.
Action: Terminates the process."
            }
            26 => {
                "Profiling timer expired.
Action: Terminates the process."
            }
            27 => {
                "Window changed.
Action: Ignored."
            }
            28 => {
                "I/O possible.
Action: Terminates the process."
            }
            29 => {
                "Power failure.
Action: Terminates the process."
            }
            30 => {
                "Bad system call.
Action: Terminates the process."
            }
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
}
impl fmt::Debug for Signal {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.debug_struct("Signal")
            .field("code", &self.0)
            .field("name", &self.name())
            .field("message", &self.message())
            .finish()
    }
}

pub type Riflags = u16;

pub const RIFLAGS_RECV_PEEK: Riflags = 1 << 0;

pub const RIFLAGS_RECV_WAITALL: Riflags = 1 << 1;

pub type Roflags = u16;

pub const ROFLAGS_RECV_DATA_TRUNCATED: Roflags = 1 << 0;

pub type Siflags = u16;
pub type Sdflags = u8;

pub const SDFLAGS_RD: Sdflags = 1 << 0;

pub const SDFLAGS_WR: Sdflags = 1 << 1;

#[repr(transparent)]
#[derive(Copy, Clone, Hash, Eq, PartialEq, Ord, PartialOrd)]
pub struct Preopentype(u8);

pub const PREOPENTYPE_DIR: Preopentype = Preopentype(0);
impl Preopentype {
    pub const fn raw(&self) -> u8 {
        self.0
    }

    pub fn name(&self) -> &'static str {
        match self.0 {
            0 => "DIR",
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
    pub fn message(&self) -> &'static str {
        match self.0 {
            0 => "A pre-opened directory.",
            _ => unsafe { core::hint::unreachable_unchecked() },
        }
    }
}
impl fmt::Debug for Preopentype {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.debug_struct("Preopentype")
            .field("code", &self.0)
            .field("name", &self.name())
            .field("message", &self.message())
            .finish()
    }
}

#[repr(C)]
#[derive(Copy, Clone, Debug)]
pub struct PrestatDir {
    
    pub pr_name_len: Size,
}
#[repr(C)]
#[derive(Copy, Clone)]
pub union PrestatU {
    pub dir: PrestatDir,
}
#[repr(C)]
#[derive(Copy, Clone)]
pub struct Prestat {
    pub tag: u8,
    pub u: PrestatU,
}




pub unsafe fn args_get(argv: *mut *mut u8, argv_buf: *mut u8) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::args_get(argv as i32, argv_buf as i32);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}



/// ## Return



pub unsafe fn args_sizes_get() -> Result<(Size, Size), Errno> {
    let mut rp0 = MaybeUninit::<Size>::uninit();
    let mut rp1 = MaybeUninit::<Size>::uninit();
    let ret =
        wasi_snapshot_preview1::args_sizes_get(rp0.as_mut_ptr() as i32, rp1.as_mut_ptr() as i32);
    match ret {
        0 => Ok((
            core::ptr::read(rp0.as_mut_ptr() as i32 as *const Size),
            core::ptr::read(rp1.as_mut_ptr() as i32 as *const Size),
        )),
        _ => Err(Errno(ret as u16)),
    }
}




pub unsafe fn environ_get(environ: *mut *mut u8, environ_buf: *mut u8) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::environ_get(environ as i32, environ_buf as i32);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}



/// ## Return



pub unsafe fn environ_sizes_get() -> Result<(Size, Size), Errno> {
    let mut rp0 = MaybeUninit::<Size>::uninit();
    let mut rp1 = MaybeUninit::<Size>::uninit();
    let ret =
        wasi_snapshot_preview1::environ_sizes_get(rp0.as_mut_ptr() as i32, rp1.as_mut_ptr() as i32);
    match ret {
        0 => Ok((
            core::ptr::read(rp0.as_mut_ptr() as i32 as *const Size),
            core::ptr::read(rp1.as_mut_ptr() as i32 as *const Size),
        )),
        _ => Err(Errno(ret as u16)),
    }
}






/// ## Parameters



/// ## Return


pub unsafe fn clock_res_get(id: Clockid) -> Result<Timestamp, Errno> {
    let mut rp0 = MaybeUninit::<Timestamp>::uninit();
    let ret = wasi_snapshot_preview1::clock_res_get(id.0 as i32, rp0.as_mut_ptr() as i32);
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Timestamp)),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters




/// ## Return


pub unsafe fn clock_time_get(id: Clockid, precision: Timestamp) -> Result<Timestamp, Errno> {
    let mut rp0 = MaybeUninit::<Timestamp>::uninit();
    let ret = wasi_snapshot_preview1::clock_time_get(
        id.0 as i32,
        precision as i64,
        rp0.as_mut_ptr() as i32,
    );
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Timestamp)),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters




pub unsafe fn fd_advise(
    fd: Fd,
    offset: Filesize,
    len: Filesize,
    advice: Advice,
) -> Result<(), Errno> {
    let ret =
        wasi_snapshot_preview1::fd_advise(fd as i32, offset as i64, len as i64, advice.0 as i32);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters



pub unsafe fn fd_allocate(fd: Fd, offset: Filesize, len: Filesize) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::fd_allocate(fd as i32, offset as i64, len as i64);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}



pub unsafe fn fd_close(fd: Fd) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::fd_close(fd as i32);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}



pub unsafe fn fd_datasync(fd: Fd) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::fd_datasync(fd as i32);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Return


pub unsafe fn fd_fdstat_get(fd: Fd) -> Result<Fdstat, Errno> {
    let mut rp0 = MaybeUninit::<Fdstat>::uninit();
    let ret = wasi_snapshot_preview1::fd_fdstat_get(fd as i32, rp0.as_mut_ptr() as i32);
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Fdstat)),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters


pub unsafe fn fd_fdstat_set_flags(fd: Fd, flags: Fdflags) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::fd_fdstat_set_flags(fd as i32, flags as i32);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters


pub unsafe fn fd_fdstat_set_rights(
    fd: Fd,
    fs_rights_base: Rights,
    fs_rights_inheriting: Rights,
) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::fd_fdstat_set_rights(
        fd as i32,
        fs_rights_base as i64,
        fs_rights_inheriting as i64,
    );
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}



/// ## Return


pub unsafe fn fd_filestat_get(fd: Fd) -> Result<Filestat, Errno> {
    let mut rp0 = MaybeUninit::<Filestat>::uninit();
    let ret = wasi_snapshot_preview1::fd_filestat_get(fd as i32, rp0.as_mut_ptr() as i32);
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Filestat)),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters


pub unsafe fn fd_filestat_set_size(fd: Fd, size: Filesize) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::fd_filestat_set_size(fd as i32, size as i64);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters




pub unsafe fn fd_filestat_set_times(
    fd: Fd,
    atim: Timestamp,
    mtim: Timestamp,
    fst_flags: Fstflags,
) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::fd_filestat_set_times(
        fd as i32,
        atim as i64,
        mtim as i64,
        fst_flags as i32,
    );
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters




/// ## Return


pub unsafe fn fd_pread(fd: Fd, iovs: IovecArray<'_>, offset: Filesize) -> Result<Size, Errno> {
    let mut rp0 = MaybeUninit::<Size>::uninit();
    let ret = wasi_snapshot_preview1::fd_pread(
        fd as i32,
        iovs.as_ptr() as i32,
        iovs.len() as i32,
        offset as i64,
        rp0.as_mut_ptr() as i32,
    );
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Size)),
        _ => Err(Errno(ret as u16)),
    }
}



/// ## Return


pub unsafe fn fd_prestat_get(fd: Fd) -> Result<Prestat, Errno> {
    let mut rp0 = MaybeUninit::<Prestat>::uninit();
    let ret = wasi_snapshot_preview1::fd_prestat_get(fd as i32, rp0.as_mut_ptr() as i32);
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Prestat)),
        _ => Err(Errno(ret as u16)),
    }
}



/// ## Parameters


pub unsafe fn fd_prestat_dir_name(fd: Fd, path: *mut u8, path_len: Size) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::fd_prestat_dir_name(fd as i32, path as i32, path_len as i32);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters




/// ## Return


pub unsafe fn fd_pwrite(fd: Fd, iovs: CiovecArray<'_>, offset: Filesize) -> Result<Size, Errno> {
    let mut rp0 = MaybeUninit::<Size>::uninit();
    let ret = wasi_snapshot_preview1::fd_pwrite(
        fd as i32,
        iovs.as_ptr() as i32,
        iovs.len() as i32,
        offset as i64,
        rp0.as_mut_ptr() as i32,
    );
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Size)),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters



/// ## Return


pub unsafe fn fd_read(fd: Fd, iovs: IovecArray<'_>) -> Result<Size, Errno> {
    let mut rp0 = MaybeUninit::<Size>::uninit();
    let ret = wasi_snapshot_preview1::fd_read(
        fd as i32,
        iovs.as_ptr() as i32,
        iovs.len() as i32,
        rp0.as_mut_ptr() as i32,
    );
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Size)),
        _ => Err(Errno(ret as u16)),
    }
}











/// ## Parameters




/// ## Return


pub unsafe fn fd_readdir(
    fd: Fd,
    buf: *mut u8,
    buf_len: Size,
    cookie: Dircookie,
) -> Result<Size, Errno> {
    let mut rp0 = MaybeUninit::<Size>::uninit();
    let ret = wasi_snapshot_preview1::fd_readdir(
        fd as i32,
        buf as i32,
        buf_len as i32,
        cookie as i64,
        rp0.as_mut_ptr() as i32,
    );
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Size)),
        _ => Err(Errno(ret as u16)),
    }
}










/// ## Parameters


pub unsafe fn fd_renumber(fd: Fd, to: Fd) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::fd_renumber(fd as i32, to as i32);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters




/// ## Return


pub unsafe fn fd_seek(fd: Fd, offset: Filedelta, whence: Whence) -> Result<Filesize, Errno> {
    let mut rp0 = MaybeUninit::<Filesize>::uninit();
    let ret = wasi_snapshot_preview1::fd_seek(
        fd as i32,
        offset,
        whence.0 as i32,
        rp0.as_mut_ptr() as i32,
    );
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Filesize)),
        _ => Err(Errno(ret as u16)),
    }
}



pub unsafe fn fd_sync(fd: Fd) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::fd_sync(fd as i32);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Return


pub unsafe fn fd_tell(fd: Fd) -> Result<Filesize, Errno> {
    let mut rp0 = MaybeUninit::<Filesize>::uninit();
    let ret = wasi_snapshot_preview1::fd_tell(fd as i32, rp0.as_mut_ptr() as i32);
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Filesize)),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters


pub unsafe fn fd_write(fd: Fd, iovs: CiovecArray<'_>) -> Result<Size, Errno> {
    let mut rp0 = MaybeUninit::<Size>::uninit();
    let ret = wasi_snapshot_preview1::fd_write(
        fd as i32,
        iovs.as_ptr() as i32,
        iovs.len() as i32,
        rp0.as_mut_ptr() as i32,
    );
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Size)),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters


pub unsafe fn path_create_directory(fd: Fd, path: &str) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::path_create_directory(
        fd as i32,
        path.as_ptr() as i32,
        path.len() as i32,
    );
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters




/// ## Return


pub unsafe fn path_filestat_get(fd: Fd, flags: Lookupflags, path: &str) -> Result<Filestat, Errno> {
    let mut rp0 = MaybeUninit::<Filestat>::uninit();
    let ret = wasi_snapshot_preview1::path_filestat_get(
        fd as i32,
        flags as i32,
        path.as_ptr() as i32,
        path.len() as i32,
        rp0.as_mut_ptr() as i32,
    );
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Filestat)),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters






pub unsafe fn path_filestat_set_times(
    fd: Fd,
    flags: Lookupflags,
    path: &str,
    atim: Timestamp,
    mtim: Timestamp,
    fst_flags: Fstflags,
) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::path_filestat_set_times(
        fd as i32,
        flags as i32,
        path.as_ptr() as i32,
        path.len() as i32,
        atim as i64,
        mtim as i64,
        fst_flags as i32,
    );
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters





pub unsafe fn path_link(
    old_fd: Fd,
    old_flags: Lookupflags,
    old_path: &str,
    new_fd: Fd,
    new_path: &str,
) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::path_link(
        old_fd as i32,
        old_flags as i32,
        old_path.as_ptr() as i32,
        old_path.len() as i32,
        new_fd as i32,
        new_path.as_ptr() as i32,
        new_path.len() as i32,
    );
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}









/// ## Parameters













/// ## Return


pub unsafe fn path_open(
    fd: Fd,
    dirflags: Lookupflags,
    path: &str,
    oflags: Oflags,
    fs_rights_base: Rights,
    fs_rights_inheriting: Rights,
    fdflags: Fdflags,
) -> Result<Fd, Errno> {
    let mut rp0 = MaybeUninit::<Fd>::uninit();
    let ret = wasi_snapshot_preview1::path_open(
        fd as i32,
        dirflags as i32,
        path.as_ptr() as i32,
        path.len() as i32,
        oflags as i32,
        fs_rights_base as i64,
        fs_rights_inheriting as i64,
        fdflags as i32,
        rp0.as_mut_ptr() as i32,
    );
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Fd)),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters




/// ## Return


pub unsafe fn path_readlink(
    fd: Fd,
    path: &str,
    buf: *mut u8,
    buf_len: Size,
) -> Result<Size, Errno> {
    let mut rp0 = MaybeUninit::<Size>::uninit();
    let ret = wasi_snapshot_preview1::path_readlink(
        fd as i32,
        path.as_ptr() as i32,
        path.len() as i32,
        buf as i32,
        buf_len as i32,
        rp0.as_mut_ptr() as i32,
    );
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Size)),
        _ => Err(Errno(ret as u16)),
    }
}





/// ## Parameters


pub unsafe fn path_remove_directory(fd: Fd, path: &str) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::path_remove_directory(
        fd as i32,
        path.as_ptr() as i32,
        path.len() as i32,
    );
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters




pub unsafe fn path_rename(fd: Fd, old_path: &str, new_fd: Fd, new_path: &str) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::path_rename(
        fd as i32,
        old_path.as_ptr() as i32,
        old_path.len() as i32,
        new_fd as i32,
        new_path.as_ptr() as i32,
        new_path.len() as i32,
    );
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters



pub unsafe fn path_symlink(old_path: &str, fd: Fd, new_path: &str) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::path_symlink(
        old_path.as_ptr() as i32,
        old_path.len() as i32,
        fd as i32,
        new_path.as_ptr() as i32,
        new_path.len() as i32,
    );
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}





/// ## Parameters


pub unsafe fn path_unlink_file(fd: Fd, path: &str) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::path_unlink_file(
        fd as i32,
        path.as_ptr() as i32,
        path.len() as i32,
    );
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}



/// ## Parameters





/// ## Return


pub unsafe fn poll_oneoff(
    in_: *const Subscription,
    out: *mut Event,
    nsubscriptions: Size,
) -> Result<Size, Errno> {
    let mut rp0 = MaybeUninit::<Size>::uninit();
    let ret = wasi_snapshot_preview1::poll_oneoff(
        in_ as i32,
        out as i32,
        nsubscriptions as i32,
        rp0.as_mut_ptr() as i32,
    );
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Size)),
        _ => Err(Errno(ret as u16)),
    }
}





/// ## Parameters


pub unsafe fn proc_exit(rval: Exitcode) {
    wasi_snapshot_preview1::proc_exit(rval as i32);
}




/// ## Parameters


pub unsafe fn proc_raise(sig: Signal) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::proc_raise(sig.0 as i32);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}



pub unsafe fn sched_yield() -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::sched_yield();
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}








/// ## Parameters


pub unsafe fn random_get(buf: *mut u8, buf_len: Size) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::random_get(buf as i32, buf_len as i32);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters




/// ## Return


pub unsafe fn sock_accept(fd: Fd, flags: Fdflags) -> Result<Fd, Errno> {
    let mut rp0 = MaybeUninit::<Fd>::uninit();
    let ret = wasi_snapshot_preview1::sock_accept(fd as i32, flags as i32, rp0.as_mut_ptr() as i32);
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Fd)),
        _ => Err(Errno(ret as u16)),
    }
}





/// ## Parameters




/// ## Return


pub unsafe fn sock_recv(
    fd: Fd,
    ri_data: IovecArray<'_>,
    ri_flags: Riflags,
) -> Result<(Size, Roflags), Errno> {
    let mut rp0 = MaybeUninit::<Size>::uninit();
    let mut rp1 = MaybeUninit::<Roflags>::uninit();
    let ret = wasi_snapshot_preview1::sock_recv(
        fd as i32,
        ri_data.as_ptr() as i32,
        ri_data.len() as i32,
        ri_flags as i32,
        rp0.as_mut_ptr() as i32,
        rp1.as_mut_ptr() as i32,
    );
    match ret {
        0 => Ok((
            core::ptr::read(rp0.as_mut_ptr() as i32 as *const Size),
            core::ptr::read(rp1.as_mut_ptr() as i32 as *const Roflags),
        )),
        _ => Err(Errno(ret as u16)),
    }
}





/// ## Parameters




/// ## Return


pub unsafe fn sock_send(
    fd: Fd,
    si_data: CiovecArray<'_>,
    si_flags: Siflags,
) -> Result<Size, Errno> {
    let mut rp0 = MaybeUninit::<Size>::uninit();
    let ret = wasi_snapshot_preview1::sock_send(
        fd as i32,
        si_data.as_ptr() as i32,
        si_data.len() as i32,
        si_flags as i32,
        rp0.as_mut_ptr() as i32,
    );
    match ret {
        0 => Ok(core::ptr::read(rp0.as_mut_ptr() as i32 as *const Size)),
        _ => Err(Errno(ret as u16)),
    }
}




/// ## Parameters


pub unsafe fn sock_shutdown(fd: Fd, how: Sdflags) -> Result<(), Errno> {
    let ret = wasi_snapshot_preview1::sock_shutdown(fd as i32, how as i32);
    match ret {
        0 => Ok(()),
        _ => Err(Errno(ret as u16)),
    }
}

pub mod wasi_snapshot_preview1 {
    #[link(wasm_import_module = "wasi_snapshot_preview1")]
    extern "C" {
        
        
        
        pub fn args_get(arg0: i32, arg1: i32) -> i32;
        
        pub fn args_sizes_get(arg0: i32, arg1: i32) -> i32;
        
        
        
        pub fn environ_get(arg0: i32, arg1: i32) -> i32;
        
        pub fn environ_sizes_get(arg0: i32, arg1: i32) -> i32;
        
        
        
        
        pub fn clock_res_get(arg0: i32, arg1: i32) -> i32;
        
        
        pub fn clock_time_get(arg0: i32, arg1: i64, arg2: i32) -> i32;
        
        
        pub fn fd_advise(arg0: i32, arg1: i64, arg2: i64, arg3: i32) -> i32;
        
        
        pub fn fd_allocate(arg0: i32, arg1: i64, arg2: i64) -> i32;
        
        
        pub fn fd_close(arg0: i32) -> i32;
        
        
        pub fn fd_datasync(arg0: i32) -> i32;
        
        
        pub fn fd_fdstat_get(arg0: i32, arg1: i32) -> i32;
        
        
        pub fn fd_fdstat_set_flags(arg0: i32, arg1: i32) -> i32;
        
        
        pub fn fd_fdstat_set_rights(arg0: i32, arg1: i64, arg2: i64) -> i32;
        
        pub fn fd_filestat_get(arg0: i32, arg1: i32) -> i32;
        
        
        pub fn fd_filestat_set_size(arg0: i32, arg1: i64) -> i32;
        
        
        pub fn fd_filestat_set_times(arg0: i32, arg1: i64, arg2: i64, arg3: i32) -> i32;
        
        
        pub fn fd_pread(arg0: i32, arg1: i32, arg2: i32, arg3: i64, arg4: i32) -> i32;
        
        pub fn fd_prestat_get(arg0: i32, arg1: i32) -> i32;
        
        pub fn fd_prestat_dir_name(arg0: i32, arg1: i32, arg2: i32) -> i32;
        
        
        pub fn fd_pwrite(arg0: i32, arg1: i32, arg2: i32, arg3: i64, arg4: i32) -> i32;
        
        
        pub fn fd_read(arg0: i32, arg1: i32, arg2: i32, arg3: i32) -> i32;
        
        
        
        
        
        
        
        
        
        pub fn fd_readdir(arg0: i32, arg1: i32, arg2: i32, arg3: i64, arg4: i32) -> i32;
        
        
        
        
        
        
        
        
        pub fn fd_renumber(arg0: i32, arg1: i32) -> i32;
        
        
        pub fn fd_seek(arg0: i32, arg1: i64, arg2: i32, arg3: i32) -> i32;
        
        
        pub fn fd_sync(arg0: i32) -> i32;
        
        
        pub fn fd_tell(arg0: i32, arg1: i32) -> i32;
        
        
        pub fn fd_write(arg0: i32, arg1: i32, arg2: i32, arg3: i32) -> i32;
        
        
        pub fn path_create_directory(arg0: i32, arg1: i32, arg2: i32) -> i32;
        
        
        pub fn path_filestat_get(arg0: i32, arg1: i32, arg2: i32, arg3: i32, arg4: i32) -> i32;
        
        
        pub fn path_filestat_set_times(
            arg0: i32,
            arg1: i32,
            arg2: i32,
            arg3: i32,
            arg4: i64,
            arg5: i64,
            arg6: i32,
        ) -> i32;
        
        
        pub fn path_link(
            arg0: i32,
            arg1: i32,
            arg2: i32,
            arg3: i32,
            arg4: i32,
            arg5: i32,
            arg6: i32,
        ) -> i32;
        
        
        
        
        
        
        
        pub fn path_open(
            arg0: i32,
            arg1: i32,
            arg2: i32,
            arg3: i32,
            arg4: i32,
            arg5: i64,
            arg6: i64,
            arg7: i32,
            arg8: i32,
        ) -> i32;
        
        
        pub fn path_readlink(
            arg0: i32,
            arg1: i32,
            arg2: i32,
            arg3: i32,
            arg4: i32,
            arg5: i32,
        ) -> i32;
        
        
        
        pub fn path_remove_directory(arg0: i32, arg1: i32, arg2: i32) -> i32;
        
        
        pub fn path_rename(arg0: i32, arg1: i32, arg2: i32, arg3: i32, arg4: i32, arg5: i32)
            -> i32;
        
        
        pub fn path_symlink(arg0: i32, arg1: i32, arg2: i32, arg3: i32, arg4: i32) -> i32;
        
        
        
        pub fn path_unlink_file(arg0: i32, arg1: i32, arg2: i32) -> i32;
        
        pub fn poll_oneoff(arg0: i32, arg1: i32, arg2: i32, arg3: i32) -> i32;
        
        
        
        pub fn proc_exit(arg0: i32) -> !;
        
        
        pub fn proc_raise(arg0: i32) -> i32;
        
        
        pub fn sched_yield() -> i32;
        
        
        
        
        
        
        pub fn random_get(arg0: i32, arg1: i32) -> i32;
        
        
        pub fn sock_accept(arg0: i32, arg1: i32, arg2: i32) -> i32;
        
        
        
        pub fn sock_recv(arg0: i32, arg1: i32, arg2: i32, arg3: i32, arg4: i32, arg5: i32) -> i32;
        
        
        
        pub fn sock_send(arg0: i32, arg1: i32, arg2: i32, arg3: i32, arg4: i32) -> i32;
        
        
        pub fn sock_shutdown(arg0: i32, arg1: i32) -> i32;
    }
}
