################################################################################
# Performs two one-sided tests for proportion equivalence
# Author: Alexis Dinno <alexis.dinno@pdx.edu>
# version 3.1.9
# Date: Feb 06, 2026


### ADD by.name and by.values as tost.rank.sum does

equivalence.types <- c("delta", "epsilon")

continuity.correction.methods <- c("none", "yates", "ha")

tost.pr <- function(
  x, 
  y=NULL, 
  by=NULL, 
  by.names=NULL, 
  p0=NA, 
  eqv.type = equivalence.types, 
  eqv.level=1, 
  upper=NA, 
  ccontinuity=continuity.correction.methods,
  conf.level = 0.95, 
  x.name="",
  y.name="",
  relevance=TRUE) {
  # Calculate alpha
  alpha <- (1 - conf.level)
  # default eqv.types is "delta" if unspecified
  if (length(eqv.type)>1) {
    eqv.type <- "delta"
    }
  # make lower case eqv.type and ccontinuity options
  eqv.type <- tolower(eqv.type)
  ccontinuity <- tolower(ccontinuity)
  if (is.na(upper)) {
    upper <- abs(eqv.level)
    }
  # Valiate x has exactly two values
  if (is.factor(x)) {
    if (length(levels(sort(x[complete.cases(x)])))!=2) {
      rlang::abort(message="x must contain exactly two values!")
      }
    x.label <- pad.left(toString(levels(sort(x[complete.cases(x)]))[2]),12)
    }
   else {
    if (length(unique(sort(x[complete.cases(x)])))!=2) {
      rlang::abort(message="x must contain exactly two values!")
      }

    if (!is.na(p0) & x.name == "") {
      x.name <- pad.left(paste0(gsub(".*\\$","",deparse(substitute(x)))),12)
      x.long.name <- pad.left(paste0(gsub(".*\\$","",deparse(substitute(x)))),48)
      }
    x.label <- pad.left(paste0(trimws(x.name),"=",toString(unique(sort(x[complete.cases(x)]))[2])),12)
    }
  # Valiate y has exactly two values if not null
  if (!is.null(y)) {
    if (is.factor(y)) {
      if (length(levels(sort(y[complete.cases(y)])))!=2) {
        rlang::abort(message="y must contain exactly two values!")
        }
      y.label <- pad.left(toString(levels(sort(y[complete.cases(y)]))[2]),12)
      }
     else {
      if (length(unique(sort(y[complete.cases(y)])))!=2) {
        rlang::abort(message="y must contain exactly two values!")
        }
      y.label <- pad.left(paste0(trimws(y.name),"=",toString(unique(sort(y[complete.cases(y)]))[2])),12)
      }
    }
  if (is.null(by)) {
    if (is.null(by.names)) {
      by.names <- levels(as.factor(sort(y[])))
      }
    if (x.name == "") {
      x.name <- pad.left(paste0(gsub(".*\\$","",deparse(substitute(x)))),12)
      x.long.name <- pad.left(paste0(gsub(".*\\$","",deparse(substitute(x)))),48)
      }
    if (y.name == "") {
      y.name <- pad.left(paste0(gsub(".*\\$","",deparse(substitute(y)))),12)
      y.long.name <- pad.left(paste0(gsub(".*\\$","",deparse(substitute(y)))),48)
      }
    }
  x.outcome.name <- gsub(".*\\$","",deparse(substitute(x)))
  if (!is.null(y)) {
    y.outcome.name <- gsub(".*\\$","",deparse(substitute(y)))
    }
   else {
    y.outcome.name <- x.outcome.name
    }
  # Validate non-null by option, and use to assign x and y variables if valid
  if (!is.null(by)) {
    group.name <- pad.left(gsub(".*\\$","",deparse(substitute(by))),12)
    if (is.null(by.names)) {
      # Check if the by variable is labeled, and if so assign those labels
     if (!is.null(attr(by,"labels"))) {
       by.names <- names(attributes(sort(by))$labels)
       }
      # Otherwise use the values of the by variable
      else {
       by.names <- levels(as.factor(sort(by)))
       }
      }
    if (length(unique(by)) != 2) {
      rlang::abort(message="by must contain exactly two different values to indicate group")
      }
    if (length(by) != length(x)) {
      rlang::abort(message="x and by must have equal lengths")
      }
     else {
      y <- x[by==levels(as.factor(by))[2]]
      x <- x[by==levels(as.factor(by))[1]]
      }
    if (x.name == "") {
     x.name <- pad.left(toString(paste0(trimws(group.name),"=",toString(by.names[1]))),12)
     x.long.name <- pad.left(toString(paste0(trimws(group.name),"=",toString(by.names[1]))),48)
      }
    if (y.name == "") {
      y.name <- pad.left(toString(paste0(trimws(group.name),"=",toString(by.names[2]))),12)
      y.long.name <- pad.left(toString(paste0(trimws(group.name),"=",toString(by.names[2]))),48)
      }
    }
  # validate continuity correction selection
  ccontinuity <- tolower(ccontinuity)
  if (length(ccontinuity) > 1) {
    ccontinuity <- "none"
    }
  if (!(ccontinuity %in% c("none", "yates", "ha"))) {
    rlang::warn(message=paste0("Ignoring ccontinuity = \"", ccontinuity, "\",\n  ccontinuity must be one of c(\"none\", \"yates\", \"ha\")\n", sep=""))
    }
  # Continuity correction notice function
  cont.cor.notice <- function(ccontinuity) {
    if (ccontinuity == "yates") {
      rlang::inform(message="\nUsing the Yates continuity correction")
      } 
    if (ccontinuity == "ha" & is.na(p0)) {
      rlang::inform(message="\nUsing the Hauck-Anderson continuity correction")
      } 
    if (ccontinuity == "ha" & (!is.na(p0) | (is.na(p0) & is.null(y) & is.null(by)))) {
      rlang::warn(message=paste0("Ignoring ccontinuity argument,\n  Hauck-Anderson continuity correction not relevant in one-sample tests"))
      } 
    }
  # Demand one-sample or two-sample test
  if (is.null(y) & is.null(by) & is.na(p0)) {
    rlang::abort(message="A one-sample test requires a p0 argument\nA two-sample test requires either a y or a by argument\n")
    }
  # Set up formatting for confidence interval column heading
  cl.width <- nchar(toString(100*conf.level))
  cl.width.pad <- paste0(rep(" ",times=(6-cl.width)),collapse="")
  conf.level.display <- paste0(cl.width.pad,"[",toString(100*conf.level),"% Conf. Interval]",sep="")
  if (cl.width > 5) {
    conf.level.display <- paste0("     [",toString(100*conf.level),"% CI]")
    }
  # Set display names for variables
  by.levels <- levels(as.factor(by))
  if (is.null(by.names)) {
    by.names <- levels(as.factor(by))
    }
  if (!is.null(by.names)) {
    group.1.name <- pad.left(paste0(" ",by.names[1]), 12)
    group.2.name <- pad.left(paste0(" ",by.names[2]), 12)
    }
  if (x.name=="") {
    x.name <- pad.left(gsub(".*\\$","",deparse(substitute(x))),12)
    }
   else {
    x.name<- pad.left(x.name,12)
    }
  if (!is.null(y) & is.na(p0)) {
    # Will will display y this way in the output
    if (y.name=="") {
      y.name <- pad.left(gsub(".*\\$","",deparse(substitute(y))),12)
      }
     else {
      y.name <- pad.left(y.name,12)
      }
    }
  # Create top bar, mid bar, and bottom bar
  top.bar    <- paste0(paste0(rep("\U2500",13),collapse=""),"\U252C",paste0(rep("\U2500",66),collapse=""),collapse="")
  mid.bar    <- paste0(paste0(rep("\U2500",13),collapse=""),"\U253C",paste0(rep("\U2500",66),collapse=""),collapse="")
  bottom.bar <- paste0(paste0(rep("\U2500",13),collapse=""),"\U2534",paste0(rep("\U2500",66),collapse=""),collapse="")
  # One-sample test
  if (!is.na(p0)) {
    # Ignore settings for unpaired test
    if (!is.null(y)) {
      rlang::inform(message="Ignoring y argument, not relevant in one-sample tests")
      }
    if (!is.null(by)) {
      rlang::inform(message="Ignoring y argument, not relevant in one-sample tests")
      }
    if (ccontinuity == "ha") {
      cont.cor.notice(ccontinuity)
      ccontinuity <- "none"
      }
    # Get n, accounting for missing values of x
    n <- length(x[complete.cases(x)])
    # Format n for very large sample sizes
    n.display <- pad.left(toString(n),14)
    # Calculate the various sample statistics for this test
    if (is.factor(x)) {
      success <- levels(sort(x[complete.cases(x)]))[2]
      count.x <- length(sort(x[complete.cases(x)])[x==success])
      }
     else {
      count.x <- sum(x[complete.cases(x)])
      }
    p.hat <- count.x/n
    p.hat.display <- pad.left(toString(signif(p.hat,8)),11)
    estimate <- p.hat
    theta <- p.hat - p0
    theta.display <- pad.left(toString(signif(theta,8)),11)
    sd.p.hat <- sqrt(p.hat*(1-p.hat))
    sd.p.hat.display <- pad.left(toString(signif(sd.p.hat,7)),11)
    sd.p <- sqrt(p0*(1-p0))
    sd.p.display <- pad.left(toString(signif(sd.p,7)),11)
    se.p.hat <- sd.p.hat/sqrt(n)
    se.p.hat.display <- pad.left(toString(signif(se.p.hat,7)),11)
    se.p <- sd.p/sqrt(n)
    se.p.display <- pad.left(toString(signif(se.p,7)),11)
    # Calculate the confidence interval for p, and format for output
    z.crit <- qnorm(p=alpha/2, lower.tail=FALSE)
    ci.p.lb <- p.hat - z.crit*se.p.hat
    ci.p.ub <- p.hat + z.crit*se.p.hat
    ci.p.lb.display <- pad.left(toString(signif(ci.p.lb,7)),11)
    ci.p.ub.display <- pad.left(toString(signif(ci.p.ub,7)),11)
    ci.upper.diff.lb <- upper - theta - z.crit*se.p
    ci.upper.diff.ub <- upper - theta + z.crit*se.p
    ci.upper.diff.lb.display <- pad.left(toString(signif(ci.upper.diff.lb,7)),11)
    ci.upper.diff.ub.display <- pad.left(toString(signif(ci.upper.diff.ub,7)),11)
    ci.lower.diff.lb <- theta + abs(eqv.level) - z.crit*se.p
    ci.lower.diff.ub <- theta + abs(eqv.level) + z.crit*se.p
    ci.lower.diff.lb.display <- pad.left(toString(signif(ci.lower.diff.lb,7)),11)
    ci.lower.diff.ub.display <- pad.left(toString(signif(ci.lower.diff.ub,7)),11)
    # No continuity correction
    if (ccontinuity != "yates") {
      cont.cor <- 0
      }
    # Yates continuity correction for one-sample test
    if (ccontinuity == "yates") {
      cont.cor <- 0.5*(1/n)
      }
    # Calculate and format positivist test statistics
    if (theta >= 0) {        # Sign of theta affects direction of correction
      z.pos <- (theta - cont.cor) / se.p
      }
     else {
      z.pos <- (theta + cont.cor) / se.p
      }
    z.pos.display <- sprintf("%-8.4f",z.pos)
    se.crit <- se.p
    p.pos <- 2*pnorm(q=abs(z.pos), lower.tail=FALSE)
    p.pos.display <- format.extreme.p.vals(p.pos)
    lower <- -1*abs(eqv.level)
    if (eqv.type == "delta") {
      z1 <- (upper - (theta + cont.cor))/se.p
      z2 <- ((theta - cont.cor) + abs(eqv.level))/se.p
      }
    if (eqv.type == "epsilon") {
      z1 <- upper - (((theta + cont.cor)) / se.p)
      z2 <- (((theta - cont.cor)) / se.p) + abs(eqv.level)
      }
    z1.display <- sprintf("%-8.4g",z1)
    z2.display <- sprintf("%-8.4g",z2)
    p1 <- pnorm(q=abs(z1), lower.tail=FALSE)
    p1.display <- format.extreme.p.vals(p1)
    p2 <- pnorm(q=abs(z2), lower.tail=FALSE)
    p2.display <- format.extreme.p.vals(p2)
    if (z1 <= -10) {
      z.pad <- pad.spaces(14)
      }
    if (z1 < 0) {
      z.pad <- pad.spaces(15)
      }
    if (z1 >= 0) {
      z.pad <- pad.spaces(16)
      }
    display.width <- 80
    no.obs.and.no <- 32
    if (relevance) {
      pos.title <- "One-sample z test for proportion difference\n"
      n.obs.pad <- display.width - no.obs.and.no - nchar(trimws(x.long.name))
      z.equals.and.z <- 11
      z.pos.pad <- pad.spaces(display.width - nchar(paste0(pad.spaces(10),"p^ = prop(",trimws(x.label),") \U2212 ",substring(as.character(p0), 2)," = ",trimws(theta.display))) - z.equals.and.z)
      rlang::inform(message="\nOne-sample relevance z test of sample proportions")
      rlang::inform(message=paste0("\n",pos.title,pad.spaces(n.obs.pad),trimws(x.long.name),": Number of obs = ",n.display,"",sep=""))
      rlang::inform(message=top.bar)
      rlang::inform(message=paste0(pad.spaces(4),"Variable \U2502",pad.spaces(7),"Mean",pad.spaces(3),"Std. err.",pad.spaces(19),conf.level.display,sep=""))
      rlang::inform(message=mid.bar)
      rlang::inform(message=paste0(x.name," \U2502",p.hat.display,se.p.hat.display,pad.spaces(22),ci.p.lb.display,ci.p.ub.display,sep=""))
      rlang::inform(message=bottom.bar)
      rlang::inform(message=paste0(pad.spaces(10),"\U03B8^ = prop(",trimws(x.label),") \U2212 ",substring(as.character(p0), 2)," = ",trimws(theta.display),z.pos.pad,"z = ",z.pos.display,sep=""))
      rlang::inform(message=paste0(pad.spaces(7),"Ho: \U03B8 = 0",sep=""))
      cont.cor.notice(ccontinuity)
      rlang::inform(message=paste0("\n",pad.spaces(35),"Ha: \U03B8 \U2260 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(29),"Pr(|Z| > |z|) ",p.pos.display,"\n",sep=""))
      }
    neg.title <- "One-sample z test for proportion equivalence\n"
    n.obs.pad <- display.width - no.obs.and.no - nchar(trimws(x.long.name))
    rlang::inform(message=paste0("\n",neg.title,pad.spaces(n.obs.pad),trimws(x.long.name),": Number of obs = ",n.display,sep=""))
    rlang::inform(message=top.bar)
    rlang::inform(message=paste0(pad.spaces(4),"Variable \U2502",pad.spaces(7),"Mean",pad.spaces(3),"Std. err.",pad.spaces(19),conf.level.display,sep=""))
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(x.name," \U2502",p.hat.display,se.p.hat.display,pad.spaces(22),ci.p.lb.display,ci.p.ub.display,sep=""))
    if (eqv.type == "delta") {
      rlang::inform(message=mid.bar)
      if (upper == abs(eqv.level)) {
        upper.diff <- abs(eqv.level) - theta
        upper.diff.display <- pad.left(signif(upper.diff,7),11)
        upper.diff.name <- paste0(pad.spaces(8),"\U0394-\U03B8\U02C6")
        lower.diff <- theta + abs(eqv.level)
        lower.diff.display <- pad.left(signif(lower.diff,7),11)
        lower.diff.name <- paste0(pad.spaces(8),"\U03B8\U02C6+\U0394")
        delta.display <- trimws(trim0(sprintf("%.4f",signif(abs(eqv.level),5))))
        rlang::inform(message=paste0(upper.diff.name," \U2502",upper.diff.display,se.p.display,pad.spaces(22),ci.upper.diff.lb.display,ci.upper.diff.ub.display,sep=""))
        rlang::inform(message=paste0(lower.diff.name," \U2502",lower.diff.display,se.p.display,pad.spaces(22),ci.lower.diff.lb.display,ci.lower.diff.ub.display,sep=""))
        rlang::inform(message=bottom.bar)
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.label),") \U2212 ",substring(as.character(p0), 2)," = ",trimws(theta.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(11),"\U0394 = ",delta.display, pad.spaces(9 - nchar(delta.display)),"\U0394 expressed in same units as prop(",trimws(x.name),")",sep=""))
        criticalvalue <- se.p*qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (eqv.level <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U0394 \U2264 z-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.pr).",sep=""))
          }
        rlang::inform(message="\nHo: |\U03B8| \U2265 \U0394:")
        }
      if (upper != abs(eqv.level)) {
        upper.diff <- upper - theta
        upper.diff.display <- pad.left(round(upper.diff,7),11)
        upper.diff.name <- paste0(pad.spaces(7),"\U0394u-\U03B8\U02C6")
        lower.diff <- theta + abs(eqv.level)
        lower.diff.display <- pad.left(round(lower.diff,7),11)
        lower.diff.name <- paste0(pad.spaces(7),"\U03B8\U02C6-\U0394l")
        delta.lower.display <- trimws(trim0(sprintf("%-7.3g",(signif(-1*abs(eqv.level),5)))))
        delta.upper.display <- trimws(trim0(sprintf("%-6.3g",signif(abs(upper),5))))
        rlang::inform(message=paste0(upper.diff.name," \U2502",upper.diff.display,se.p.display,pad.spaces(22),ci.upper.diff.lb.display,ci.upper.diff.ub.display,sep=""))
        rlang::inform(message=paste0(lower.diff.name," \U2502",lower.diff.display,se.p.display,pad.spaces(22),ci.lower.diff.lb.display,ci.lower.diff.ub.display,sep=""))
        rlang::inform(message=bottom.bar)
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.label),") \U2212 ",substring(as.character(p0), 2)," = ",trimws(theta.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(10),"\U0394l = ",delta.lower.display,pad.spaces(9 - nchar(delta.lower.display)),"\U0394l expressed in same units as prop(",trimws(x.label),")",sep=""))
        rlang::inform(message=paste0(pad.spaces(10),"\U0394u =  ",delta.upper.display,pad.spaces(8 - nchar(delta.upper.display)),"\U0394u expressed in same units as prop(",trimws(x.label),")",sep=""))
        criticalvalue <- se.p*qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (abs(eqv.level) <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if |\U0394l| \U2264 z-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.pr).",sep=""))
          }
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (upper <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U0394u \U2264 z-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.pr).",sep=""))
          }
        rlang::inform(message="\nHo: \U03B8 \U2264 \U0394l, or \U03B8 \U2265 \U0394u:")
        }
      }
    if (eqv.type=="epsilon") {
      if (upper == abs(eqv.level)) {
        epsilon.display <- trimws(trim0(sprintf("%5.3f",signif(abs(eqv.level),5))))
        rlang::inform(message=bottom.bar)
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.label),") \U2212 ",substring(as.character(p0), 2)," = ",trimws(theta.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(11),"\U03B5 = ",epsilon.display,pad.spaces(9 - nchar(epsilon.display)),"\U03B5 expressed in units of the z distribution",sep=""))
        criticalvalue <- qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (eqv.level <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U03B5 \U2264 z-crit (", criticalvalue.display, "). See help(tost.pr).",sep=""))
          }
        rlang::inform(message="\nHo: |Z| \U2265 \U03B5:")
        }
      if (upper != abs(eqv.level)) {
        epsilon.upper.display <- trimws(trim0(sprintf("%5.3f",signif(upper,5))))
        epsilon.lower.display <- trimws(trim0(sprintf("%5.3f",signif(-1*abs(eqv.level),5))))
        rlang::inform(message=bottom.bar)
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.label),") \U2212 ",substring(as.character(p0), 2)," = ",trimws(theta.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(10),"\U03B5l = ",epsilon.lower.display,pad.spaces(9 - nchar(epsilon.lower.display)),"\U03B5l expressed in units of the z distribution",sep=""))
        rlang::inform(message=paste0(pad.spaces(10),"\U03B5u =  ",epsilon.upper.display,pad.spaces(8 - nchar(epsilon.upper.display)),"\U03B5u expressed in units of the z distribution",sep=""))
        criticalvalue <- qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (abs(eqv.level) <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if |\U03B5l| \U2264 z-crit (", criticalvalue.display, "). See help(tost.pr).",sep=""))
          }
        if (upper <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U03B5u \U2264 z-crit (", criticalvalue.display, "). See help(tost.pr).",sep=""))
          }
        rlang::inform(message="\nHo: Z \U2264 \U03B5l, or Z \U2265 \U03B5u:")
        }
      }
    cont.cor.notice(ccontinuity)
    rlang::inform(message="")
    rlang::inform(message=paste0(pad.spaces(8), "z1 = ",z1.display,z.pad,"z2 = ",z2.display, "\n", sep=""))
    if (upper != abs(eqv.level)) {
      if (eqv.type=="delta") {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394u-\U03B8 \U2264 0",pad.spaces(16),"Ho2: \U03B8-\U0394l \U2264 0"))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394u-\U03B8 > 0",pad.spaces(16),"Ha2: \U03B8-\U0394l > 0"))
        }
       else {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5u\U2013Z \U2264 0",pad.spaces(16),"Ho2: Z-\U03B5l \U2264 0", sep=""))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5u\U2013Z > 0",pad.spaces(16),"Ha2: Z-\U03B5l > 0", sep=""))
        }
      }
     else {
      if (eqv.type=="delta") {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394-\U03B8 \U2264 0",pad.spaces(17),"Ho2: \U03B8+\U0394 \U2264 0", sep=""))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394-\U03B8 > 0",pad.spaces(17),"Ha2: \U03B8+\U0394 > 0", sep=""))
        }
       else {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5\U2013Z \U2264 0",pad.spaces(17),"Ho2: Z+\U03B5 \U2264 0", sep=""))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5\U2013Z > 0",pad.spaces(17),"Ha2: Z+\U03B5 > 0", sep=""))
        }
      }
    } 
  # Two-sample test
  if (is.na(p0)) {
    # Get n1 and n2, accounting for missing values of x
    nx <- length(x[complete.cases(x)])
    ny <- length(y[complete.cases(y)])
    # Format n1 & n2
    nx.display <- pad.left(toString(nx),14)
    ny.display <- pad.left(toString(ny),14)
    # Calculate the various sample statistics for this test
    if (is.factor(x)) {
      success <- levels(sort(x[complete.cases(x)]))[2]
      count.x <- length(sort(x[complete.cases(x)])[x==success])
      }
     else {
      count.x <- sum(x[complete.cases(x)])
      }
    if (is.factor(y)) {
      success <- levels(sort(y[complete.cases(y)]))[2]
      count.y <- length(sort(y[complete.cases(y)])[y==success])
      }
     else {
      count.y <- sum(y[complete.cases(y)])
      }
    px.hat <- count.x/nx
    px.hat.display <- pad.left(toString(round(px.hat,7)),11)
    py.hat <- count.y/ny
    py.hat.display <- pad.left(toString(round(py.hat,7)),11)
    p.hat <- (count.x + count.y) / (nx + ny)
    p.hat.display <- pad.left(toString(round(p.hat,8)),11)
    estimate1 <- px.hat
    estimate2 <- py.hat
    theta.hat <- px.hat - py.hat
    theta.hat.display <- pad.left(toString(round(theta.hat,7)),11)
    se.px.hat <- sqrt((px.hat * (1 - px.hat)) / nx)
    se.px.hat.display <- pad.left(toString(round(se.px.hat,7)),11)
    se.py.hat <- sqrt((py.hat * (1 - py.hat)) / ny)
    se.py.hat.display <- pad.left(toString(round(se.py.hat,7)),11)
    se.p.hat <- sqrt(p.hat * (1 - p.hat) * ((1/nx) + (1/ny)))
    se.p.hat.display <- pad.left(toString(round(se.p.hat,7)),11)
    se.theta.hat <- sqrt(se.px.hat^2 + se.py.hat^2)
    se.theta.hat.display <- pad.left(toString(round(se.theta.hat,7)),11)
    # Calculate the confidence intervals for x, y, and theta, and format for output
    z.crit <- qnorm(p=alpha/2, lower.tail=FALSE)
    # No continuity correction
    if (ccontinuity == "none") {
      cont.cor <- 0
      }
    # Yates continuity correction
    if (ccontinuity == "yates") {
      cont.cor <- sign(theta.hat)*0.5*(1.0/nx + 1.0/ny)
      }
    # Hauck-Anderson continuity correction
    if (ccontinuity == "ha") {
      cont.cor <- (1.0/(2.0 * min(nx,ny)))
      se.p.hat <- sqrt( ((px.hat * (1 - px.hat)) / (nx - 1)) + ((py.hat * (1 - py.hat)) / (ny - 1)) )
      se.p.hat.display <- pad.left(toString(round(se.p.hat,7)),11)
      ci.p.lb <- theta.hat - z.crit * se.p.hat
      ci.p.ub <- theta.hat + z.crit * se.p.hat
      ci.p.lb.display <- pad.left(toString(round(ci.p.lb,7)),11)
      ci.p.ub.display <- pad.left(toString(round(ci.p.ub,7)),11)
      }
    ci.p1.lb <- px.hat - z.crit*se.px.hat
    ci.p1.ub <- px.hat + z.crit*se.px.hat
    ci.p1.lb.display <- pad.left(toString(round(ci.p1.lb,7)),11)
    ci.p1.ub.display <- pad.left(toString(round(ci.p1.ub,7)),11)
    ci.p2.lb <- py.hat - z.crit*se.py.hat
    ci.p2.ub <- py.hat + z.crit*se.py.hat
    ci.p2.lb.display <- pad.left(toString(round(ci.p2.lb,7)),11)
    ci.p2.ub.display <- pad.left(toString(round(ci.p2.ub,7)),11)
    ci.p.lb <- theta.hat - z.crit * se.p.hat
    ci.p.ub <- theta.hat + z.crit * se.p.hat
    ci.p.lb.display <- pad.left(toString(round(ci.p.lb,7)),11)
    ci.p.ub.display <- pad.left(toString(round(ci.p.ub,7)),11)
    ci.theta.lb <- theta.hat - z.crit * sqrt(se.px.hat^2 + se.py.hat^2)
    ci.theta.ub <- theta.hat + z.crit * sqrt(se.px.hat^2 + se.py.hat^2)
    ci.theta.lb.display <- pad.left(toString(round(ci.theta.lb,7)),11)
    ci.theta.ub.display <- pad.left(toString(round(ci.theta.ub,7)),11)
    ci.upper.diff.lb <- upper - theta.hat - z.crit*se.p.hat
    ci.upper.diff.ub <- upper - theta.hat + z.crit*se.p.hat
    ci.upper.diff.lb.display <- pad.left(toString(round(ci.upper.diff.lb,7)),11)
    ci.upper.diff.ub.display <- pad.left(toString(round(ci.upper.diff.ub,7)),11)
    ci.lower.diff.lb <- theta.hat + abs(eqv.level) - z.crit*se.p.hat
    ci.lower.diff.ub <- theta.hat + abs(eqv.level) + z.crit*se.p.hat
    ci.lower.diff.lb.display <- pad.left(toString(round(ci.lower.diff.lb,7)),11)
    ci.lower.diff.ub.display <- pad.left(toString(round(ci.lower.diff.ub,7)),11)
    # Calculate and format positivist test statistics
    if (theta.hat >= 0) {        # Sign of theta affects direction of correction
      z.pos <- (theta.hat - cont.cor) / se.p.hat
      }
     else {
      z.pos <- (theta.hat + cont.cor) / se.p.hat
      }
    z.pos.display <- sprintf("%-8.4f",z.pos)
    z.pos.table   <- sprintf("%-8.3f",z.pos)
    se.crit <- se.p.hat
    p.pos <- 2*pnorm(q=abs(z.pos), lower.tail=FALSE)
    p.pos.display <- format.extreme.p.vals(p.pos)
    lower <- -1*abs(eqv.level)
    if (eqv.type == "delta") {
      z1 <- (upper - (theta.hat + cont.cor))/se.p.hat
      z2 <- ((theta.hat - cont.cor) + abs(eqv.level))/se.p.hat
      }
    if (eqv.type == "epsilon") {
      z1 <- upper - z.pos
      z2 <- z.pos + abs(eqv.level)
      }
    z1.display <- sprintf("%-8.4g",z1)
    z2.display <- sprintf("%-8.4g",z2)
    p1 <- pnorm(q=abs(z1), lower.tail=FALSE)
    p1.display <- format.extreme.p.vals(p1)
    p2 <- pnorm(q=abs(z2), lower.tail=FALSE)
    p2.display <- format.extreme.p.vals(p2)
    if (z1 <= -10) {
      z.pad <- pad.spaces(14)
      }
    if (z1 < 0) {
      z.pad <- pad.spaces(15)
      }
    if (z1 >= 0) {
      z.pad <- pad.spaces(16)
    }
    nx.obs.pad <- 48 - nchar(trimws(x.long.name))
    ny.obs.pad <- 48 - nchar(trimws(y.long.name))
    if (!is.null(by)) {
      z.pos.pad <- 80 - nchar(paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.outcome.name),"=1|",trimws(x.name),") - prop(",trimws(y.outcome.name),"=1|",trimws(y.name),")", "z =", z.pos.display, sep=""))
      }
    if (is.null(by)) {
      z.pos.pad <- 80 - nchar(paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.outcome.name),"=1) - prop(",trimws(y.outcome.name),"=1)", "z =", z.pos.display, sep=""))
      }
    if (z.pos.pad < 0 ) {
      z.pos.pad <- paste0("\n",pad.spaces(11))
      }
     else {
      z.pos.pad <- pad.spaces(z.pos.pad)
      }
    if (relevance) {
      pos.title <- "Two-sample z test for proportion difference\n"
      rlang::inform(message="\nTwo-sample relevance z test of sample proportions")
      rlang::inform(message=paste0("\n",pos.title,sep=""))
      rlang::inform(message=paste0(pad.spaces(nx.obs.pad),trimws(x.long.name),": Number of obs = ",nx.display,sep=""))
      rlang::inform(message=paste0(pad.spaces(ny.obs.pad),trimws(y.long.name),": Number of obs = ",ny.display,sep=""))
      rlang::inform(message=top.bar)
      rlang::inform(message=paste0(pad.spaces(4),"Variable \U2502",pad.spaces(7),"Mean",pad.spaces(3),"Std. err.",pad.spaces(4),"z",pad.spaces(3),"Pr(|Z|>|z|)",conf.level.display,sep=""))
      rlang::inform(message=mid.bar)
      rlang::inform(message=paste0(x.name," \U2502",px.hat.display,se.px.hat.display,pad.spaces(22),ci.p1.lb.display,ci.p1.ub.display,sep=""))
      rlang::inform(message=paste0(y.name," \U2502",py.hat.display,se.py.hat.display,pad.spaces(22),ci.p2.lb.display,ci.p2.ub.display,sep=""))
      rlang::inform(message=mid.bar)
      rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 \U2502",theta.hat.display,se.theta.hat.display,pad.spaces(22),ci.theta.lb.display,ci.theta.ub.display,sep=""))
      rlang::inform(message=paste0(pad.spaces(13),"\U2502  under Ho:",se.p.hat.display,pad.spaces(3),z.pos.table,pad.spaces(0),trimws(sprintf("%8.4f",p.pos)),sep=""))
      rlang::inform(message=bottom.bar)
      if (!is.null(by)) {
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.outcome.name),"=1|",trimws(x.name),") - prop(",trimws(y.outcome.name),"=1|",trimws(y.name),")",z.pos.pad,"z = ",z.pos.display,sep=""))
        }
      if (is.null(by)) {
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.outcome.name),"=1) - prop(",trimws(y.outcome.name),"=1)",z.pos.pad,"z = ",z.pos.display,sep=""))
        }
      rlang::inform(message=paste0(pad.spaces(12)," = ",trimws(theta.hat.display), sep=""))
      rlang::inform(message=paste0(pad.spaces(7), "Ho: \U03B8 = 0",sep=""))
      cont.cor.notice(ccontinuity)
      rlang::inform(message=paste0("\n",pad.spaces(35),"Ha: \U03B8 \U2260 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(29),"Pr(|Z| > |z|) ",p.pos.display,"\n",sep=""))
      }
    neg.title <- "Two-sample z test for proportion equivalence"
    rlang::inform(message=paste0("\n",neg.title,sep=""))
    rlang::inform(message=paste0(pad.spaces(nx.obs.pad),trimws(x.long.name),": Number of obs = ",nx.display,sep=""))
    rlang::inform(message=paste0(pad.spaces(ny.obs.pad),trimws(y.long.name),": Number of obs = ",ny.display,sep=""))
    rlang::inform(message=top.bar)
    rlang::inform(message=paste0(pad.spaces(4),"Variable \U2502",pad.spaces(7),"Mean",pad.spaces(3),"Std. err.",pad.spaces(19),conf.level.display,sep=""))
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(x.name," \U2502",px.hat.display,se.px.hat.display,pad.spaces(22),ci.p1.lb.display,ci.p1.ub.display,sep=""))
    rlang::inform(message=paste0(y.name," \U2502",py.hat.display,se.py.hat.display,pad.spaces(22),ci.p2.lb.display,ci.p2.ub.display,sep=""))
    if (eqv.type == "delta") {
      rlang::inform(message=mid.bar)
      if (upper == abs(eqv.level)) {
        upper.diff <- abs(eqv.level) - theta.hat
        upper.diff.display <- pad.left(round(upper.diff,7),11)
        upper.diff.name <- paste0(pad.spaces(8),"\U0394-\U03B8\U02C6")
        lower.diff <- theta.hat + abs(eqv.level)
        lower.diff.display <- pad.left(round(lower.diff,7),11)
        lower.diff.name <- paste0(pad.spaces(8),"\U03B8\U02C6+\U0394")
        delta.display <- trimws(trim0(sprintf("%.4f",round(abs(eqv.level),5))))
        rlang::inform(message=paste0(upper.diff.name," \U2502",upper.diff.display,se.p.hat.display,pad.spaces(22),ci.upper.diff.lb.display,ci.upper.diff.ub.display,sep=""))
        rlang::inform(message=paste0(lower.diff.name," \U2502",lower.diff.display,se.p.hat.display,pad.spaces(22),ci.lower.diff.lb.display,ci.lower.diff.ub.display,sep=""))
        rlang::inform(message=bottom.bar)
        if (!is.null(by)) {
          rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.outcome.name),"=1|",trimws(x.name),") - prop(",trimws(y.outcome.name),"=1|",trimws(y.name),")",sep=""))
          }
        if (is.null(by)) {
          rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.outcome.name),"=1) - prop(",trimws(y.outcome.name),"=1)",sep=""))
          }
        rlang::inform(message=paste0(pad.spaces(12)," = ",trimws(theta.hat.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(11),"\U0394 = ",delta.display,pad.spaces(9 - nchar(delta.display)),"\U0394 expressed in same units as prop(",trimws(x.outcome.name),"=1)",sep=""))
        criticalvalue <- se.p.hat*qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (eqv.level <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U0394 \U2264 z-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.pr).",sep=""))
          }
        rlang::inform(message="\nHo: |\U03B8| \U2265 \U0394:")
        }
      if (upper != abs(eqv.level)) {
        upper.diff <- upper - theta.hat
        upper.diff.display <- pad.left(round(upper.diff,7),11)
        upper.diff.name <- paste0(pad.spaces(7),"\U0394u-\U03B8\U02C6")
        lower.diff <- theta.hat + abs(eqv.level)
        lower.diff.display <- pad.left(round(lower.diff,7),11)
        lower.diff.name <- paste0(pad.spaces(7),"\U03B8\U02C6-\U0394l")
        delta.lower.display <- trimws(trim0(sprintf("%-.4f",(round(-1*abs(eqv.level),5)))))
        delta.upper.display <- trimws(trim0(sprintf("%-.4f",round(abs(upper),5))))
        rlang::inform(message=paste0(upper.diff.name," \U2502",upper.diff.display,se.p.hat.display,pad.spaces(22),ci.upper.diff.lb.display,ci.upper.diff.ub.display,sep=""))
        rlang::inform(message=paste0(lower.diff.name," \U2502",lower.diff.display,se.p.hat.display,pad.spaces(22),ci.lower.diff.lb.display,ci.lower.diff.ub.display,sep=""))
        rlang::inform(message=bottom.bar)
        rlang::inform(message=paste0(pad.spaces(8),"\U03B8\U02C6 = prop(",trimws(x.outcome.name),"=1) - prop(",trimws(y.outcome.name),"=1) = ",trimws(theta.hat.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(8),"\U0394l = ",delta.lower.display,pad.spaces(11 - nchar(delta.lower.display)),"\U0394l expressed in same units as prop(",trimws(x.outcome.name),"=1)",sep=""))
        rlang::inform(message=paste0(pad.spaces(8),"\U0394u =  ",delta.upper.display,pad.spaces(10 - nchar(delta.upper.display)),"\U0394u expressed in same units as prop(",trimws(x.outcome.name),"=1)",sep=""))
        criticalvalue <- se.p.hat*qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (abs(eqv.level) <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if |\U0394l| \U2264 z-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.pr).",sep=""))
          }
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (upper <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U0394u \U2264 z-crit\U00D7S.E. (", criticalvalue.display, "). See help(tost.pr).",sep=""))
          }
        rlang::inform(message="\nHo: \U03B8 \U2264 \U0394l, or \U03B8 \U2265 \U0394u:")
        }
      }
    if (eqv.type=="epsilon") {
      rlang::inform(message=mid.bar)
      rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 \U2502",theta.hat.display,se.p.hat.display,pad.spaces(22),ci.p.lb.display,ci.p.ub.display,sep=""))
      if (upper == abs(eqv.level)) {
        epsilon.display <- trimws(trim0(sprintf("%5.3f",signif(abs(eqv.level),5))))
        rlang::inform(message=bottom.bar)
        if (!is.null(by)) {
          rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.outcome.name),"=1|",trimws(x.name),") - prop(",trimws(y.outcome.name),"=1|",trimws(y.name),")",sep=""))
          }
        if (is.null(by)) {
          rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.outcome.name),"=1) - prop(",trimws(y.outcome.name),"=1)",sep=""))
          }
        rlang::inform(message=paste0(pad.spaces(12)," = ",trimws(theta.hat.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(11),"\U03B5 = ",epsilon.display,pad.spaces(9 - nchar(epsilon.display)),"\U03B5 expressed in units of the z distribution",sep=""))
        criticalvalue <- qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (eqv.level <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U03B5 \U2264 z-crit (", criticalvalue.display, "). See help(tost.pr).",sep=""))
          }
        rlang::inform(message="\nHo: |Z| \U2265 \U03B5:")
        }
      if (upper != abs(eqv.level)) {
        epsilon.upper.display <- trimws(trim0(sprintf("%5.3f",signif(upper,5))))
        epsilon.lower.display <- trimws(trim0(sprintf("%5.3f",signif(-1*abs(eqv.level),5))))
        rlang::inform(message=bottom.bar)
        rlang::inform(message=paste0(pad.spaces(10),"\U03B8\U02C6 = prop(",trimws(x.outcome.name),"=1) - prop(",trimws(y.outcome.name),"=1)",sep=""))
        rlang::inform(message=paste0(pad.spaces(12)," = ",trimws(theta.hat.display),sep=""))
        rlang::inform(message=paste0(pad.spaces(10),"\U03B5l = ",epsilon.lower.display,pad.spaces(9 - nchar(epsilon.lower.display)),"\U03B5l expressed in units of the z distribution",sep=""))
        rlang::inform(message=paste0(pad.spaces(10),"\U03B5u =  ",epsilon.upper.display,pad.spaces(8 - nchar(epsilon.upper.display)),"\U03B5u expressed in units of the z distribution",sep=""))
        criticalvalue <- qnorm(p=alpha, lower.tail=FALSE)
        criticalvalue.display <- trimws(sprintf("%-6.4g",signif(criticalvalue,6)))
        if (abs(eqv.level) <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if |\U03B5l| \U2264 z-crit (", criticalvalue.display, "). See help(tost.pr).",sep=""))
          }
        if (upper <= criticalvalue) {
          rlang::inform(message=paste0("\n Impossible to reject any Ho if \U03B5u \U2264 z-crit (", criticalvalue.display, "). See help(tost.pr).",sep=""))
          }
        rlang::inform(message="\nHo: Z \U2264 \U03B5l, or Z \U2265 \U03B5u:")
        }
      }
    cont.cor.notice(ccontinuity)
    rlang::inform(message="")
    rlang::inform(message=paste0(pad.spaces(8), "z1 = ", z1.display, z.pad, "z2 = ", z2.display, "\n", sep=""))
    if (upper != abs(eqv.level)) {
      if (eqv.type=="delta") {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394u-\U03B8 \U2264 0",pad.spaces(16),"Ho2: \U03B8-\U0394l \U2264 0", sep=""))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394u-\U03B8 > 0",pad.spaces(16),"Ha2: \U03B8-\U0394l > 0", sep=""))
        }
       else {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5u\U2013Z \U2264 0",pad.spaces(16),"Ho2: T-\U03B5l \U2264 0", sep=""))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5u\U2013Z > 0",pad.spaces(16),"Ha2: T-\U03B5l > 0", sep=""))
        }
      }
     else {
      if (eqv.type=="delta") {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394-\U03B8 \U2264 0",pad.spaces(17),"Ho2: \U03B8+\U0394 \U2264 0", sep=""))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394-\U03B8 > 0",pad.spaces(17),"Ha2: \U03B8+\U0394 > 0", sep=""))
        }
       else {
        rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5\U2013Z \U2264 0",pad.spaces(17),"Ho2: Z+\U03B5 \U2264 0", sep=""))
        rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5\U2013Z > 0",pad.spaces(17),"Ha2: Z+\U03B5 > 0", sep=""))
        }
      }
    }
  rlang::inform(message=paste0(pad.spaces(3),"Pr(Z > z1)",p1.display,pad.spaces(11),"Pr(Z > z2)",p2.display, sep=""))
  if (relevance) {
    if (upper != abs(eqv.level)) {
      if (tolower(eqv.type) == "delta") {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ",round(alpha,cl.width),", \U0394l = ", trim0(delta.lower.display),", and \U0394u = ", trim0(delta.upper.display),":", sep=""))
        }
       else {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ",round(alpha,cl.width),", \U03B5l = ", trim0(epsilon.lower.display),", and \U03B5u = ", trim0(epsilon.upper.display),":", sep=""))
        }
      }
     else {
      if (tolower(eqv.type) == "delta") {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ",round(alpha,cl.width)," and \U0394 = ", trim0(delta.display),":", sep=""))
        }
       else {
        rlang::inform(message=paste0("\n\nRelevance test conclusion for \U03B1 = ",round(alpha,cl.width)," and \U03B5 = ",trim0(epsilon.display),":", sep=""))
        }
      }
    if (p.pos <= alpha) {
      pos.decision <- "Reject"
      }
     else {
      pos.decision <- "Fail to reject"
      }
    if (p1 <= alpha & p2 <= alpha) {
      neg.decision <- "Reject"
      }
     else {
      neg.decision <- "Fail to reject"
      }
    rel.conclusion <- "Indeterminate (underpowered test)"
    if (pos.decision == "Reject" & neg.decision == "Fail to reject") {
      rel.conclusion <- "Relevant difference"
      }
     else {
      if (pos.decision == "Fail to reject" & neg.decision == "Reject") {
        rel.conclusion = "Equivalence"
        }
       else {
        if (pos.decision == "Reject" & neg.decision == "Reject") {
          rel.conclusion <- "Trivial difference (overpowered test)"
          }
        }
      }
    rlang::inform(message=paste0("  Ho test for difference:  ",pos.decision))
    rlang::inform(message=paste0("  Ho test for equivalence: ",neg.decision,"\n"))
    rlang::inform(message=paste0("Conclusion from combined tests: ",rel.conclusion))
    }

  out <- list() 
  # Prepare return stuff for one-sample test
  if (!is.na(p0)) {
    if (!relevance) {
      out$statistics <- c(z1,z2)
      names(out$statistics) <- c("z1","z2")
      out$p.values <- c(p1,p2)
      names(out$p.values) <- c("p1","p2")
      }
    else {
      out$statistics <- c(z1,z2,z.pos)
      names(out$statistics) <- c("z1","z2","z")
      out$p.values <- c(p1,p2,p.pos)
      names(out$p.values) <- c("p1","p2","p")
     }
    out$proportion <- p.hat
    names(out$proportion) <- paste0("prop(",trimws(x.name),"=1)",collapse="")
    out$sample_size <- n
    names(out$sample_size) <- "n"
    if (eqv.type=="delta") {
      if (upper==abs(eqv.level)) {
        out$threshold <- lower
        names(out$threshold) <- "\U394"
        }
       else {
         out$threshold <- c(upper, lower)
         names(out$threshold) <- c("\U0394u", "\U0394l")
        }
      }
     else {
      if (upper==abs(eqv.level)) {
        out$threshold <- lower
        names(out$threshold) <- "\U3B5"
        }
       else {
         out$threshold <- c(upper, lower)
         names(out$threshold) <- c("\U03B5u", "\U03B5l")
        }
      }
    if(relevance) {
      out$conclusion <- rel.conclusion
      names(out$conclusion) <- "relevance conclusion"
      }
    }

  # Prepare return stuff for two-sample test
  if (is.na(p0)) {
    if (!relevance) {
      out$statistics <- c(z1,z2)
      names(out$statistics) <- c("z1","z2")
      out$p.values <- c(p1,p2)
      names(out$p.values) <- c("p1","p2")
      }
     else {
      out$statistics <- c(z1,z2,z.pos)
      names(out$statistics) <- c("z1","z2","z")
      out$p.values <- c(p1,p2,p.pos)
      names(out$p.values) <- c("p1","p2","p")
      }
    out$proportion <- c(px.hat, py.hat, p.hat)
    if (!is.null(by)) {
      names(out$proportion) <- c(paste0("prop(",trimws(x.outcome.name),"=1|",trimws(x.name),")",collapse=""), paste0("prop(",trimws(y.outcome.name),"=1|",trimws(y.name),")",collapse=""), paste0("prop(",trimws(x.outcome.name),"=1)", collapse=""))
      }
     else {
      names(out$proportion) <- c(paste0("prop(",trimws(x.name),"=1)",collapse=""), paste0("prop(",trimws(y.name),"=1)",collapse=""), paste0("prop(",trimws(x.name),"=1 + ",trimws(y.name),"=1)", collapse=""))
      }
    out$sample_size <- c(nx,ny,nx+ny)
    names(out$sample_size) <- c("nx","ny","nx+ny")
    if (eqv.type=="delta") {
      if (upper==abs(eqv.level)) {
        out$threshold <- lower
        names(out$threshold) <- "\U394"
        }
       else {
         out$threshold <- c(upper, lower)
         names(out$threshold) <- c("\U0394u", "\U0394l")
        }
      }
     else {
      if (upper==abs(eqv.level)) {
        out$threshold <- lower
        names(out$threshold) <- "\U3B5"
        }
       else {
         out$threshold <- c(upper, lower)
         names(out$threshold) <- c("\U03B5u", "\U03B5l")
        }
      }
    if(relevance) {
      out$conclusion <- rel.conclusion
      names(out$conclusion) <- "relevance conclusion"
      }
    }

  invisible(out)
  }
