% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/all_paths_fun.R
\name{all_paths_fun}
\alias{all_paths_fun}
\title{Enumerate entry-to-sink call paths and compute risk metrics at the node and path level}
\usage{
all_paths_fun(
  graph,
  alpha = 0.6,
  beta = 0.3,
  gamma = 0.1,
  risk_form = c("additive", "power_mean"),
  p = 1,
  eps = 1e-12,
  complexity_col = "cyclo",
  weight_tol = 1e-08
)
}
\arguments{
\item{graph}{A directed \code{tidygraph::tbl_graph}. Graph nodes must have a \code{name}
attribute (i.e., \code{igraph::V(as.igraph(graph))$name}) and a numeric node
attribute specified by \code{complexity_col}.}

\item{alpha, beta, gamma}{Numeric non-negative weights for the risk score,
constrained such that \code{alpha + beta + gamma == 1} (within \code{weight_tol}).}

\item{risk_form}{Character. Form used to combine normalized node metrics into a node risk score.
One of \code{"additive"} or \code{"power_mean"}. Default \code{"additive"}.}

\item{p}{Numeric scalar. Power parameter used when \code{risk_form = "power_mean"}.
Must be finite and lie in the interval \eqn{[-1, 2]}. Default \code{1}.}

\item{eps}{Numeric. Small positive constant \eqn{\epsilon} used for numerical stability in the
\eqn{p \to 0} (geometric mean) case. Default \code{1e-12}.}

\item{complexity_col}{Character scalar. Name of the node attribute containing
cyclomatic complexity. Default \code{"cyclo"}.}

\item{weight_tol}{Numeric tolerance for enforcing the weight-sum constraint.
Default \code{1e-8}.}
}
\value{
A named list with two tibbles:
\describe{
\item{nodes}{Node-level metrics with columns \code{name}, \code{cyclomatic_complexity},
\code{indeg} (in-degree), \code{outdeg} (out-degree), \code{btw} (betweenness), \code{risk_score}.}
\item{paths}{Path-level metrics with columns \code{path_id}, \code{path_nodes},
\code{path_str}, \code{hops}, \code{path_risk_score}, \code{path_cc}, \code{gini_node_risk},
\code{risk_slope}, \code{risk_mean}, \code{risk_sum}}
}
}
\description{
Given a directed call graph (\code{tidygraph::tbl_graph}) with a node attribute for
cyclomatic complexity, this function:
\itemize{
\item computes node-level metrics (in-degree, out-degree, betweenness),
\item calculates a node risk score as a weighted combination of rescaled metrics,
\item enumerates all simple paths from entry nodes (in-degree = 0) to sink
nodes (out-degree = 0),
\item computes path-level summaries and a path-level risk score.
\item calculates a gini index and the slope of risk at the path-level.
}
}
\details{
The normalized node metrics are computed using \code{scales::rescale()} and denoted
by a tilde \eqn{\tilde{\cdot}}.

If \code{risk_form = "additive"}, the risk score for node \eqn{v_i} is computed as

\deqn{r_{(v_i)} = \alpha\,\tilde{C}_{(v_i)} + \beta\, \tilde{d}_{(v_i)}^{\mathrm{in}} + \gamma\,\tilde{b}_{(v_i)}\,,}

where the weights \eqn{\alpha}, \eqn{\beta} and \eqn{\gamma} reflect the relative
importance of complexity, coupling and network position, with the constraint
\eqn{\alpha + \beta + \gamma = 1}.

If \code{risk_form = "power_mean"}, the risk score for node \eqn{v_i} is computed as
the weighted power mean of normalized metrics:

\deqn{r_{(v_i)} =
\left(\alpha\,\tilde{C}_{(v_i)}^{p} + \beta\,\tilde{d}_{(v_i)}^{\mathrm{in}\,p} + \gamma\,\tilde{b}_{(v_i)}^{p}\right)^{1/p}\,,}

where \eqn{p} is the power-mean parameter. In the limit \eqn{p \to 0}, this reduces
to a weighted geometric mean, implemented with a small constant \eqn{\epsilon} to
ensure numerical stability:

\deqn{r_{(v_i)} =
\exp\left(\alpha\log(\max(\tilde{C}_{(v_i)}, \epsilon)) +
\beta\log(\max(\tilde{d}_{(v_i)}^{\mathrm{in}}, \epsilon)) +
\gamma\log(\max(\tilde{b}_{(v_i)}, \epsilon))\right)\,.}

The path-level risk score is calculated as

\deqn{P_k = 1 - \prod_{i=1}^{n_k} (1 - r_{k(v_i)})\,,}

where \eqn{r_{k(v_i)}} is the risk of the \eqn{i}-th function in path \eqn{k} and
\eqn{n_k} is the number of functions in that path. The equation behaves like a
saturating OR-operator: \eqn{P_k} is at least as large as the maximum individual
function risk and monotonically increases as more functions on the path become risky,
approaching 1 when several functions have high risk scores.

The Gini index of path \eqn{k} is computed as

\deqn{G_k = \frac{\sum_i \sum_j |r_{k(v_i)} - r_{k(v_j)}|}{2 n_k^2 \bar{r}_k}\,,}

where \eqn{\bar{r}_k} is the mean node risk in path \eqn{k}.

Finally, the trend of risk is defined by the slope of the regression

\deqn{r_{k(v_i)} = \theta_{0k} + \theta_{1k}\, i + \epsilon_i \,,}

where \eqn{r_{k(v_i)}} is the risk score of the function at position \eqn{i} along
path \eqn{k} (ordered from upstream to downstream execution) and \eqn{\epsilon_i} is a residual term.

The returned \code{paths} tibble includes \code{path_cc}, a list-column giving the numeric
vector of cyclomatic complexity values in the same order as \code{path_nodes} /
\code{path_str}.
}
\examples{
# synthetic_graph is a tidygraph::tbl_graph with node attribute "cyclo"
data(synthetic_graph)

# additive risk (default)
out1 <- all_paths_fun(
  graph = synthetic_graph,
  alpha = 0.6, beta = 0.3, gamma = 0.1,
  risk_form = "additive",
  complexity_col = "cyclo"
)

# power-mean risk (p = 0 ~ weighted geometric mean)
out2 <- all_paths_fun(
  graph = synthetic_graph,
  alpha = 0.6, beta = 0.3, gamma = 0.1,
  risk_form = "power_mean",
  p = 0,
  eps = 1e-12,
  complexity_col = "cyclo"
)

out1$nodes
out1$paths


}
